/**
 * @file      any_buffer.h
 * @brief     Template of any memory array
 * @date      2011/03/11
 *
 * @copyright
 * Copyright 2010-2011 Japan Aerospace Exploration Agency
 *
 */

#ifndef _ANY_BUFFER_H
#define _ANY_BUFFER_H

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* **********************************************************************
* Macro for declare struct
* @param T  item name.
* @param U  item's data type.
********************************************************************** */
#define ANY_BUFFER_DEFINE_STRUCT(T, U)                            \
  typedef struct _ ## T {                                         \
    size_t capacity;                                              \
    size_t size;                                                  \
    U* ptr;                                                       \
  } T;

/* **********************************************************************
* Macro for declare function
* @param T  item name.
* @param U  item's data type.
********************************************************************** */
#define ANY_BUFFER_DECLARE_FUNCTION(T, U)                         \
  typedef void (*T ## _each_callback)(U);                         \
  typedef void (*T ## _each_with_data_callback)(U,void *);        \
  typedef void (*T ## _each_with_index_callback)(U,int,void *);   \
  T* T ## _create(size_t size);                                   \
  T* T ## _create_with_data(const U *data, size_t size);          \
  void T ## _free(T *self);                                       \
  void* T ## _ptr(T *self);                                       \
  size_t T ## _size(T *self);                                     \
  U T ## _at(T *self, size_t index);                              \
  int T ## _set(T *self, const U *ptr, size_t size);              \
  int T ## _append(T *self, const U *ptr, size_t size);           \
  int T ## _truncate(T *self, size_t size);                       \
  int T ## _shrink(T *self);                                      \
  int T ## _reserve(T *self, size_t size);                        \
  void T ## _each(T *self, T ## _each_callback func);             \
  void T ## _each_with_data(T *self,                              \
                            T ## _each_with_data_callback func,   \
                            void* data);                          \
  void T ## _each_with_index(T *self,                             \
                             T ## _each_with_index_callback func, \
                             void* data);

/* **********************************************************************
* Macro for define function
* @param T  item name.
* @param U  item's data type.
********************************************************************** */
#define ANY_BUFFER_DEFINE_FUNCTION(T, U)                          \
  ANY_BUFFER_F_CREATE(T, U)                                       \
  ANY_BUFFER_F_CREATE_WITH_DATA(T, U)                             \
  ANY_BUFFER_F_FREE(T, U)                                         \
  ANY_BUFFER_F_PTR(T, U)                                          \
  ANY_BUFFER_F_SIZE(T, U)                                         \
  ANY_BUFFER_F_SET(T, U)                                          \
  ANY_BUFFER_F_APPEND(T, U)                                       \
  ANY_BUFFER_F_TRUNCATE(T, U)                                     \
  ANY_BUFFER_F_SHRINK(T, U)                                       \
  ANY_BUFFER_F_RESERVE(T, U)                                      \
  ANY_BUFFER_F_EACH(T, U)                                         \
  ANY_BUFFER_F_EACH_WITH_DATA(T, U)                               \
  ANY_BUFFER_F_AT(T, U)

/* **********************************************************************
* Functions
********************************************************************** */

#define ANY_BUFFER_F_CREATE(T, U)                                 \
  T *T ## _create(size_t size)                                    \
  {                                                               \
    T* self = malloc(sizeof(T));                                  \
    if (!self) {                                                  \
      return NULL;                                                \
    }                                                             \
    memset(self, 0, sizeof(T));                                   \
    if (!T ## _reserve(self, size)) {                             \
      return NULL;                                                \
    }                                                             \
    return self;                                                  \
  }

#define ANY_BUFFER_F_CREATE_WITH_DATA(T, U)                       \
  T *T ## _create_with_data(const U *data, size_t size)           \
  {                                                               \
    T* self = T ## _create(size*sizeof(U)*1.3);                   \
    if (!self) {                                                  \
      return NULL;                                                \
    }                                                             \
    if (!T ## _set(self, data, size)) {                           \
      T ## _free(self);                                           \
      return NULL;                                                \
    }                                                             \
    return self;                                                  \
  }


#define ANY_BUFFER_F_FREE(T, U)                                   \
  void T ## _free(T *self)                                        \
  {                                                               \
    if (self->ptr) {                                              \
      free(self->ptr);                                            \
      memset(self, 0, sizeof(T));                                 \
      free(self);                                                 \
    }                                                             \
  }

#define ANY_BUFFER_F_PTR(T, U)                                    \
  void* T ## _ptr(T *self)                                        \
  {                                                               \
    return self->ptr;                                             \
  }

#define ANY_BUFFER_F_SIZE(T, U)                                   \
  size_t T ## _size(T *self)                                      \
  {                                                               \
    return self->size;                                            \
  }

#define ANY_BUFFER_F_AT(T, U)                                     \
  U T ## _at(T *self, size_t index)                               \
  {                                                               \
    U* ptr = self->ptr + index;                                   \
    return *ptr;                                                  \
  }

#define ANY_BUFFER_F_SET(T, U)                                    \
  int T ## _set(T *self, const U *ptr, size_t size)               \
  {                                                               \
    if (!T ## _reserve(self, size * 1.3)) {                       \
      return 0;                                                   \
    }                                                             \
    memcpy(self->ptr, ptr, size * sizeof(U));                     \
    self->size = size;                                            \
    return 1;                                                     \
  }

#define ANY_BUFFER_F_APPEND(T, U)                                 \
  int T ## _append(T *self, const U *ptr, size_t size)            \
  {                                                               \
    if (!T ## _reserve(self, (self->size + size) * 1.3)) {        \
      return 0;                                                   \
    }                                                             \
    memcpy(self->ptr + self->size, ptr, size * sizeof(U));        \
    self->size += size;                                           \
    return 1;                                                     \
  }

#define ANY_BUFFER_F_TRUNCATE(T, U)                               \
  int T ## _truncate(T *self, size_t size)                        \
  {                                                               \
    if (0 < size && self->size > size) {                          \
      self->size = size;                                          \
    }                                                             \
    return 1;                                                     \
  }

#define ANY_BUFFER_F_SHRINK(T, U)                                 \
  int T ## _shrink(T *self)                                       \
  {                                                               \
    U* ptr;                                                       \
    if (0 < self->capacity &&                                     \
        0 < self->size &&                                         \
        self->size * 1.3 < self->capacity) {                      \
      ptr = malloc(self->size * sizeof(U) * 1.3);                 \
      if (!ptr) {                                                 \
        return 0;                                                 \
      }                                                           \
      memcpy(ptr, self->ptr, self->size * sizeof(U));             \
      free(self->ptr);                                            \
      self->ptr = ptr;                                            \
      self->capacity = self->size * 1.3;                          \
    }                                                             \
    return 1;                                                     \
  }

#define ANY_BUFFER_F_RESERVE(T, U)                                \
  int T ## _reserve(T *self, size_t size)                         \
  {                                                               \
    void* ptr;                                                    \
    if (!self->ptr) {                                             \
      self->ptr = malloc(size * sizeof(U));                       \
      if (!self->ptr) {                                           \
        return 0;                                                 \
      }                                                           \
    }                                                             \
    else if (self->capacity < size) {                             \
      ptr = realloc(self->ptr, size * sizeof(U));                 \
      if (!ptr) {                                                 \
        return 0;                                                 \
      }                                                           \
      self->capacity = size;                                      \
      self->ptr = ptr;                                            \
    }                                                             \
    return 1;                                                     \
  }

#define ANY_BUFFER_F_EACH(T, U)                                   \
  void T ## _each(T *self, T ## _each_callback func)              \
  {                                                               \
    size_t i = 0;                                                 \
    for (; i < self->size; i++) {                                 \
      func(T ## _at(self, i));                                    \
    }                                                             \
  }

#define ANY_BUFFER_F_EACH_WITH_DATA(T, U)                         \
  void T ## _each_with_data(T *self,                              \
                            T ## _each_with_data_callback func,   \
                            void* data)                           \
  {                                                               \
    size_t i = 0;                                                 \
    for (; i < self->size; i++) {                                 \
      func(T ## _at(self, i), data);                              \
    }                                                             \
  }

#define ANY_BUFFER_F_EACH_WITH_INDEX(T, U)                        \
  void T ## _each_with_index(T *self,                             \
                             T ## _each_with_index_callback func, \
                             void* data)                          \
  {                                                               \
    size_t i = 0;                                                 \
    for (; i < self->size; i++) {                                 \
      func(T ## _at(self, ai), i, data);                          \
    }                                                             \
  }

#endif /* _ANY_BUFFER_H */
