/**
 * @file      canvas.c
 * @brief     Provide 2-D memory canvas
 *
 * @date      2010/02/23
 *
 * @copyright
 * Copyright 2010 AstroArts Inc.
 *
 * @par       History:
 * - 2010/02/03 First release of AstroArts. Inc.
 * - 2012/08/09 Y. Yamamoto
 *   -# Remove meaningless codes
 *   -# Remove accessor to canvas pointer to keep consistency
 *   -# Optimize source code
 *   -# Add validate-routine
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <math.h>
#include "canvas.h"

#define CVMAX(a,b)        (((a)<(b)) ? (b) : (a))
#define CVMIN(a,b)        (((a)<(b)) ? (a) : (b))
#define CVBETWEEN(x,a,b)  CVMIN(CVMAX((x),(a)),(b))
#define CV8TO16(x)        ((uint16_t)(x)|((uint16_t)(x)<<8))
#define CV16TO8(x)        ((x)>>8)
#define CV8ITOF(x)        ((float)(x)/(float)0xff)
#define CV8FTOI(x)        ((unsigned char)(floorf(CVBETWEEN((x),0.0f,1.0f)*(float)0xff)))
#define CV16ITOF(x)       ((float)(x)/(float)0xffff)
#define CV16FTOI(x)       ((uint16_t)(floorf(CVBETWEEN((x),0.0f,1.0f)*(float)0xffff)))

/**
 * @brief Check x,y position in the canvas
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position
 * @param[in]   y       y position
 * @return      cv_check_area() returns 1 if specified x,y position are valid.
 *              Otherwise, 0 is returned.
 */
static int cv_check_position(const canvas* self, int x, int y)
{
  if( !(x >= 0 && x < self->width) ) {
    return 0;
  }

  if( !(y >= 0 && y < self->height) ) {
    return 0;
  }

  return 1;
}

/**
 * @brief Check sub-area in the canvas
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position
 * @param[in]   y       y position
 * @param[in]   w       width of sub-area
 * @param[in]   h       height of sub-area
 * @return      cv_check_subarea() returns 1 if specified sub-area is valid.
 *              Otherwise, 0 is returned.
 */
static int cv_check_subarea(const canvas* self, int x, int y, int w, int h)
{
  if( !cv_check_position(self, x, y) ) {
    return 0;
  }

  if( !(w > 0 && w <= self->width) ) {
    return 0;
  }

  if( !(h > 0 && h <= self->height) ) {
    return 0;
  }

  if ( !((x+w) < self->width) ) {
    return 0;
  }

  if ( !((y+h) < self->height) ) {
    return 0;
  }

  return 1;
}

canvas* cv_create(int width, int height, int depth)
{
  canvas* cv = cv_create_null(width, height, depth);
  if (!cv) {
    return NULL;
  }
  cv->buffer = calloc(cv->buffer_size, 1);
  if (!cv->buffer) {
    free(cv);
    return NULL;
  }
  return cv;
}

canvas* cv_create_null(int width, int height, int depth)
{
  int bits[CVDepthMAX] = {
    -1,
    8,                      // CVDepthGray8
    16,                     // CVDepthGA8
    16,                     // CVDepthAG8
    16,                     // CVDepthGray16
    32,                     // CVDepthGA16
    32,                     // CVDepthAG16
    24,                     // CVDepthRGB8
    32,                     // CVDepthARGB8
    32,                     // CVDepthRGBA8
    48,                     // CVDepthRGB16
    64,                     // CVDepthARGB16
    64                      // CVDepthRGBA16
  };

  canvas* cv = calloc(sizeof(canvas),1);
  if (!cv) {
    return NULL;
  }

  cv->width = width;
  cv->height = height;
  cv->depth = depth;

  assert(depth >= 1 && depth < CVDepthMAX);
  cv->bits_per_pixel = bits[depth];
  cv->bpp = cv->bits_per_pixel / 8;
  cv->buffer_size = cv->width * cv->height * cv->bpp;
  return cv;
}

canvas* cv_duplicate(const canvas* self)
{
  canvas* cv = cv_create_null(self->width, self->height, self->depth);
  if (!cv) {
    return NULL;
  }

  cv->buffer = malloc(cv->buffer_size);
  if (!cv->buffer) {
    free(cv);
    return NULL;
  }

  memcpy(cv->buffer, self->buffer, self->buffer_size);
  return cv;
}

canvas* cv_convert_depth(const canvas* self, int depth)
{
  canvas* cv = NULL;
  char* color[8];
  int x, y;

  if (depth == self->depth) {
    cv = cv_duplicate(self);
    return cv;
  }

  cv = cv_create(self->width, self->height, depth);
  if (!cv) {
    return NULL;
  }
  for (y = 0; y < self->height; y++) {
    for (x = 0; x < self->width; x++) {
      cv_get_pixel(self, x, y, color);
      cv_set_pixel_ex(cv, x, y, self->depth, color);
    }
  }
  return cv;
}

canvas* cv_copy(const canvas* self, int x, int y, int w, int h)
{
  canvas* cv = NULL;
  int dy;
  const unsigned char* src = NULL;
  unsigned char* dst = NULL;

  if (!cv_check_subarea(self,x,y,w,h)) {
    return NULL;
  }

  cv = cv_create(w, h, self->depth);
  if (!cv) {
    return NULL;
  }
  for (dy = 0; dy < h; dy++) {
    src = cv_get_row_ptr(self, y + dy) + self->bpp * x;
    dst = cv_get_row_ptr(cv, dy);
    memcpy(dst, src, self->bpp * w);
  }
  return cv;
}

canvas* cv_flip(canvas* self)
{
  int row;
  int size = (self->bpp) * self->width;
  canvas* cv = cv_create(self->width, self->height, self->depth);
  if (!cv) {
    return NULL;
  }
  for (row = 0; row < self->height; row++) {
    memcpy(cv_get_row_ptr(cv, cv->height - row - 1), cv_get_row_ptr(self, row), size);
  }
  return cv;
}

canvas* cv_extract_plane(const canvas* self, canvasColorPlane plane)
{
  static const int iplane_table[CVDepthMAX] = {
    -1,
    CVColorPlaneGray,                                                                                // CVDepthGray8
    CVColorPlaneGray | CVColorPlaneAlpha,                                                            // CVDepthGA8
    CVColorPlaneGray | CVColorPlaneAlpha,                                                            // CVDepthAG8
    CVColorPlaneGray,                                                                                // CVDepthGray16
    CVColorPlaneGray | CVColorPlaneAlpha,                                                            // CVDepthGA16
    CVColorPlaneGray | CVColorPlaneAlpha,                                                            // CVDepthAG16
    CVColorPlaneRed | CVColorPlaneGreen | CVColorPlaneBlue,                                          // CVDepthRGB8
    CVColorPlaneRed | CVColorPlaneGreen | CVColorPlaneBlue | CVColorPlaneAlpha,                      // CVDepthARGB8
    CVColorPlaneRed | CVColorPlaneGreen | CVColorPlaneBlue | CVColorPlaneAlpha,                      // CVDepthRGBA8
    CVColorPlaneRed | CVColorPlaneGreen | CVColorPlaneBlue,                                          // CVDepthRGB16
    CVColorPlaneRed | CVColorPlaneGreen | CVColorPlaneBlue | CVColorPlaneAlpha,                      // CVDepthARGB16
    CVColorPlaneRed | CVColorPlaneGreen | CVColorPlaneBlue | CVColorPlaneAlpha                       // CVDepthRGBA16
  };

  static const int output_depth[CVDepthMAX] = {
    -1,
    CVDepthGray8,                      // CVDepthGray8
    CVDepthGray8,                      // CVDepthGA8
    CVDepthGray8,                      // CVDepthAG8
    CVDepthGray16,                     // CVDepthGray16
    CVDepthGray16,                     // CVDepthGA16
    CVDepthGray16,                     // CVDepthAG16
    CVDepthGray8,                      // CVDepthRGB8
    CVDepthGray8,                      // CVDepthARGB8
    CVDepthGray8,                      // CVDepthRGBA8
    CVDepthGray16,                     // CVDepthRGB16
    CVDepthGray16,                     // CVDepthARGB16
    CVDepthGray16                      // CVDepthRGBA16
  };

  canvas* cv = NULL;

  int depth;
  int iplane;
  int x;
  int y;
  int cidx = -1;
  const uint8_t* c8 = NULL;
  const uint16_t* c16 = NULL;
  const void* ptr = cv_get_row_ptr(self, 0);

  iplane = iplane_table[self->depth];

  if (0 == (plane & iplane)) {
    fprintf(stderr, "WARN: original canvas does not have specified color plane. %s:%d\n", __FILE__, __LINE__);
    return NULL;
  }

  /* same as copy */
  switch (self->depth) {
  case CVDepthGray8:
  case CVDepthGray16:
    return cv_duplicate(self);
  }

  /* check output depth */
  depth = output_depth[self->depth];

  /* check extract plane */
  if (plane == CVColorPlaneGray) {
    switch (self->depth) {
    case CVDepthGA8:  cidx = 0; break;
    case CVDepthAG8:  cidx = 1; break;
    case CVDepthGA16: cidx = 0; break;
    case CVDepthAG16: cidx = 1; break;
    }
  }
  else if (plane == CVColorPlaneRed) {
    switch (self->depth) {
    case CVDepthRGB8:   cidx = 0; break;
    case CVDepthRGBA8:  cidx = 0; break;
    case CVDepthARGB8:  cidx = 1; break;
    case CVDepthRGB16:  cidx = 0; break;
    case CVDepthRGBA16: cidx = 0; break;
    case CVDepthARGB16: cidx = 1; break;
    }
  }
  else if (plane == CVColorPlaneGreen) {
    switch (self->depth) {
    case CVDepthRGB8:   cidx = 1; break;
    case CVDepthRGBA8:  cidx = 1; break;
    case CVDepthARGB8:  cidx = 2; break;
    case CVDepthRGB16:  cidx = 1; break;
    case CVDepthRGBA16: cidx = 1; break;
    case CVDepthARGB16: cidx = 2; break;
    }
  }
  else if (plane == CVColorPlaneBlue) {
    switch (self->depth) {
    case CVDepthRGB8:   cidx = 2; break;
    case CVDepthRGBA8:  cidx = 2; break;
    case CVDepthARGB8:  cidx = 3; break;
    case CVDepthRGB16:  cidx = 2; break;
    case CVDepthRGBA16: cidx = 2; break;
    case CVDepthARGB16: cidx = 3; break;
    }
  }
  else if (plane == CVColorPlaneAlpha) {
    switch (self->depth) {
    case CVDepthGA8:    cidx = 1; break;
    case CVDepthAG8:    cidx = 0; break;
    case CVDepthGA16:   cidx = 1; break;
    case CVDepthAG16:   cidx = 0; break;
    case CVDepthRGBA8:  cidx = 3; break;
    case CVDepthARGB8:  cidx = 0; break;
    case CVDepthRGBA16: cidx = 3; break;
    case CVDepthARGB16: cidx = 0; break;
    }
  }

  if (cidx == -1) {
    fprintf(stderr, "WARN: original canvas does not have specified color plane. %s:%d\n", __FILE__, __LINE__);
    return NULL;
  }

  cv = cv_create(self->width, self->height, depth);
  if (!cv) {
    return NULL;
  }

  if (depth == CVDepthGray16) {
    for (y = 0; y < self->height; y++) {
      for (x = 0; x < self->width; x++) {
        c16 = ptr + self->bpp * (self->width * y + x);
        cv_set_pixel(cv, x, y, &c16[cidx]);
      }
    }
  }
  else {
    for (y = 0; y < self->height; y++) {
      for (x = 0; x < self->width; x++) {
        c8 = ptr + self->bpp * (self->width * y + x);
        cv_set_pixel(cv, x, y, &c8[cidx]);
      }
    }
  }
  return cv;
}

void cv_free(canvas* self)
{
  if (self) {
    if (self->buffer) {
      free(self->buffer);
    }
    free(self);
  }
}

void* cv_get_row_ptr(const canvas* self, int row)
{
  if (self->buffer == NULL) {
    return NULL;
  }

  if (row < 0 || self->height <= row) {
    return NULL;
  }
  return (char *)self->buffer + self->bpp * self->width * row;
}

int cv_isnull(const canvas* self)
{
  return self->buffer ? 0 : 1;
}

void cv_clear(canvas* self)
{
  if (self->buffer) {
    memset(self->buffer, 0, self->buffer_size);
  }
}

int cv_paste(canvas* self, canvas* src, int x, int y, canvasBlendFunction blend)
{
  return cv_paste_ex(self, src, x, y, src->width, src->height, blend);
}

int cv_paste_ex(canvas* self, canvas* src, int x, int y, int w, int h, canvasBlendFunction blend)
{
  canvas* cv = src;
  canvas* cv_pool1 = NULL;
  canvas* cv_pool2 = NULL;
  int xx, yy;
  const void* sptr = NULL;
  void* dptr = NULL;

  // check paste size
  if (src->width < w || src->height < h) {
    return 0;
  }

  if (self->buffer == NULL) {
    return 1;
  }

  w = (x + w < self->width)  ? w : self->width - x;
  h = (y + h < self->height) ? h : self->height - y;

  if (CVBlendNone == blend) {
    /* change size */
    if (src->width != w || src->height != h) {
      cv_pool1 = cv_copy(src, 0, 0, w, h);
      assert(cv_pool1 != NULL);
      cv = cv_pool1;
    }
    /* change color depth */
    if (cv->depth != self->depth) {
      cv_pool2 = cv_convert_depth(self, self->depth);
      assert(cv_pool2 != NULL);
      cv = cv_pool2;
    }
    for (yy = 0; yy < h; yy++) {
      sptr = cv_get_row_ptr(cv, yy);
      dptr = cv_get_row_ptr(self, y + yy) + self->bpp * x;
      memcpy(dptr, sptr, self->bpp * w);
    }
  }
  else {
    uint16_t c[4];
    for (yy = 0; yy < h; yy++) {
      for (xx = 0; xx < w; xx++) {
        cv_get_pixel_ex(src, xx, yy, CVDepthARGB16, c);
        cv_blend_pixel_ex(self, x+xx, y+yy, CVDepthARGB16, c, blend);
      }
    }
  }

  if (cv_pool1) cv_free(cv_pool1);
  if (cv_pool2) cv_free(cv_pool2);

  return 1;
}

void cv_set_pixel(canvas* self, int x, int y, const void* color)
{
  cv_set_pixel_ex(self, x, y, self->depth, color);
}

void cv_set_pixel_ex(canvas* self, int x, int y, int depth, const void* color)
{
  unsigned char* pixel = NULL;
  uint16_t argb16[] = {0,0,0,0};
  if (self->buffer == NULL || !cv_check_position(self, x, y)) {
    return;
  }
  pixel = self->buffer + self->bpp * (self->width * y + x);
  if (depth == self->depth) {
    memcpy(pixel, color, self->bpp);
  }
  else {
    _cv_convert_to_argb16(color, depth, argb16);
    _cv_convert_from_argb16(argb16, self->depth, pixel);
  }
}

void cv_blend_pixel(canvas* self, int x, int y, const void* color, canvasBlendFunction blend)
{
  cv_blend_pixel_ex(self, x, y, self->depth, color, blend);
}

void cv_blend_pixel_ex(canvas* self, int x, int y, int depth, const void* color, canvasBlendFunction blend)
{
  unsigned char* pixel = NULL;
  uint16_t c[] = {0,0,0,0};
  uint16_t cc[] = {0,0,0,0};
  int i;
  float af, f;
  float caf, cf;

  // check
  if (self->buffer == NULL || !cv_check_position(self, x, y)) {
    return;
  }

  if (blend == CVBlendNone) {
    cv_set_pixel_ex(self, x, y, depth, color);
    return;
  }

  pixel = self->buffer + self->bpp * (self->width * y + x);
  cv_get_pixel_ex(self, x, y, CVDepthARGB16, cc);
  _cv_convert_to_argb16(color, depth, c);
  switch (blend) {
  case CVBlendAlpha:
    if (0xFFFF == c[0] || 0 == cc[0]) {
      for (i = 0; i < 4; i++) {
        cc[i] = c[i];
      }
    }
    else if (0 == c[0]) {
      return;
    }
    else {
      float df, r;
      af = CV16ITOF(c[0]);
      caf = CV16ITOF(cc[0]);
      r = af + caf;
      for (i = 1; i < 4; i++) {
        f = CV16ITOF(c[i]);
        cf = CV16ITOF(cc[i]);
        df = f - cf;
        cf = cf + df * af;
        f = f - df * caf;
        cf = cf * caf / r + f * af / r;
        cc[i] = CV16FTOI(cf);
      }
      cf = caf + (1.0 - caf) * af;
      cc[0] = CV16FTOI(cf);
    }
    break;
  case CVBlendAdd:
    for (i = 0; i < 4; i++) {
      cc[i] = (0xFFFF < (int)cc[i]+(int)c[i]) ? 0xFFFF : cc[i]+c[i];
    }
    break;
  case CVBlendSub:
    for (i = 0; i < 4; i++) {
      cc[i] = ((int)cc[i]-(int)c[i] < 0) ? 0 : cc[i]-c[i];
    }
    break;
  case CVBlendAnd:
    for (i = 0; i < 4; i++) {
      cc[i] &= c[i];
    }
    break;
  case CVBlendOr:
    for (i = 0; i < 4; i++) {
      cc[i] |= c[i];
    }
    break;
  case CVBlendXor:
    for (i = 0; i < 4; i++) {
      cc[i] ^= c[i];
    }
    break;
  case CVBlendMul:
    for (i = 0; i < 4; i++) {
      f = (float)c[i] / (float)0xffff;
      cf = (float)cc[i] / (float)0xffff;
      cc[i] = (uint16_t)floorf(f*cf*(float)0xffff);
    }
    break;
  case CVBlendNone:
    /* never through */
    break;
  }

  _cv_convert_from_argb16(cc, self->depth, pixel);
}

void cv_blend_pixel2(canvas* self, int x, int y, const void* color, int sfactor, int dfactor)
{
  cv_blend_pixel2_ex(self, x, y, self->depth, color, sfactor, dfactor);
}

void cv_blend_pixel2_ex(canvas* self, int x, int y, int depth, const void* color, int sfactor, int dfactor)
{
  unsigned char* pixel = NULL;
  uint16_t dc[] = {0,0,0,0};
  uint16_t sc[] = {0,0,0,0};
  int i;
  float osc[4];
  float odc[4];
  float csc[4];
  float cdc[4];

  if (self->buffer == NULL || !cv_check_position(self, x, y)) {
    return;
  }

  if (sfactor == CVBlendZero && dfactor == CVBlendOne) {
    cv_set_pixel_ex(self, x, y, depth, color);
    return;
  }
  pixel = self->buffer + self->bpp * (self->width * y + x);
  cv_get_pixel_ex(self, x, y, CVDepthARGB16, dc);
  _cv_convert_to_argb16(color, depth, sc);
  for (i = 0; i < 4; i++) {
    osc[i] = CV16ITOF(sc[i]);
    odc[i] = CV16ITOF(dc[i]);
  }
  switch (sfactor) {
  case CVBlendZero:
    for (i = 0; i < 4; i++) csc[i] = 0;
    break;
  case CVBlendOne:
    for (i = 0; i < 4; i++) csc[i] = osc[i];
    break;
  case CVBlendSrcColor:
    for (i = 0; i < 4; i++) csc[i] = osc[i] * osc[i];
    break;
  case CVBlendOneMinusSrcColor:
    for (i = 0; i < 4; i++) csc[i] = osc[i] * (1.0f - osc[i]);
    break;
  case CVBlendDstColor:
    for (i = 0; i < 4; i++) csc[i] = osc[i] * odc[i];
    break;
  case CVBlendOneMinusDstColor:
    for (i = 0; i < 4; i++) csc[i] = osc[i] * (1.0f - odc[i]);
    break;
  case CVBlendSrcAlpha:
    for (i = 0; i < 4; i++) csc[i] = osc[i] * osc[0];
    break;
  case CVBlendOneMinusSrcAlpha:
    for (i = 0; i < 4; i++) csc[i] = osc[i] * (1.0f - osc[0]);
    break;
  case CVBlendDstAlpha:
    for (i = 0; i < 4; i++) csc[i] = osc[i] * odc[0];
    break;
  case CVBlendOneMinusDstAlpha:
    for (i = 0; i < 4; i++) csc[i] = osc[i] * (1.0f - odc[0]);
    break;
  }
  switch (dfactor) {
  case CVBlendZero:
    for (i = 0; i < 4; i++) csc[i] = 0;
    break;
  case CVBlendOne:
    for (i = 0; i < 4; i++) cdc[i] = odc[i];
    break;
  case CVBlendSrcColor:
    for (i = 0; i < 4; i++) cdc[i] = odc[i] * osc[i];
    break;
  case CVBlendOneMinusSrcColor:
    for (i = 0; i < 4; i++) cdc[i] = odc[i] * (1.0f - osc[i]);
    break;
  case CVBlendDstColor:
    for (i = 0; i < 4; i++) cdc[i] = odc[i] * odc[i];
    break;
  case CVBlendOneMinusDstColor:
    for (i = 0; i < 4; i++) cdc[i] = odc[i] * (1.0f - odc[i]);
    break;
  case CVBlendSrcAlpha:
    for (i = 0; i < 4; i++) cdc[i] = odc[i] * osc[0];
    break;
  case CVBlendOneMinusSrcAlpha:
    for (i = 0; i < 4; i++) cdc[i] = odc[i] * (1.0f - osc[0]);
    break;
  case CVBlendDstAlpha:
    for (i = 0; i < 4; i++) cdc[i] = odc[i] * odc[0];
    break;
  case CVBlendOneMinusDstAlpha:
    for (i = 0; i < 4; i++) cdc[i] = odc[i] * (1.0f - odc[0]);
    break;
  }

  for (i = 0; i < 4; i++) {
    dc[i] = CV16FTOI(csc[i] + cdc[i]);
  }
  _cv_convert_from_argb16(dc, self->depth, pixel);
}

void cv_set_pixel_gray8(canvas* self, int x, int y, uint8_t color)
{
  cv_set_pixel_ex(self, x, y, CVDepthGray8, &color);
}

void cv_set_pixel_gray16(canvas* self, int x, int y, uint16_t color)
{
  cv_set_pixel_ex(self, x, y, CVDepthGray16, &color);
}

void cv_set_pixel_argb8(canvas* self, int x, int y, uint32_t color)
{
  uint8_t color_c[4];
  color_c[0] = (color >> 24) & 0xFF;
  color_c[1] = (color >> 16) & 0xFF;
  color_c[2] = (color >> 8) & 0xFF;
  color_c[3] = (color & 0xFF);
  cv_set_pixel_ex(self, x, y, CVDepthARGB8, color_c);
}

void cv_set_pixel_argb16(canvas* self, int x, int y, uint64_t color)
{
  int16_t color_s[4];
  color_s[0] = (color >> 48) & 0xFFFF;
  color_s[1] = (color >> 32) & 0xFFFF;
  color_s[2] = (color >> 16) & 0xFFFF;
  color_s[3] = (color & 0xFFFF);
  cv_set_pixel_ex(self, x, y, CVDepthARGB16, color_s);
}

void cv_get_pixel(const canvas* self, int x, int y, void* color)
{
  cv_get_pixel_ex(self, x, y, self->depth, color);
}

void cv_get_pixel_ex(const canvas* self, int x, int y, int depth, void* color)
{
  unsigned char* pixel = NULL;
  uint16_t argb16[] = {0,0,0,0};
  if (self->buffer == NULL || !cv_check_position(self, x, y)) {
    _cv_convert_from_argb16(argb16, depth, color);
    return;
  }
  pixel = (self->buffer + self->bpp * (self->width * y + x));
  if (depth == self->depth) {
    memcpy(color, pixel, self->bpp);
  }
  else {
    _cv_convert_to_argb16(pixel, self->depth, argb16);
    _cv_convert_from_argb16(argb16, depth, color);
  }
}

uint8_t cv_get_pixel_gray8(const canvas* self, int x, int y)
{
  uint8_t c;
  cv_get_pixel_ex(self, x, y, CVDepthGray8, &c);
  return c;
}

uint16_t cv_get_pixel_gray16(const canvas* self, int x, int y)
{
  uint16_t s;
  cv_get_pixel_ex(self, x, y, CVDepthGray16, &s);
  return s;
}

uint32_t cv_get_pixel_argb8(const canvas* self, int x, int y)
{
  uint8_t color_c[4];
  uint32_t color;
  cv_get_pixel_ex(self, x, y, CVDepthARGB8, color_c);
  color  = ((uint32_t)color_c[0] << 24);
  color |= ((uint32_t)color_c[1] << 16);
  color |= ((uint32_t)color_c[2] <<  8);
  color |= (uint32_t)color_c[3];
  return color;
}

uint64_t cv_get_pixel_argb16(const canvas* self, int x, int y)
{
  uint16_t color_s[4];
  uint64_t color;
  cv_get_pixel_ex(self, x, y, CVDepthARGB16, color_s);
  color  = ((uint64_t)color_s[0] << 48);
  color |= ((uint64_t)color_s[1] << 32);
  color |= ((uint64_t)color_s[2] << 16);
  color |= (uint64_t)color_s[3];
  return color;
}

void cv_draw_line(canvas* self, int sx, int sy, int ex, int ey, const void* color)
{
  cv_draw_line_ex(self, sx, sy, ex, ey, self->depth, color);
}

void cv_draw_line_ex(canvas* self, int sx, int sy, int ex, int ey, int depth, const void* color)
{
  uint16_t argb16[] = {0,0,0,0};
  uint8_t clr[sizeof(uint16_t)*4];
  int i, x, y;
  int dx, dy;
  int ddx, ddy;
  int e;

  if (self->buffer == NULL) {
    return;
  }

  // const void *color -> uint16_t argb16[4] -> uint8_t or uint16_t
  _cv_convert_to_argb16(color, self->depth, argb16);
  _cv_convert_from_argb16(argb16, depth, clr);

  dx = ex - sx;
  if (0 < dx) {
    ddx = 1;
  }
  else {
    dx = -dx;
    ddx = -1;
  }

  dy = ey - sy;
  if (0 < dy) {
    ddy = 1;
  }
  else {
    dy = -dy;
    ddy = -1;
  }

  x = sx;
  y = sy;
  if (dx > dy) {
    e = -dx;
    for (i = 0; i <= dx; i++) {
      cv_set_pixel_ex(self, x, y, self->depth, clr);
      x += ddx;
      e += 2 * dy;
      if (e >= 0) {
        y += ddy;
        e -= 2 * dx;
      }
    }
  }
  else {
    e = -dy;
    for (i = 0; i <= dy; i++) {
      cv_set_pixel_ex(self, x, y, self->depth, clr);
      y += ddy;
      e += 2 * dx;
      if (e >= 0) {
        x += ddx;
        e -= 2 * dy;
      }
    }
  }
}

void cv_draw_line_gray8(canvas* self, int sx, int sy, int ex, int ey, uint8_t color)
{
  cv_draw_line_ex(self, sx, sy, ex, ey, CVDepthGray8, &color);
}

void cv_draw_line_gray16(canvas* self, int sx, int sy, int ex, int ey, uint16_t color)
{
  cv_draw_line_ex(self, sx, sy, ex, ey, CVDepthGray16, &color);
}

void cv_draw_line_argb8(canvas* self, int sx, int sy, int ex, int ey, uint32_t color)
{
  cv_draw_line_ex(self, sx, sy, ex, ey, CVDepthARGB8, &color);
}

void cv_draw_line_argb16(canvas* self, int sx, int sy, int ex, int ey, uint64_t color)
{
  cv_draw_line_ex(self, sx, sy, ex, ey, CVDepthARGB16, &color);
}

void cv_draw_rect(canvas* self, int x, int y, int width, int height, const void* color, canvasBlendFunction blend)
{
  cv_draw_rect_ex(self, x, y, width, height, self->depth, color, blend);
}

void cv_draw_rect_ex(canvas* self, int x, int y, int width, int height, int depth, const void* color, canvasBlendFunction blend)
{
  int xx, yy;
  if (self->buffer == NULL || !cv_check_position(self, x, y)) {
    return;
  }

  if (self->width < x + width) {
    width = self->width - x;
  }
  if (self->height < y + height) {
    height = self->height - y;
  }
  if (blend == CVBlendNone) {
    for (yy = y; yy < y+height; yy++) {
      for (xx = x; xx < x+width; xx++) {
        cv_set_pixel_ex(self, xx, yy, depth, color);
      }
    }
  }
  else {
    for (yy = y; yy < y+height; yy++) {
      for (xx = x; xx < x+width; xx++) {
        cv_blend_pixel_ex(self, xx, yy, depth, color, blend);
      }
    }
  }
}

void _cv_convert_to_argb16(const void* src, int depth, uint16_t* dst)
{
  const unsigned char* color_c = src;
  const uint16_t* color_s = src;
  int i;
  switch (depth) {
  case CVDepthGray8:
    dst[0] = 0xFFFF;
    dst[1] = dst[2] = dst[3] = CV8TO16(color_c[0]);
    break;
  case CVDepthAG8:
    dst[0] = CV8TO16(color_c[0]);
    dst[1] = dst[2] = dst[3] = CV8TO16(color_c[1]);
    break;
  case CVDepthGA8:
    dst[0] = CV8TO16(color_c[1]);
    dst[1] = dst[2] = dst[3] = CV8TO16(color_c[0]);
    break;
  case CVDepthGray16:
    dst[0] = 0xFFFF;
    dst[1] = dst[2] = dst[3] = color_s[0];
    break;
  case CVDepthAG16:
    dst[0] = color_s[0];
    dst[1] = dst[2] = dst[3] = color_s[1];
    break;
  case CVDepthGA16:
    dst[0] = color_s[1];
    dst[1] = dst[2] = dst[3] = color_s[0];
    break;
  case CVDepthRGB8:
    dst[0] = 0xFFFF;
    dst[1] = CV8TO16(color_c[0]);
    dst[2] = CV8TO16(color_c[1]);
    dst[3] = CV8TO16(color_c[2]);
    break;
  case CVDepthRGB16:
    dst[0] = 0xFFFF;
    dst[1] = color_s[0];
    dst[2] = color_s[1];
    dst[3] = color_s[2];
    break;
  case CVDepthRGBA8:
    dst[0] = CV8TO16(color_c[3]);
    dst[1] = CV8TO16(color_c[0]);
    dst[2] = CV8TO16(color_c[1]);
    dst[3] = CV8TO16(color_c[2]);
    break;
  case CVDepthRGBA16:
    dst[0] = color_s[3];
    dst[1] = color_s[0];
    dst[2] = color_s[1];
    dst[3] = color_s[2];
    break;
  case CVDepthARGB8:
    for (i = 0; i < 4; i++) dst[i] = CV8TO16(color_c[i]);
    break;
  case CVDepthARGB16:
    for (i = 0; i < 4; i++) dst[i] = color_s[i];
    break;
  }
}

void _cv_convert_from_argb16(const uint16_t* src, int depth, void* dst)
{
  uint8_t* color_c = dst;
  uint16_t* color_s = dst;
  int i;
  switch (depth) {
  case CVDepthGray8:
    color_c[0] = CV16TO8((src[1] + src[2] + src[3]) / 3);
    break;
  case CVDepthAG8:
    color_c[0] = CV16TO8(src[0]);
    color_c[1] = CV16TO8((src[1] + src[2] + src[3]) / 3);
    break;
  case CVDepthGA8:
    color_c[1] = CV16TO8(src[0]);
    color_c[0] = CV16TO8((src[1] + src[2] + src[3]) / 3);
    break;
  case CVDepthGray16:
    color_s[0] = (src[1] + src[2] + src[3]) / 3;
    break;
  case CVDepthAG16:
    color_s[0] = src[0];
    color_s[1] = (src[1] + src[2] + src[3]) / 3;
    break;
  case CVDepthGA16:
    color_s[1] = src[0];
    color_s[0] = (src[1] + src[2] + src[3]) / 3;
    break;
  case CVDepthRGB8:
    color_c[0] = CV16TO8(src[1]);
    color_c[1] = CV16TO8(src[2]);
    color_c[2] = CV16TO8(src[3]);
    break;
  case CVDepthRGB16:
    color_s[0] = src[1];
    color_s[1] = src[2];
    color_s[2] = src[3];
    break;
  case CVDepthRGBA8:
    color_c[0] = CV16TO8(src[1]);
    color_c[1] = CV16TO8(src[2]);
    color_c[2] = CV16TO8(src[3]);
    color_c[3] = CV16TO8(src[0]);
    break;
  case CVDepthRGBA16:
    color_s[0] = src[1];
    color_s[1] = src[2];
    color_s[2] = src[3];
    color_s[3] = src[0];
    break;
  case CVDepthARGB8:
    for (i = 0; i < 4; i++) color_c[i] = CV16TO8(src[i]);
    break;
  case CVDepthARGB16:
    for (i = 0; i < 4; i++) color_s[i] = src[i];
    break;
  }
}
