/**
 * @file      canvas.h
 * @brief     Provide 2-D memory canvas
 *
 * @date      2010/02/23
 *
 * @copyright
 * Copyright 2010 AstroArts Inc.
 *
 * @par       History:
 * - 2010/02/23 First release of AstroArts. Inc.
 *   -# Add Comment in each function
 *
 */

#ifndef _CANVAS_H
#define _CANVAS_H

#include <stdint.h>

/**
 * Color bits definition used in canvas depth
 */
typedef enum _canvasColorDepth {
  CVDepthGray8 = 1,
  CVDepthGA8,
  CVDepthAG8,
  CVDepthGray16,
  CVDepthGA16,
  CVDepthAG16,
  CVDepthRGB8,
  CVDepthARGB8,
  CVDepthRGBA8,
  CVDepthRGB16,
  CVDepthARGB16,
  CVDepthRGBA16,
  CVDepthMAX
} canvasColorDepth;

/**
 * Color plane definitions
 */
typedef enum _canvasColorPlane {
  CVColorPlaneGray  = 0x01,
  CVColorPlaneRed   = 0x02,
  CVColorPlaneGreen = 0x04,
  CVColorPlaneBlue  = 0x08,
  CVColorPlaneAlpha = 0x10
} canvasColorPlane;

/**
 * Blend method definitions (1)
 */
typedef enum _canvasBlendFunction {
  CVBlendNone  = 0x00,
  CVBlendAdd,
  CVBlendSub,
  CVBlendAnd,
  CVBlendOr,
  CVBlendXor,
  CVBlendMul,
  CVBlendAlpha
} canvasBlendFunction;

/**
 * Blend method definitions (2)
 * used in cv_blend_pixel2*
 */
#define CVBlendZero                 1
#define CVBlendOne                  2
#define CVBlendSrcColor             3
#define CVBlendOneMinusSrcColor     4
#define CVBlendDstColor             5
#define CVBlendOneMinusDstColor     6
#define CVBlendSrcAlpha             7
#define CVBlendOneMinusSrcAlpha     8
#define CVBlendDstAlpha             9
#define CVBlendOneMinusDstAlpha    10

/** @brief read/write attributes */
#define CVFormatBottomUp    1

typedef struct _canvas {
  int32_t* index;
  unsigned char* buffer;
  size_t buffer_size;
  int depth;
  int bits_per_pixel;
  int bpp;       // bytes per pixel
  unsigned width;
  unsigned height;
} canvas;

__BEGIN_DECLS
/**
 * @brief Create a new canvas
 * @param[in]   width   canvas width
 * @param[in]   height  canvas height
 * @param[in]   depth   canvas depth
 * @return      Upon successful completion, cv_create() returns a canvas pointer.
 *              Otherwise, NULL is returned.
 */
canvas* cv_create(int width, int height, int depth);

/**
 * @brief Create a null canvas (null canvas means a buffer is not allocated)
 * @param[in]   width   canvas width
 * @param[in]   height  canvas height
 * @param[in]   depth   canvas depth
 * @return      Upon successful completion, cv_create_null() returns canvas pointer.
 *              Otherwise, NULL is returned.
 */
canvas* cv_create_null(int width, int height, int depth);

/**
 * @brief Duplicate a canvas
 * @param[in]   self    canvas pointer
 * @return      Upon successful completion, cv_duplicate() returns duplicated-pointer.
 *              Otherwise, NULL is returned.
 */
canvas* cv_duplicate(const canvas* self);

/**
 * @brief Duplicate a canvas with change color bits
 * @param[in]   self    canvas pointer
 * @param[in]   depth   new canvas depth
 * @return      Upon successful completion, cv_convert_depth() returns new canvas pointer
 *              with specified depth. Otherwise, NULL is returned.
 */
canvas* cv_convert_depth(const canvas* self, int depth);

/**
 * @brief Copy sub-area from source canvas
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position of start point
 * @param[in]   y       y position of start point
 * @param[in]   w       width of sub-area
 * @param[in]   h       height of sub-area
 * @return      Upon successful completion, cv_copy() returns new canvas pointer
 *              with specified depth. Otherwise, NULL is returned.
 */
canvas* cv_copy(const canvas* self, int x, int y, int w, int h);

/**
 * @brief Vertical flip
 * @param[in]   self    canvas pointer
 * @return      Upon successful completion, cv_flip() returns new canvas pointer
 *              with vertical flip. Otherwise, NULL is returned.
 */
canvas* cv_flip(canvas* self);

/**
 * @brief Extract one of color/alpha planes from source.
 * @param[in]   self    canvas pointer
 * @param[in]   plane
 *
 * @return      Upon successful completion, cv_extract_plane() returns a new canvas pointer
 *              which is extracted plane from source. Otherwise, NULL is returned.
 */
canvas* cv_extract_plane(const canvas* self, canvasColorPlane plane);

/**
 * @brief Free allocated canvas
 * @param[in]   self    canvas pointer
 */
void cv_free(canvas* self);

/**
 * @brief Return canvas's start address of specified row
 *
 * @param[in]   self    canvas pointer
 * @param[in]   row
 *
 * @return      Upon successful completion, cv_get_row_ptr() returns a start address.
 *              Otherwise, NULL is returned.
 */
void* cv_get_row_ptr(const canvas* self, int row);

/**
 * @brief Check whether canvas is null-canvas
 *
 * @param[in]   self    canvas pointer
 *
 * @return      cv_isnull() returns 1 if canvas is null-canvas.
 *              Otherwise, 0 is returned.
 */
int cv_isnull(const canvas* self);

/**
 * @brief Clear the canvas with zero
 *
 * @param[in]   self    canvas pointer
 */
void cv_clear(canvas* self);

/**
 * @brief Paste another canvas's content to specified coordinates
 *
 * @param[in]   self    canvas pointer
 * @param[in]   src     source canvas
 * @param[in]   x       x position
 * @param[in]   y       y position
 * @param[in]   blend   blend function
 *
 * @return      cv_paste() returns 1 if the canvas is null-canvas.
 *              Otherwise, 0 is returned.
 */
int cv_paste(canvas* self, canvas* src, int x, int y, canvasBlendFunction blend);

/**
 * @brief Paste another canvas's sub-content to specified coordinates
 *
 * @param[in]   self    canvas pointer
 * @param[in]   src     source canvas
 * @param[in]   x       x position in source canvas
 * @param[in]   y       y position in source canvas
 * @param[in]   w       width of paste window size
 * @param[in]   h       height of paste window size
 * @param[in]   blend   blend function
 *
 * @return      cv_paste_ex() returns 1 if the canvas is null-canvas.
 *              Otherwise, 0 is returned.
 */
int cv_paste_ex(canvas* self, canvas* src, int x, int y, int w, int h, canvasBlendFunction blend);

/**
 * @brief Set color to specified coordinates
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position in source canvas
 * @param[in]   y       y position in source canvas
 * @param[in]   color   color array
 */
void cv_set_pixel(canvas* self, int x, int y, const void* color);

/**
 * @brief Set color to specified coordinates in the specified color depth
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position in source canvas
 * @param[in]   y       y position in source canvas
 * @param[in]   depth   color depth
 * @param[in]   color   color array
 */
void cv_set_pixel_ex(canvas* self, int x, int y, int depth, const void* color);

/**
 * @brief Set color to specified coordinates in the specified color depth
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position in source canvas
 * @param[in]   y       y position in source canvas
 * @param[in]   color   color array
 * @param[in]   blend   blend function
 */
void cv_blend_pixel(canvas* self, int x, int y, const void* color, canvasBlendFunction blend);

/**
 * @brief Set color to specified coordinates in the specified color depth
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position in the canvas
 * @param[in]   y       y position in the canvas
 * @param[in]   depth   color depth
 * @param[in]   color   color array
 * @param[in]   blend   blend function
 */
void cv_blend_pixel_ex(canvas* self, int x, int y, int depth, const void* color, canvasBlendFunction blend);

/**
 * @brief Set color with blend to specified coordinates following OpenGL manners
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position in the canvas
 * @param[in]   y       y position in the canvas
 * @param[in]   color   color array
 * @param[in]   sfactor sfactor
 * @param[in]   dfactor dfactor
 */
void cv_blend_pixel2(canvas* self, int x, int y, const void* color, int sfactor, int dfactor);

/**
 * @brief Set color with blend to specified coordinates in the specified color depth following OpenGL manners
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position in the canvas
 * @param[in]   y       y position in the canvas
 * @param[in]   depth   color depth
 * @param[in]   color   color array
 * @param[in]   sfactor sfactor
 * @param[in]   dfactor dfactor
 */
void cv_blend_pixel2_ex(canvas* self, int x, int y, int depth, const void* color, int sfactor, int dfactor);

/**
 * @brief Get color from a position
 * The size of the variable 'color' must be the size of color bits.
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position
 * @param[in]   y       y position
 * @param[out]  color   color array
 * @return      Get color to specified coordinates.
 */
void cv_get_pixel(const canvas* self, int x, int y, void* color);

/**
 * @brief Get color from a position with depth
 * The size of the variable 'color' must be the size of color bits.
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position
 * @param[in]   y       y position
 * @param[in]   depth   canvas depth
 * @param[out]  color   color in the specified depth
 * @return      Get color to specified coordinates.
 */
void cv_get_pixel_ex(const canvas* self, int x, int y, int depth, void* color);

/**
 * @brief Draw line from start position to end position
 * The argument 'color' must have the same color-bits of canvas
 * @param[in]   self    canvas pointer
 * @param[in]   sx      start x position
 * @param[in]   sy      start y position
 * @param[in]   ex      end x position
 * @param[in]   ey      end y position
 * @param[in]   color   color array
 */
void cv_draw_line(canvas* self, int sx, int sy, int ex, int ey, const void* color);

/**
 * @brief Draw line from start position to end position with depth
 * using Bresenham's line algorithm.
 *
 * @param[in]   self    canvas pointer
 * @param[in]   sx      start x position
 * @param[in]   sy      start y position
 * @param[in]   ex      end x position
 * @param[in]   ey      end y position
 * @param[in]   depth   color bits of color
 * @param[in]   color   color array
 */
void cv_draw_line_ex(canvas* self, int sx, int sy, int ex, int ey, int depth, const void* color);

/**
 * @brief Draw rectangle
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       start x position
 * @param[in]   y       start y position
 * @param[in]   width   width of rectangle
 * @param[in]   height  height of rectangle
 * @param[in]   color   color array
 * @param[in]   blend   blend function
 */
void cv_draw_rect(canvas* self, int x, int y, int width, int height, const void* color, canvasBlendFunction blend);

/**
 * @brief Draw rectangle width depth
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       start x position
 * @param[in]   y       start y position
 * @param[in]   width   width of rectangle
 * @param[in]   height  height of rectangle
 * @param[in]   depth   canvas depth
 * @param[in]   color   color array
 * @param[in]   blend   blend function
 */
void cv_draw_rect_ex(canvas* self, int x, int y, int width, int height, int depth, const void* color, canvasBlendFunction blend);

/**
 * @brief Set gray8 color to specified coordinates
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position in the canvas
 * @param[in]   y       y position in the canvas
 * @param[in]   color   8-bits color
 */
void cv_set_pixel_gray8(canvas* self, int x, int y, uint8_t color);

/**
 * @brief Set gray16 color to specified coordinates
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position in the canvas
 * @param[in]   y       y position in the canvas
 * @param[in]   color   16-bits color
 */
void cv_set_pixel_gray16(canvas* self, int x, int y, uint16_t color);

/**
 * @brief Set ARGB8 color to specified coordinates
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position in the canvas
 * @param[in]   y       y position in the canvas
 * @param[in]   color   32-bits color
 */
void cv_set_pixel_argb8(canvas* self, int x, int y, uint32_t color);

/**
 * @brief Set ARGB16 color to specified coordinates
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position in the canvas
 * @param[in]   y       y position in the canvas
 * @param[in]   color   64-bits color
 */
void cv_set_pixel_argb16(canvas* self, int x, int y, uint64_t color);

/**
 * @brief Get GRAY8 color from specified coordinates.
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position
 * @param[in]   y       y position
 * @return      Get GRAY8 color from specified coordinates.
 */
uint8_t cv_get_pixel_gray8(const canvas* self, int x, int y);

/**
 * @brief Get GRAY16 color from specified coordinates.
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position
 * @param[in]   y       y position
 * @return      Get GRAY16 color from specified coordinates.
 */
uint16_t cv_get_pixel_gray16(const canvas* self, int x, int y);

/**
 * @brief Get ARGB8 color from specified coordinates.
 *
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position
 * @param[in]   y       y position
 * @return      Get ARGB8 color from specified coordinates.
 */
uint32_t cv_get_pixel_argb8(const canvas* self, int x, int y);

/**
 * @brief Get ARGB16 color from specified coordinates.
 * @param[in]   self    canvas pointer
 * @param[in]   x       x position
 * @param[in]   y       y position
 * @return      Get ARGB16 color from specified coordinates.
 */
uint64_t cv_get_pixel_argb16(const canvas* self, int x, int y);

/**
 * @brief Draw line from start position to end position with Gray8 depth
 * @param[in]   self    canvas pointer
 * @param[in]   sx      start x position
 * @param[in]   sy      start y position
 * @param[in]   ex      end x position
 * @param[in]   ey      end y position
 * @param[in]   color   8-bits color
 */
void cv_draw_line_gray8(canvas* self, int sx, int sy, int ex, int ey, uint8_t color);

/**
 * @brief Draw line from start position to end position with Gray16 depth
 * @param[in]   self    canvas pointer
 * @param[in]   sx      start x position
 * @param[in]   sy      start y position
 * @param[in]   ex      end x position
 * @param[in]   ey      end y position
 * @param[in]   color   16-bits color
 */
void cv_draw_line_gray16(canvas* self, int sx, int sy, int ex, int ey, uint16_t color);

/**
 * @brief Draw line from start position to end position with ARGB8 depth
 * @param[in]   self    canvas pointer
 * @param[in]   sx      start x position
 * @param[in]   sy      start y position
 * @param[in]   ex      end x position
 * @param[in]   ey      end y position
 * @param[in]   color   32-bits color
 */
void cv_draw_line_argb8(canvas* self, int sx, int sy, int ex, int ey, uint32_t color);

/**
 * @brief Draw line from start position to end position with ARGB16 depth
 * @param[in]   self    canvas pointer
 * @param[in]   sx      start x position
 * @param[in]   sy      start y position
 * @param[in]   ex      end x position
 * @param[in]   ey      end y position
 * @param[in]   color   64-bits color
 */
void cv_draw_line_argb16(canvas* self, int sx, int sy, int ex, int ey, uint64_t color);


/**
 * @brief Convert src color from a canvas depth to ARGB16 format
 *
 * @param[in]   src     source color
 * @param[in]   depth   canvas depth
 * @param[out]  dst     destination color
 */
void _cv_convert_to_argb16(const void* src, int depth, uint16_t* dst);

/**
 * @brief Convert color from ARGB16 format to a specified color depth format
 * @param[in]   src     source color
 * @param[in]   depth   canvas depth
 * @param[out]  dst     destination color
 */
void _cv_convert_from_argb16(const uint16_t* src, int depth, void* dst);

__END_DECLS

#endif /* _CANVAS_H */
