/**
 * @file      canvas_fitsio.c
 * @brief     FITS IO support for canvas.
 * @date      2010-02-24
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#ifdef HAVE_CFITSIO

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <assert.h>
#include <fitsio.h>
#include "canvas_fitsio.h"

static int cv_write_fits_gray(canvas* self, fitsfile* fits,
                              canvasFitsioAfterCreateImageCallback cb, void* userdata);
static int cv_write_fits_rgbcube(canvas* self, fitsfile* fits,
                                 canvasFitsioAfterCreateImageCallback cb, void* userdata);
static int cv_write_fits_image_extension(canvas* self, fitsfile* fits,
                                         canvasFitsioAfterCreateImageCallback cb, void* userdata);

canvas* cv_load_fits(const char* path, int attr)
{
  canvas* cv;
  canvas* cv_pool = NULL;
  fitsfile* fits;
  int status = 0;
  int axis = 0;
  int bitpix = 0;
  long whc[3];
  long w, h, c;
  long xyc[] = {1,1,1};
  int bpp;
  canvasColorDepth depth;
  int psize;
  struct stat st;
  FILE* io;

  /* file check */
  if (0 != stat(path, &st)) {
    return NULL;
  }
  if (st.st_mode & S_IFDIR) {
    return NULL;
  }
  io = fopen(path, "rb");
  if (!io) {
    return NULL;
  }
  fclose(io);


  if (0 != fits_open_file(&fits, path, READONLY, &status)) {
    fits_report_error(stderr, status);
    return NULL;
  }
  if (0 != fits_get_img_param(fits, 3, &bitpix, &axis, whc, &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return NULL;
  }
  if (axis != 2 && axis != 3) {
    fits_close_file(fits, &status);
    fprintf(stderr, "WARN: cv_load_file can read 2 or 3-axis FITS file only.\n");
    return NULL;
  }
  w = whc[0];
  h = whc[1];
  c = whc[2];       /* c is only set when axis equal to 3 */
  switch (bitpix) {
  case 8:
    if (axis == 2) {
      depth = CVDepthGray8;
    }
    else if (axis == 3 && c == 3) {
      depth = CVDepthRGB8;
    }
    else {
      fits_close_file(fits, &status);
      fprintf(stderr, "ERROR: unknown fits format.\n");
      return NULL;
    }
    cv = cv_create(w, h, depth);
    bpp = cv->bits_per_pixel/8;
    psize = TBYTE;
    break;
  case 16:
  case 32:        /* scale down to 16bit */
  case -32:       /* scale down to 16bit */
  case -64:       /* scale down to 16bit */
    if (axis == 2) {
      depth = CVDepthGray16;
    }
    else if (axis == 3 && c == 3) {
      depth = CVDepthRGB16;
    }
    else {
      fits_close_file(fits, &status);
      fprintf(stderr, "ERROR: unknown fits format.\n");
      return NULL;
    }
    cv = cv_create(w, h, depth);
    bpp = cv->bits_per_pixel/8;
    psize = TUSHORT;
    break;
  }
  if (fits_read_pix(fits, psize, xyc,
                    cv->buffer_size/bpp, NULL, cv->buffer, NULL, &status)) {
    fits_report_error(stderr, status);
    status = 0;
    cv_free(cv);
    fits_close_file(fits, &status);
    return NULL;
  }
  fits_close_file(fits, &status);

  if (attr & CVFormatBottomUp) {
    /* bottom up format */
    cv_pool = cv;
    cv = cv_flip(cv_pool);
    if (!cv) {
      cv_free(cv_pool);
      return NULL;
    }
  }
  if (cv_pool) cv_free(cv_pool);

  return cv;
}

int cv_save_fits(canvas* self, const char* path, int attr,
                 canvasFitsioAfterCreateImageCallback cb, void* userdata)
{
  fitsfile* fits;
  int status = 0;
  int datatype;
  long whc[3];
  char* wpath;
  int bpp;
  int bitpix;
  int axis;
  long elements;
  canvas* cv = self;

  if (cv_isnull(self)) {
    fprintf(stderr, "%s: we expect non-NULL canvas.\n", __func__);
    abort();
  }

  wpath = malloc(strlen(path)+2);
  if (!wpath) {
    return 0;
  }
  strcpy(wpath, "!");
  strcat(wpath, path);

  switch (cv->depth) {
  case CVDepthRGB8:
  case CVDepthRGBA8:
  case CVDepthARGB8:
    datatype = TBYTE;
    bitpix = BYTE_IMG;
    axis = 3;
    break;
  case CVDepthGray8:
    datatype = TBYTE;
    bitpix = BYTE_IMG;
    axis = 2;
    break;
  case CVDepthRGB16:
  case CVDepthRGBA16:
  case CVDepthARGB16:
    datatype = TUSHORT;
    bitpix = SHORT_IMG;
    axis = 3;
    break;
  case CVDepthGray16:
    datatype = TUSHORT;
    bitpix = SHORT_IMG;
    axis = 2;
    break;
  }
  whc[0] = cv->width;
  whc[1] = cv->height;
  whc[2] = 3;       /* use RGB fits only */
  bpp = cv->bits_per_pixel / 8;
  elements = cv->buffer_size / bpp;

  if (0 != fits_create_file(&fits, wpath, &status)) {
    fits_report_error(stderr, status);
    free(wpath);
    return 0;
  }
  free(wpath);

  if (!cv_write_fits(cv, fits, attr, cb, userdata)) {
    return 0;
  }
  fits_close_file(fits, &status);
  fits_report_error(stderr, status);

  return 1;
}

int cv_write_fits(canvas* self, fitsfile* fits, int attr,
                  canvasFitsioAfterCreateImageCallback cb, void* userdata)
{
  canvas* cv = self;
  canvas* cv_pool = NULL;
  canvas* cv_pool2 = NULL;
  int datatype;
  canvasColorDepth wdep;
  int nplane = 1;
  int format = (attr & CVFormatFitsCube) ? CVFormatFitsCube
               : CVFormatFitsImageExtension;

  if (cv_isnull(self)) {
    fprintf(stderr, "%s: we expect non-NULL canvas.\n", __func__);
    abort();
  }
  wdep = cv->depth;
  switch (cv->depth) {
  case CVDepthGA8:
  case CVDepthAG8:
  case CVDepthGray8:
    datatype = TBYTE;
    break;
  case CVDepthRGB8:
  case CVDepthRGBA8:
  case CVDepthARGB8:
    wdep = CVDepthRGB8;
    nplane = 3;
    datatype = TBYTE;
    break;
  case CVDepthGray16:
  case CVDepthGA16:
  case CVDepthAG16:
    datatype = TUSHORT;
    break;
  case CVDepthRGB16:
  case CVDepthRGBA16:
  case CVDepthARGB16:
    wdep = CVDepthRGB16;
    nplane = 3;
    datatype = TUSHORT;
    break;
  }

  /* convert color depth */
  if (cv->depth != wdep) {
    cv_pool = cv_convert_depth(cv, wdep);
    assert(cv_pool != NULL);
    cv = cv_pool;
  }
  /* flip if needed */
  if (attr & CVFormatBottomUp) {
    cv_pool2 = cv_flip(cv);
    if (!cv_pool2) {
      if (cv_pool) cv_free(cv_pool);
      return 0;
    }
    cv = cv_pool2;
    if (cv_pool) cv_free(cv_pool);
    cv_pool = cv_pool2;
    cv_pool2 = NULL;
  }

  /* Check color. */
  if (wdep == CVDepthGray8 || wdep == CVDepthGray16) {
    cv_write_fits_gray(cv, fits, cb, userdata);
  }
  else {
    if (format & CVFormatFitsCube) {
      cv_write_fits_rgbcube(cv, fits, cb, userdata);
    }
    else {
      cv_write_fits_image_extension(cv, fits, cb, userdata);
    }
  }

  if (cv_pool) cv_free(cv_pool);
  cv_pool = NULL;

  return 1;
}

/**
 * Writre 2-D standard fits.
 */
static int cv_write_fits_gray(canvas* self, fitsfile* fits,
                              canvasFitsioAfterCreateImageCallback cb, void* userdata)
{
  int bpp;
  int elements;
  long xy[] = {1,1};
  long wh[] = {self->width, self->height};
  int datatype;
  int axis = 2;
  int status = 0;
  int bitpix;

  if (cv_isnull(self)) {
    fprintf(stderr, "%s: we expect non-NULL canvas.\n", __func__);
    abort();
  }
  switch (self->depth) {
  case CVDepthGray8:
    datatype = TBYTE;
    bitpix = BYTE_IMG;
    break;
  case CVDepthGray16:
    datatype = TUSHORT;
    bitpix = SHORT_IMG;
    break;
  }

  if (0 != fits_create_img(fits, bitpix, axis, wh, &status)) {
    fits_report_error(stderr, status);
    return 0;
  }
  cb(self, fits, userdata);

  bpp = self->bits_per_pixel / 8;
  elements = wh[0] * wh[1] * bpp;
  if (0 != fits_write_pix(fits, datatype, xy, elements,
                          cv_get_row_ptr(self, 0), &status)) {
    fits_report_error(stderr, status);
    return 0;
  }
  return 1;
}

/**
 * Writre RGB fits as Cube format.
 */
static int cv_write_fits_rgbcube(canvas* self, fitsfile* fits,
                                 canvasFitsioAfterCreateImageCallback cb, void* userdata)
{
  int idx;
  canvas* cv_plane = NULL;
  int bitpix;
  int datatype;
  int bpp;
  int elements;
  int axis = 3;
  long whc[] = {self->width, self->height, 3};
  long xyc[] = {1, 1, 1};
  int status = 0;
  canvasColorPlane colorPlanes[] = {CVColorPlaneRed, CVColorPlaneGreen, CVColorPlaneBlue};

  if (cv_isnull(self)) {
    fprintf(stderr, "%s: we expect non-NULL canvas.\n", __func__);
    abort();
  }

  switch (self->depth) {
  case CVDepthRGB8:
    datatype = TBYTE;
    bitpix = BYTE_IMG;
    break;
  case CVDepthRGB16:
    datatype = TUSHORT;
    bitpix = SHORT_IMG;
    break;
  }

  if (0 != fits_create_img(fits, bitpix, axis, whc, &status)) {
    fits_report_error(stderr, status);
    return 0;
  }
  cb(self, fits, userdata);

  /* RGB -- three plane */
  for (idx = 0; idx < 3; idx++) {
    cv_plane = cv_extract_plane(self, colorPlanes[idx]);
    if (!cv_plane) {
      return 0;
    }
    bpp = cv_plane->bits_per_pixel / 8;
    elements = whc[0] * whc[1] * bpp;
    xyc[2] = idx+1;
    if (0 != fits_write_pix(fits, datatype, xyc, elements,
                            cv_get_row_ptr(cv_plane, 0), &status)) {
      fits_report_error(stderr, status);
      if (cv_plane) cv_free(cv_plane);
      return 0;
    }
    cv_free(cv_plane);
  }
  return 1;
}

/**
 * Write RGB fits as Image Extension format.
 */
static int cv_write_fits_image_extension(canvas* self, fitsfile* fits,
                                         canvasFitsioAfterCreateImageCallback cb, void* userdata)
{
  int idx;
  int status = 0;
  canvasColorPlane colorPlanes[] = {CVColorPlaneRed, CVColorPlaneGreen, CVColorPlaneBlue};
  int bitpix;
  int datatype;
  int bpp;
  int elements;
  canvas* cv;
  long wh[] = {self->width, self->height};
  long xy[] = {1,1};

  if (cv_isnull(self)) {
    fprintf(stderr, "%s: we expect non-NULL canvas.\n", __func__);
    abort();
  }

  switch (self->depth) {
  case CVDepthRGB8:
    datatype = TBYTE;
    bitpix = BYTE_IMG;
    break;
  case CVDepthRGB16:
    datatype = TUSHORT;
    bitpix = SHORT_IMG;
    break;
  }

  /* Create primary HDU. (NULL data) */
  if (fits_create_img(fits, bitpix, 0, wh, &status)) {
    fits_report_error(stderr, status);
    fprintf(stderr, "ERROR: %s:%d\n", __FILE__, __LINE__);
    return 0;
  }
  cb(self, fits, userdata);

  for (idx = 0; idx < 3; idx++) {
    if (fits_create_img(fits, bitpix, 2, wh, &status)) {
      fits_report_error(stderr, status);
      fprintf(stderr, "ERROR: %s:%d\n", __FILE__, __LINE__);
      return 0;
    }
    cb(self, fits, userdata);
    /* Extract color plane. */
    if (!(cv = cv_extract_plane(self, colorPlanes[idx]))) {
      fprintf(stderr, "ERROR: %s:%d\n", __FILE__, __LINE__);
      return 0;
    }
    bpp = cv->bits_per_pixel / 8;
    elements = cv->width * cv->height * bpp;
    if (fits_write_pix(fits, datatype, xy, elements,
                       cv_get_row_ptr(cv, 0), &status)) {
      fits_report_error(stderr, status);
      fprintf(stderr, "ERROR: %s:%d\n", __FILE__, __LINE__);
      cv_free(cv);
      return 0;
    }
    cv_free(cv);
    cv = NULL;
  }

  return 1;
}
#endif
