/**
 * @file      canvas_fitsio.h
 * @brief     FITS IO support for canvas.
 * @date      2017-05-20
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

#ifndef _CANVAS_FITSIO_H
#define _CANVAS_FITSIO_H

#include <fitsio.h>
#include "canvas.h"

/**
 * @def   CVFormatFitsImageExtension
 * @brief Image Extension format used for attr in cv_write_fits().
 */
#define CVFormatFitsImageExtension   0x01000000

/**
 * @def   CVFormatFitsCube
 * @brief RGBCube format used for attr in cv_write_fits().
 */
#define CVFormatFitsCube             0x02000000

typedef int (* canvasFitsioAfterCreateImageCallback)(canvas* cv, fitsfile* fits, void* userdata);

__BEGIN_DECLS

/**
 * @brief Load a FITS file
 * @param[in] path  the file path of a FITS file to load
 * @param[in] attr  attribute mask (CVFormatBottomUp: bottom-up direction)
 * @return cv_load_fits() returns the canvas instance. you must call
 *         canvas_free() at done.
 */
canvas* cv_load_fits(const char* path, int attr);

/**
 * @brief Save an image into a FITS file
 * @param[in] self      canvas instance
 * @param[in] path      the file path of a FITS file to save
 * @param[in] attr      attribute mask (CVFormatBottomUp: bottom-up direction)
 * @param[in] cb        callback function called before image data is writen
 * @param[in] userdata  argument of the callback function
 * @return cv_load_fits() returns the canvas instance. you must call
 */
int cv_save_fits(canvas* self, const char* path, int attr,
                 canvasFitsioAfterCreateImageCallback cb, void* userdata);

/**
 * @brief Write a canvas data into fitsfile object
 * @param[in] self      canvas instance
 * @param[in] fits      fitsfile object
 * @param[in] attr      attribute mask (CVFormatBottomUp: bottom-up direction)
 * @param[in] cb        callback function called before image data is writen
 * @param[in] userdata  image data
 * @return cv_write_fits() writes a canvas data into fitsfile object.
 *         fits format must conform to the following rules:
 *         - naxis = 2
 *         - naxes[0] >= width
 *         - naxes[1] >= height
 *         .
 *         Read the cfitsio manual for more details.
 */
int cv_write_fits(canvas* self, fitsfile* fits, int attr,
                  canvasFitsioAfterCreateImageCallback cb, void* userdata);

__END_DECLS

#endif // _CANVAS_FITSIO_H
