/**
 * @file      canvas_jpegio.c
 * @brief     JPEG IO support for canvas.
 * @date      2010-12-09
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#ifdef HAVE_JPEGLIB_H
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <sys/types.h>
#include "canvas_jpegio.h"

canvas* cv_load_jpeg(const char* path, int attr)
{
  struct jpeg_decompress_struct cinfo;
  struct jpeg_error_mgr jerr;
  canvas* cv = NULL;
  FILE* in = NULL;
  void* ptr;
  JSAMPARRAY buf;
  canvasColorDepth depth;

  /* setup JPEG decompressor */
  cinfo.err = jpeg_std_error(&jerr);
  jpeg_create_decompress(&cinfo);

  /* open file */
  in = fopen(path, "rb");
  if (in == NULL) {
    fprintf(stderr, "ERROR: cv_load_jpeg: can't open %s.\n", path);
    goto cv_load_jpeg_error;
  }
  jpeg_stdio_src(&cinfo, in);
  jpeg_read_header(&cinfo, TRUE);
  jpeg_start_decompress(&cinfo);

  /* create canvas */
  if (1 == cinfo.output_components) {
    depth = CVDepthGray8;
  }
  else if (3 == cinfo.output_components) {
    depth = CVDepthRGB8;
  }
  else {
    fprintf(stderr, "ERROR: cv_load_jpeg: unsupported color components.\n");
    goto cv_load_jpeg_error;
  }
  cv = cv_create(cinfo.image_width, cinfo.image_height, depth);
  if (!cv) {
    fprintf(stderr, "ERROR: can't create canvas.\n");
    goto cv_load_jpeg_error;
  }

  buf = (*cinfo.mem->alloc_sarray)((j_common_ptr)&cinfo, JPOOL_IMAGE,
                                   cinfo.output_width * cinfo.output_components, 1);
  /* extract JPEG to canvas */
  while (cinfo.output_scanline < cinfo.output_height) {
    ptr = cv_get_row_ptr(cv, cinfo.output_scanline);
    jpeg_read_scanlines(&cinfo, buf, 1);
    memcpy(ptr, buf[0], cinfo.output_width * cinfo.output_components);
  }

  /* finish */
  jpeg_finish_decompress(&cinfo);
  jpeg_destroy_decompress(&cinfo);
  fclose(in);
  in = NULL;

  return cv;

  /* ERROR */
cv_load_jpeg_error:
  jpeg_destroy_decompress(&cinfo);
  if (in) fclose(in);
  if (cv) cv_free(cv);
  return NULL;
}

int cv_save_jpeg(canvas* self, const char* path, int quality, int attr,
                 canvasJpegioAfterCreateCallback cb, void* userdata)
{
  FILE* io = NULL;
  int ret;

  if (cv_isnull(self)) {
    fprintf(stderr, "%s: we expect non-NULL canvas.\n", __func__);
    abort();
  }

  io = fopen(path, "wb");
  if (io == NULL) {
    fprintf(stderr, "ERROR: sv_save_jpeg: can't open file %s\n", path);
    return 0;
  }
  ret = cv_save_jpeg_with_std(self, io, quality, attr, cb, userdata);
  fclose(io);
  return ret;
}

int cv_save_jpeg_with_std(canvas* self, FILE* io, int quality, int attr,
                          canvasJpegioAfterCreateCallback cb, void* userdata)
{
  struct jpeg_compress_struct cinfo;
  struct jpeg_error_mgr jerr;

  if (cv_isnull(self)) {
    fprintf(stderr, "%s: we expect non-NULL canvas.\n", __func__);
    abort();
  }

  cinfo.err = jpeg_std_error(&jerr);
  jpeg_create_compress(&cinfo);
  jpeg_stdio_dest(&cinfo, io);
  if (!cv_write_jpeg(self, &cinfo, quality, attr, cb, userdata)) {
    goto cv_save_jpeg_with_std_error;
  }
  fflush(io);
  jpeg_destroy_compress(&cinfo);
  return 1;

  /* ERROR */
cv_save_jpeg_with_std_error:
  jpeg_destroy_compress(&cinfo);
  return 0;
}

int cv_write_jpeg(canvas* self, struct jpeg_compress_struct* cinfo, int quality,
                  int attr, canvasJpegioAfterCreateCallback cb, void* userdata)
{
  canvas* cv_pool = NULL;
  canvas* cv = self;
  int nplane = 1;
  J_COLOR_SPACE color_space = JCS_GRAYSCALE;
  void* ptr;
  JSAMPROW buf[1];

  if (cv_isnull(self)) {
    fprintf(stderr, "%s: we expect non-NULL canvas.\n", __func__);
    abort();
  }

  switch (cv->depth) {
  case CVDepthGA8:
  case CVDepthAG8:
  case CVDepthGA16:
  case CVDepthAG16:
  case CVDepthGray16:
    cv_pool = cv_convert_depth(cv, CVDepthGray8);
    if (!cv_pool) {
      goto cv_write_jpeg_error;
    }
    cv = cv_pool;
    nplane = 1;
    color_space = JCS_GRAYSCALE;
    break;
  case CVDepthRGBA8:
  case CVDepthRGBA16:
  case CVDepthARGB8:
  case CVDepthARGB16:
  case CVDepthRGB16:
    cv_pool = cv_convert_depth(cv, CVDepthRGB8);
    if (!cv_pool) {
      goto cv_write_jpeg_error;
    }
    cv = cv_pool;
    nplane = 3;
    color_space = JCS_RGB;
    break;
  case CVDepthGray8:
    nplane = 1;
    color_space = JCS_GRAYSCALE;
    break;
  case CVDepthRGB8:
    nplane = 3;
    color_space = JCS_RGB;
    break;
  }

  cinfo->image_width = cv->width;
  cinfo->image_height = cv->height;
  cinfo->input_components = nplane;
  cinfo->in_color_space = color_space;
  jpeg_set_defaults(cinfo);
  jpeg_set_quality(cinfo, quality, TRUE);
  if (cb) {
    cb(self, cinfo, userdata);             /* callback function */
  }
  jpeg_start_compress(cinfo, TRUE);
  while (cinfo->next_scanline < cinfo->image_height) {
    ptr = cv_get_row_ptr(cv, cinfo->next_scanline);
    assert(ptr != NULL);
    buf[0] = ptr;
    jpeg_write_scanlines(cinfo, buf, 1);
  }
  jpeg_finish_compress(cinfo);
  return 1;

  /* ERROR */
cv_write_jpeg_error:
  if (cv_pool) {
    cv_free(cv_pool);
  }
  return 0;
}

#endif
