/**
 * @file      canvas_jpegio.h
 * @brief     JPEG IO support for canvas.
 * @date      2017-05-21
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

#ifndef _CANVAS_JPEGIO_H
#define _CANVAS_JPEGIO_H

#include <jpeglib.h>
#include <jerror.h>
#include "canvas.h"

typedef int (* canvasJpegioAfterCreateCallback)(canvas* cv,
                                                struct jpeg_compress_struct* cinfo,
                                                void* userdata);

/**
 * @def   CVFormatJpegProgressive
 * @brief Flag for creation of a progressive JPEG
 */
#define CVFormatJpegProgressive  0x01000000

__BEGIN_DECLS
/**
 * @brief Load a JPEG file
 * @param[in] path  the file path of a JPEG file to load
 * @param[in] attr  attribute mask (not used now)
 * @return cv_load_jpeg() returns the canvas instance. you must call
 *         canvas_free() at done.
 */
canvas* cv_load_jpeg(const char* path, int attr);

/**
 * @brief Save an image into a JPEG file
 * @param[in] self      canvas instance
 * @param[in] path      the file path of a JPEG file to save
 * @param[in] quality   quality flag (low:0 to high:100)
 * @param[in] attr      attribute mask
 *                      (CVFormatJpegProgressive creates a progressive JPEG)
 * @param[in] cb        callback function called before image data is writen
 * @param[in] userdata  argument of the callback function
 * @return cv_save_jpeg() returns 1 if successful, 0 otherwise.
 * @see cv_write_jpeg()
 */
int cv_save_jpeg(canvas* self, const char* path, int quality, int attr,
                 canvasJpegioAfterCreateCallback cb, void* userdata);

/**
 * @brief Save an image into a JPEG file
 * @param[in] self      canvas instance
 * @param[in] io        output stream (file open must be done by mode "wb")。
 * @param[in] quality   quality flag (low:0 to high:100)
 * @param[in] attr      attribute mask
 *                      (CVFormatJpegProgressive creates a progressive JPEG)
 * @param[in] cb        callback function called before image data is writen
 * @param[in] userdata  argument of the callback function
 * @return cv_save_jpeg_wih_std() returns 1 if successful, 0 otherwise.
 * @see cv_write_jpeg()
 */
int cv_save_jpeg_with_std(canvas* self, FILE* io, int quality, int attr,
                          canvasJpegioAfterCreateCallback cb, void* userdata);

/**
 * @brief Write the data to jpeg_compress_struct object
 * @param[in] self      canvas instance
 * @param[in] cinfo     jpeg_compress_struct object
 * @param[in] quality   quality flag (low:0 to high:100)
 * @param[in] attr      attribute mask
 *                      (CVFormatJpegProgressive creates a progressive JPEG)
 * @param[in] cb        callback function called before image data is writen
 * @param[in] userdata  argument of the callback function
 * @return cv_write_jpeg() returns 1 if successful, 0 otherwise.
 * @see cv_write_jpeg()
 */
int cv_write_jpeg(canvas* self, struct jpeg_compress_struct* cinfo, int quality,
                  int attr, canvasJpegioAfterCreateCallback cb, void* userdata);

__END_DECLS

#endif // _CANVAS_JPEGIO_H
