/**
 * @file      canvas_pngio.c
 * @brief     PNG IO support for canvas.
 * @date      2010-05-26
 *
 * @copyright
 * Copyright 2010 AstroArts Inc.
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#ifdef HAVE_LIBPNG

#include <png.h>
#include <zlib.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>
#include "canvas_pngio.h"

#define CV_PNG_LOAD_FILE    1
#define CV_PNG_LOAD_STDIO   2
#define CV_PNG_LOAD_MEMORY  3
typedef struct _canvasPNGLoadParam {
  int type;
  const char* path;
  FILE* io;
  const unsigned char* bytes;
  size_t size;
  size_t pos;
} canvasPNGLoadParam;

canvasPNGLoadParam* _loadParam = NULL;

/**
 * @brief Load data from a file
 *
 * @param[in]  param pointer of parameters
 * @param[in]  path file path
 */
static void cv_png_set_file_load_param(canvasPNGLoadParam* param, const char* path)
{
  param->type = CV_PNG_LOAD_FILE;
  param->path = path;
}

/**
 * @brief Load data from file stream
 *
 * @param[in]  param pointer of parameters
 * @param[in]  io    file stream
 */
static void cv_png_set_stdio_load_param(canvasPNGLoadParam* param, FILE* io)
{
  param->type = CV_PNG_LOAD_STDIO;
  param->io = io;
}

/**
 * @brief Load data from memory
 *
 * @param[in]  param pointer of parameters
 * @param[in]  bytes pointer of memory
 * @param[in]  size  size of memory
 */
static void cv_png_set_memory_load_param(canvasPNGLoadParam* param, const void* bytes, size_t size)
{
  param->type = CV_PNG_LOAD_MEMORY;
  param->bytes = (const unsigned char *)bytes;
  param->size = size;
  param->pos = 0;
}

/**
 * @brief callback function in memory load
 *
 * @param[in]  png_ptr pointer of png structure
 * @param[in]  data    pointer of png data
 * @param[in]  length  length of png data
 */
static void cv_png_user_read_data(png_structp png_ptr, png_bytep data, png_size_t length)
{
  canvasPNGLoadParam* param = _loadParam;
  if (param->size < param->pos + length) {
    png_error(png_ptr, "cv_png_user_read_data(): End of buffer.");
    return;
  }
  memcpy(data, param->bytes+param->pos, length);
  param->pos += length;
}

/**
 * @result -n: chunk has error.
 *          0: did not recognize.
 *          n: success
 */
static int cv_png_read_user_chunk_callback(png_structp ptr, png_unknown_chunkp chunk)
{
  /*
     png_byte name[5];
     png_byte *data;
     png_size_t size;
   */
  return 0;
}

static void cv_png_read_row_callback(png_structp ptr, png_uint_32 row, int pass)
{
}

static void cv_png_error_callback(png_structp png_ptr, png_const_charp errptr)
{
}

static void cv_png_warn_callback(png_structp png_ptr, png_const_charp errptr)
{
}

static void cv_png_write_row_callback(png_structp png_ptr, png_uint_32 row, int pass)
{
}

/**
 * @brief internal common function for PNG loading
 *
 * @param[in]  param pointer of parameters
 */
static canvas* cv_load_png_internal(canvasPNGLoadParam* param)
{
  FILE* io;
  int number = 8;
  unsigned char header[number];
  png_structp png_ptr;
  png_infop info_ptr;
  png_infop end_info;
  int compress_buffer_size = 8192;
  char* chunk = NULL;
  png_bytep* row_pointers;
  int i;
  png_voidp errptr = NULL;
  /* IHDR header */
  png_uint_32 width, height;
  int bit_depth;
  int color_type;
  int interlace_type;
  int compression_type;
  int filter_method;
  int channels;
  int call_fclose = 0;
  /* canvas */
  canvasColorDepth cv_depth;
  canvas* cv;

  if (param->type == CV_PNG_LOAD_FILE) {
    call_fclose = 1;
    io = fopen(param->path, "rb");
    if (!io) {
      fprintf(stderr, "%s(): file %s open error.\n", __func__, param->path);
      return NULL;
    }
  }
  else if (param->type == CV_PNG_LOAD_STDIO) {
    io = param->io;
  }

  if (param->type == CV_PNG_LOAD_FILE || param->type == CV_PNG_LOAD_STDIO) {
    fread(header, 1, number, io);
    if (!png_check_sig((png_bytep)header, number)) {
      if (call_fclose) fclose(io);
      fprintf(stderr, "%s(): stream is not PNG file.\n", __func__);
      return NULL;
    }
  }
  else if (param->type == CV_PNG_LOAD_MEMORY) {
    if (param->size < number ||
        !png_check_sig((png_bytep)param->bytes, number)) {
      fprintf(stderr, "%s(): data is not PNG file.\n", __func__);
      return NULL;
    }
    param->pos = number;
  }

  png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING,
                                   (png_voidp)errptr,
                                   cv_png_error_callback,
                                   cv_png_warn_callback);
  if (!png_ptr) {
    fprintf(stderr, "png_create_read_struct error.\n");
    if (call_fclose) fclose(io);
    return NULL;
  }

  info_ptr = png_create_info_struct(png_ptr);
  if (!info_ptr) {
    fprintf(stderr, "png_create_info_struct error.\n");
    png_destroy_read_struct(&png_ptr, (png_infopp)NULL, (png_infopp)NULL);
    return NULL;
  }

  end_info = png_create_info_struct(png_ptr);
  if (!end_info) {
    fprintf(stderr, "png_create_info_struct error.\n");
    png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
    return NULL;
  }

  if (setjmp(png_jmpbuf(png_ptr))) {
    fprintf(stderr, "setjmp: png_jmpbuf error.\n");
    png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);
    if (call_fclose) fclose(io);
    return NULL;
  }

  if (param->type == CV_PNG_LOAD_FILE || param->type == CV_PNG_LOAD_STDIO) {
    png_init_io(png_ptr, io);
  }
  else if (param->type == CV_PNG_LOAD_MEMORY) {
    png_set_read_fn(png_ptr, &info_ptr, cv_png_user_read_data);
  }
  png_set_sig_bytes(png_ptr, number);

  png_set_compression_buffer_size(png_ptr, compress_buffer_size);

  png_set_read_user_chunk_fn(png_ptr, chunk,
                             cv_png_read_user_chunk_callback);

  png_set_read_status_fn(png_ptr, cv_png_read_row_callback);

  png_set_keep_unknown_chunks(png_ptr, PNG_HANDLE_CHUNK_NEVER, NULL, 0);

  png_read_info(png_ptr, info_ptr);

  png_get_IHDR(png_ptr, info_ptr,
               &width, &height, &bit_depth, &color_type,
               &interlace_type, &compression_type, &filter_method);
  channels = png_get_channels(png_ptr, info_ptr);

  /* expand pixel data */
  if (color_type == PNG_COLOR_TYPE_PALETTE) {
    png_set_palette_to_rgb(png_ptr);
  }
  if (color_type == PNG_COLOR_TYPE_GRAY && bit_depth < 8) {
    png_set_expand_gray_1_2_4_to_8(png_ptr);
  }
  if (png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS)) {
    png_set_tRNS_to_alpha(png_ptr);
  }
#ifndef __BIG_ENDIAN__
  if (bit_depth == 16) {
    png_set_swap(png_ptr);
  }
#endif
  png_read_update_info(png_ptr, info_ptr);

  switch (color_type) {
  case PNG_COLOR_TYPE_PALETTE:
    if (bit_depth <= 8) {
      cv_depth = CVDepthRGB8;
    }
    else {
      cv_depth = CVDepthRGB16;
    }
    break;
  case PNG_COLOR_TYPE_GRAY:
    if (bit_depth <= 8) {
      cv_depth = CVDepthGray8;
    }
    else {
      cv_depth = CVDepthGray16;
    }
    break;
  case PNG_COLOR_TYPE_GRAY_ALPHA:
    if (bit_depth <= 8) {
      cv_depth = CVDepthGA8;
    }
    else {
      cv_depth = CVDepthGA16;
    }
    break;
  case PNG_COLOR_TYPE_RGB:
    if (bit_depth <= 8) {
      cv_depth = CVDepthRGB8;
    }
    else {
      cv_depth = CVDepthRGB16;
    }
    break;
  case PNG_COLOR_TYPE_RGB_ALPHA:
    if (bit_depth <= 8) {
      cv_depth = CVDepthRGBA8;
    }
    else {
      cv_depth = CVDepthRGBA16;
    }
    break;
  }

  cv = cv_create(width, height, cv_depth);
  row_pointers = malloc(sizeof(png_bytep)*height);
  for (i = 0; i < height; i++) {
    row_pointers[i] = cv_get_row_ptr(cv, i);
  }
  png_read_image(png_ptr, row_pointers);

  png_read_end(png_ptr, end_info);
  png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);
  free(row_pointers);
  if (call_fclose) fclose(io);

  return cv;
}

canvas* cv_load_png(const char* path)
{
  canvasPNGLoadParam param;
  cv_png_set_file_load_param(&param, path);
  return cv_load_png_internal(&param);
}

canvas* cv_load_png_with_std(FILE* io)
{
  canvasPNGLoadParam param;
  cv_png_set_stdio_load_param(&param, io);
  return cv_load_png_internal(&param);
}

canvas* cv_load_png_with_bytes(const void* ptr, size_t size)
{
  canvasPNGLoadParam param;
  canvas* cv;
  if (_loadParam) return NULL;
  cv_png_set_memory_load_param(&param, ptr, size);
  _loadParam = &param;
  cv = cv_load_png_internal(&param);
  _loadParam = NULL;
  return cv;
}

int cv_save_png(canvas* self, const char* path, int compression)
{
  FILE* io;
  int ret;

  if (cv_isnull(self)) {
    fprintf(stderr, "%s: we expect non-NULL canvas.\n", __func__);
    abort();
  }

  io = fopen(path, "wb");
  if (!io) {
    return 0;
  }
  ret = cv_save_png_with_std(self, io, compression);
  fclose(io);
  return ret;
}

int cv_save_png_with_std(canvas* self, FILE* io, int compression)
{
  png_structp png_ptr;
  png_infop info_ptr;
  png_bytep* row_pointers;
  char* errptr = NULL;
  int bit_depth = 8;
  int color_type = PNG_COLOR_TYPE_RGB_ALPHA;
  int interlace_type = PNG_INTERLACE_NONE;
  int compression_type = PNG_COMPRESSION_TYPE_DEFAULT;
  int filter_method = PNG_FILTER_TYPE_DEFAULT;
  int i;
  int height;
  canvasColorDepth depth;
  canvas* cv;
  canvas* cv_pool = NULL;

  if (cv_isnull(self)) {
    fprintf(stderr, "%s: we expect non-NULL canvas.\n", __func__);
    abort();
  }

  cv = self;

  switch (cv->depth) {
  case CVDepthGray8:
    bit_depth = 8;
    color_type = PNG_COLOR_TYPE_GRAY;
    break;
  case CVDepthAG8:
    cv_pool = cv_convert_depth(cv, CVDepthGA8);
    cv = cv_pool;
  case CVDepthGA8:
    color_type = PNG_COLOR_TYPE_GRAY_ALPHA;
    bit_depth = 8;
    break;
  case CVDepthRGB8:
    color_type = PNG_COLOR_TYPE_RGB;
    bit_depth = 8;
    break;
  case CVDepthARGB8:
    cv_pool = cv_convert_depth(cv, CVDepthRGBA8);
    cv = cv_pool;
  case CVDepthRGBA8:
    color_type = PNG_COLOR_TYPE_RGB_ALPHA;
    bit_depth = 8;
    break;
  case CVDepthGray16:
    color_type = PNG_COLOR_TYPE_GRAY;
    bit_depth = 16;
    break;
  case CVDepthAG16:
    cv_pool = cv_convert_depth(cv, CVDepthGA16);
    cv = cv_pool;
  case CVDepthGA16:
    color_type = PNG_COLOR_TYPE_GRAY_ALPHA;
    bit_depth = 16;
    break;
  case CVDepthRGB16:
    color_type = PNG_COLOR_TYPE_RGB;
    bit_depth = 16;
    break;
  case CVDepthARGB16:
    cv_pool = cv_convert_depth(cv, CVDepthRGBA16);
    cv = cv_pool;
  case CVDepthRGBA16:
    color_type = PNG_COLOR_TYPE_RGB_ALPHA;
    bit_depth = 16;
    break;
  }

  png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING,
                                    errptr,
                                    cv_png_error_callback,
                                    cv_png_warn_callback);
  if (!png_ptr) {
    fprintf(stderr, "png_create_write_struct error.\n");
    return 0;
  }

  info_ptr = png_create_info_struct(png_ptr);
  if (!info_ptr) {
    fprintf(stderr, "png_create_info_ptr error.\n");
    png_destroy_write_struct(&png_ptr, NULL);
    return 0;
  }

  if (setjmp(png_jmpbuf(png_ptr))) {
    fprintf(stderr, "setjmp: png_jmpbuf error.\n");
    png_destroy_write_struct(&png_ptr, &info_ptr);
    return 0;
  }

  png_init_io(png_ptr, io);
  /* png_set_sig_bytes(png_ptr, 8); for MNG */
  png_set_write_status_fn(png_ptr, cv_png_write_row_callback);
  png_set_filter(png_ptr, 0, PNG_FILTER_NONE);
  png_set_compression_level(png_ptr, Z_BEST_COMPRESSION);   /* or compression */

  /* set other zlib parameters */
  png_set_compression_mem_level(png_ptr, 8);
  png_set_compression_strategy(png_ptr,
                               Z_DEFAULT_STRATEGY);
  png_set_compression_window_bits(png_ptr, 15);
  png_set_compression_method(png_ptr, 8);
  png_set_compression_buffer_size(png_ptr, 8192);

  png_set_IHDR(png_ptr, info_ptr,
               cv->width, cv->height,
               bit_depth, color_type, interlace_type,
               compression_type, filter_method);

  height = cv->height;
  depth = cv->depth;
  png_write_info(png_ptr, info_ptr);
  row_pointers = malloc(sizeof(png_bytep)*height);
  for (i = 0; i < height; i++) {
    row_pointers[i] = cv_get_row_ptr(cv, i);
  }
  png_write_image(png_ptr, row_pointers);
  png_write_end(png_ptr, info_ptr);

  png_destroy_write_struct(&png_ptr, &info_ptr);
  fflush(io);
  free(row_pointers);
  if (cv_pool) cv_free(cv_pool);

  return 1;
}
#endif
