/**
 * @file      epd.c
 * @brief     ExPanDable file format
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2011 Japan Aerospace Exploration Agency
 *
 */

/*
 * EPD = HEADER BLOCK (BLOCK | JOINED_BLOCK)
 *
 * ---HEAER
 * char[4]     // 4 byte file magic "EPDF"
 * uint32_t    // version major = v>>16, minor = ((v>>8)&0xFF), release = (v&0xFF)
 * uint32_t    // block size (byte)
 * uint32_t    // header size (byte)
 * ---
 * ---BLOCK (block size)
 * char        // block type. {}
 * char        // whether block used. 1: used, 0: unused
 * char[2]     // (reserved)
 * uint32_t    // chunk size. (byte)
 * uint32_t    // number of blocks joined. joined block does not have chunk header.
 * uint32_t    // chunk's byte offset.
 * uint32_t    // chunk's first block index.
 * uint32_t    // chunk's next block index.
 * uint32_t    // chunk's previous block index.
 * void *      // data
 * ---
 * ---JOINED_BLOCK (block size)
 * void *      // data
 * ---
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <fcntl.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <assert.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#if defined(__MINGW32__)
#include <winsock.h>
#else
#include <sys/uio.h>
#include <arpa/inet.h>
#endif

#include "epd.h"

#ifdef HAVE_FDATASYNC
#  define wrap_fdatasync(fd)  fdatasync(fd)
#else
#  define wrap_fdatasync(fd)  fsync(fd)
#endif

#define epd_debug_info(s)   (EPD_DEBUG_LEVEL_INFO   <= (s)->debug_level)
#define epd_debug_warn(s)   (EPD_DEBUG_LEVEL_WARN   <= (s)->debug_level)
#define epd_debug_notice(s) (EPD_DEBUG_LEVEL_NOTICE <= (s)->debug_level)
#define epd_debug_error(s)  (EPD_DEBUG_LEVEL_ERROR  <= (s)->debug_level)

typedef struct _epd_head_store {
  char magic[EPD_MAGIC_SIZE];
  char __reserved[112];
  uint32_t version;
  uint32_t block_size;
  uint32_t block_start;
} epd_head_store; /* 128 byte */
#define EPD_HEADER_SIZE  sizeof(epd_head_store)

typedef struct _epd_chunk_head_store {
  char __reserved[2];
  uint32_t size;
  uint32_t njoin;
  uint32_t offset;
  uint32_t start;
  uint32_t next;
  uint32_t previous;
  char type;
  char used;
} epd_chunk_head_store;

static off_t epd_chunk_head_pos(epd* self, uint32_t index);
static off_t epd_chunk_data_pos(epd* self, uint32_t index);
static size_t epd_chunk_part_size(epd* self, epd_chunk_head* chead);
static epd_status epd_read_header(epd* self);
static epd_status epd_write_header(epd* self);
static epd* epd_create(const epd_config* conf);
static void epd_free(epd* self);
static epd_status epd_expand(epd* self, epd_chunk_head* chead, size_t size);

static epd* epd_create(const epd_config* conf)
{
  static epd_config conf_default = {
    EPD_DEFAULT_BLOCK_SIZE,
    EPD_FALSE,
    EPD_DEBUG_LEVEL_NONE
  };
  epd* self = malloc(sizeof(epd));
  if (!self) {
    return NULL;
  }
  memset(self, 0, sizeof(*self));
  if (!conf) {
    conf = &conf_default;
  }
  self->config = *conf;
  self->block_start = EPD_HEADER_SIZE;
  self->block_size = self->config.block_size;
  self->version = EPD_VERSION;
  self->debug_level = self->config.debug_level;
  self->write_sync = self->config.write_sync;
  return self;
}

static void epd_free(epd* self)
{
  if (self) {
    free(self);
  }
}

epd_status epd_close(epd* self)
{
  assert(0 < self->fd);
  if (0 != close(self->fd)) {
    return EPD_FILE_CLOSE_ERROR;
  }
  self->fd = 0;
  self->block_size = self->config.block_size;
  self->version = EPD_VERSION;
  epd_free(self);
  return EPD_NO_ERROR;
}

epd* epd_open(const char* path, epd_opmode mode, const epd_config* conf)
{
  epd* self = NULL;
  int fd = 0;
  int opmode = 0;
  int read_only = 0;
  int is_new = 0;
  epd_status err;
  struct stat st;

  if (EPD_RDWR == (mode & EPD_RDWR)) {
    opmode |= O_RDWR;
  }
  else if (mode & EPD_RDONLY) {
    opmode |= O_RDONLY;
    read_only = 1;
  }
  if (!opmode) {
    return NULL;             //EPD_INVALID_OPEN_MODE;
  }
  if (mode & EPD_CREAT) {
    opmode |= O_CREAT;
  }

  if (0 == stat(path, &st)) {
    if (S_IFREG != (st.st_mode & S_IFMT)) {
      return NULL;                   //EPD_INVALID_FORMAT;
    }
  }
  else {
    switch (errno) {
    case ENOENT:
      is_new = 1;
      break;
    case EACCES:
      return NULL;                   //EPD_PERMISSION_DENIED;
    default:
      return NULL;                   //EPD_FILE_OPEN_ERROR;
    }
  }

  fd = open(path, opmode, 0777);
  if (-1 == fd) {
    switch (errno) {
    case EACCES:
      return NULL;                   //EPD_PERMISSION_DENIED;
    case ENOENT:
      return NULL;                   //EPD_FILE_NOT_FOUND;
    default:
      return NULL;                   //EPD_FILE_OPEN_ERROR;
    }
  }

  self = epd_create(conf);
  if (!self) {
    close(fd);
    return NULL;             //EPD_MEMORY_FULL
  }
  self->fd = fd;
  self->mode = mode;

  if (is_new) {
    /* write header */
    err = epd_write_header(self);
    if (EPD_NO_ERROR != err) {
      epd_close(self);
      return NULL;                   //EPD_READ_ERROR
    }
  }
  else {
    /* read header */
    err = epd_read_header(self);
    if (EPD_NO_ERROR != err) {
      epd_close(self);
      return NULL;                   //EPD_READ_ERROR
    }
  }

  return self;
}

static off_t epd_chunk_head_pos(epd* self, uint32_t index)
{
  return self->block_start + self->block_size * index;
}

static off_t epd_chunk_data_pos(epd* self, uint32_t index)
{
  return self->block_start + self->block_size * index + sizeof(epd_chunk_head_store);
}

static size_t epd_chunk_part_size(epd* self, epd_chunk_head* chead)
{
  return self->block_size * chead->njoin - sizeof(epd_chunk_head_store);
}

static epd_status epd_read_header(epd* self)
{
  epd_head_store head;
  ssize_t nread;
  assert(0 < self->fd);

  if ((off_t)-1 == lseek(self->fd, 0, SEEK_SET)) {
    return EPD_SEEK_ERROR;
  }
  nread = read(self->fd, &head, sizeof(head));
  if (nread != sizeof(head)) {
    return EPD_READ_ERROR;
  }
  if (0 != memcmp(head.magic, EPD_MAGIC, 4)) {
    return EPD_INVALID_FORMAT;
  }
  self->block_size  = ntohl(head.block_size);
  self->block_start = ntohl(head.block_start);
  self->version     = ntohl(head.version);
  if (self->version != EPD_VERSION) {
    return EPD_INVALID_VERSION;
  }
  return EPD_NO_ERROR;
}

static epd_status epd_write_header(epd* self)
{
  ssize_t nwrite;
  epd_head_store head;
  assert(0 < self->fd);

  if (0 == (self->mode & EPD_WRONLY)) {
    return EPD_WRITE_NOT_ALLOW;
  }
  if (0 != ftruncate(self->fd, 0)) {
    return EPD_WRITE_ERROR;
  }
  memset(&head, 0, sizeof(head));
  memcpy(head.magic, EPD_MAGIC, EPD_MAGIC_SIZE);
  head.version     = htonl(self->version);
  head.block_size  = htonl(self->block_size);
  head.block_start = htonl(sizeof(head));
  nwrite = write(self->fd, &head, sizeof(head));
  if (nwrite != sizeof(head)) {
    return EPD_WRITE_ERROR;
  }
  if (self->write_sync) {
    wrap_fdatasync(self->fd);
  }
  return EPD_NO_ERROR;
}

epd_status epd_read_chunk_head(epd* self, int32_t index, epd_chunk_head* chead)
{
  epd_chunk_head_store chunk_head;
  off_t pos;
  ssize_t nread;
  struct stat st;
  assert(0 < self->fd);

  if (epd_debug_info(self)) {
    fprintf(stderr, "epd_read_chunk_head: index: %u\n", index);
  }

  if (0 != fstat(self->fd, &st)) {
    return EPD_READ_ERROR;
  }
  pos = epd_chunk_head_pos(self, index);
  if (st.st_size <= pos) {
    return EPD_CHUNK_NOT_FOUND;
  }
  if ((off_t)-1 == lseek(self->fd, pos, SEEK_SET)) {
    return EPD_SEEK_ERROR;
  }
  nread = read(self->fd, &chunk_head, sizeof(chunk_head));
  if (nread != sizeof(chunk_head)) {
    return EPD_READ_ERROR;
  }
  chead->type     = chunk_head.type;
  chead->used     = chunk_head.used;
  chead->size     = ntohl(chunk_head.size);
  chead->njoin    = ntohl(chunk_head.njoin);
  chead->offset   = ntohl(chunk_head.offset);
  chead->start    = ntohl(chunk_head.start);
  chead->next     = ntohl(chunk_head.next);
  chead->previous = ntohl(chunk_head.previous);
  chead->index  = index;
  return EPD_NO_ERROR;
}

epd_status epd_read(epd* self, epd_chunk_head* chead, void* data)
{
  off_t pos;
  epd_chunk_head head;
  epd_chunk_head* h;
  ssize_t nread;
  ssize_t sum_read;
  ssize_t size;
  size_t rest;
  epd_status err;
  assert(0 < self->fd);

  if (epd_debug_info(self)) {
    fprintf(stderr, "epd_read: index: %d  size: %d\n", chead->index, chead->size);
  }

  sum_read = 0;
  h = chead;
  rest = chead->size;
  while (1) {
    size = epd_chunk_part_size(self, h);
    size = (rest < size) ? rest : size;
    pos = epd_chunk_data_pos(self, h->index);
    if ((off_t)-1 == lseek(self->fd, pos, SEEK_SET)) return EPD_SEEK_ERROR;
    if (epd_debug_info(self)) {
      fprintf(stderr, "  pos: %u\n", (unsigned)pos);
    }
    nread = read(self->fd, data+sum_read, size);
    if (nread != size) {
      return EPD_READ_ERROR;
    }
    sum_read += nread;
    rest -= size;
    if (0 == rest) break;
    err = epd_read_chunk_head(self, h->next, &head);
    if (EPD_NO_ERROR != err) {
      return err;
    }
    h = &head;
  }

  if (epd_debug_info(self)) {
    int len = (16 < chead->size) ? 16 : chead->size;
    int i;
    const unsigned char* p = data;
    fprintf(stderr, "  data:");
    for (i = 0; i < len; i++) fprintf(stderr, " %02x", p[i]);
    if (len < chead->size) fprintf(stderr, " ...");
    fprintf(stderr, "\n");
  }
  return EPD_NO_ERROR;
}

static epd_status epd_write_chunk_head(epd* self, const epd_chunk_head* chead)
{
  off_t pos;
  epd_chunk_head_store hs;
  size_t nwrite;
  assert(0 < self->fd);

  if (epd_debug_info(self)) {
    fprintf(stderr, "epd_write_chunk_head: index: %u\n", chead->index);
  }

  memset(&hs, 0, sizeof(hs));
  pos = epd_chunk_head_pos(self, chead->index);
  if ((off_t)-1 == lseek(self->fd, pos, SEEK_SET)) return EPD_SEEK_ERROR;
  hs.type     = chead->type;
  hs.used     = chead->used;
  hs.size     = htonl(chead->size);
  hs.njoin    = htonl(chead->njoin);
  hs.offset   = htonl(chead->offset);
  hs.start    = htonl(chead->start);
  hs.next     = htonl(chead->next);
  hs.previous = htonl(chead->previous);
  nwrite = write(self->fd, &hs, sizeof(hs));
  if (nwrite != sizeof(hs)) {
    return EPD_WRITE_ERROR;
  }
  if (self->write_sync) {
    wrap_fdatasync(self->fd);
  }
  return EPD_NO_ERROR;
}

epd_status epd_reserve(epd* self, size_t size, epd_chunk_head* chead)
{
  size_t nblock;
  char buf[8192];
  off_t pos;
  size_t nwrite;
  uint32_t njoin;
  int32_t rest;
  size_t wsize;
  epd_chunk_head ch;
  epd_status err;
  assert(0 < self->fd);

  if (epd_debug_info(self)) {
    fprintf(stderr, "epd_reserve: size: %zd\n", size);
  }

  pos = lseek(self->fd, 0, SEEK_END);
  if (pos == (off_t)-1) {
    return EPD_SEEK_ERROR;
  }
  nblock = (pos - self->block_start) / self->block_size;
  njoin = 1;
  rest = size - (self->block_size - sizeof(epd_chunk_head_store));
  if (0 < rest) {
    njoin += rest / self->block_size;
    if (0 != rest % self->block_size) njoin += 1;
  }
  ch.type     = EPD_CHUNK_START;
  ch.used     = EPD_TRUE;
  ch.index    = nblock;
  ch.size     = 0;
  ch.njoin    = njoin;
  ch.offset   = 0;
  ch.start    = nblock;
  ch.next     = EPD_EMPTY_INDEX;
  ch.previous = EPD_EMPTY_INDEX;
  err = epd_write_chunk_head(self, &ch);
  if (EPD_NO_ERROR != err) {
    ftruncate(self->fd, pos);
    return err;
  }
  rest = self->block_size * njoin - sizeof(epd_chunk_head_store);
  memset(buf, 0, sizeof(buf));
  while (0 < rest) {
    wsize = (sizeof(buf) < rest) ? sizeof(buf) : rest;
    nwrite = write(self->fd, buf, wsize);
    if (nwrite != wsize) {
      ftruncate(self->fd, pos);
      return EPD_WRITE_ERROR;
    }
    rest -= nwrite;
  }
  *chead = ch;
  if (self->write_sync) {
    wrap_fdatasync(self->fd);
  }
  return EPD_NO_ERROR;
}

epd_status epd_unlink(epd* self, epd_chunk_head* chead)
{
  epd_chunk_head head;
  epd_chunk_head* h;
  uint32_t next_index;
  epd_status err;
  assert(0 < self->fd);

  h = chead;
  while (1) {
    next_index = h->next;
    h->next     = EPD_EMPTY_INDEX;
    h->previous = EPD_EMPTY_INDEX;
    h->start    = EPD_EMPTY_INDEX;
    h->used     = 0;
    err = epd_write_chunk_head(self, h);
    if (EPD_NO_ERROR != err) {
      return err;
    }
    if (EPD_EMPTY_INDEX == next_index) {
      break;
    }
    err = epd_read_chunk_head(self, next_index, &head);
    if (EPD_NO_ERROR != err) {
      return err;
    }
    h = &head;
  }
  if (self->write_sync) {
    wrap_fdatasync(self->fd);
  }
  return EPD_NO_ERROR;
}

epd_status epd_write(epd* self, epd_chunk_head* chead, size_t size, const void* data)
{
  epd_chunk_head* h;
  epd_chunk_head head;
  size_t rest;
  epd_status err;
  size_t wsize;
  size_t nwrite;
  size_t bsize;
  size_t sum_write;
  off_t pos;
  assert(0 < self->fd);

  if (epd_debug_info(self)) {
    int len = (16 < size) ? 16 : size;
    int i;
    const unsigned char* p = data;
    fprintf(stderr, "epd_write: index: %d  size: %zd\n",
            chead->index, size);
    fprintf(stderr, "  data:");
    for (i = 0; i < len; i++) fprintf(stderr, " %02x", p[i]);
    if (len < size) fprintf(stderr, " ...");
    fprintf(stderr, "\n");
  }

  err = epd_expand(self, chead, size);
  if (EPD_NO_ERROR != err) return err;
  rest = size;
  h = chead;
  sum_write = 0;
  while (1) {
    pos = epd_chunk_data_pos(self, chead->index);
    bsize = epd_chunk_part_size(self, h);
    wsize = (bsize < rest) ? bsize : rest;
    h->size = size;
    err = epd_write_chunk_head(self, h);
    if (EPD_NO_ERROR != err) return err;
    if ((off_t)-1 == lseek(self->fd, pos, SEEK_SET)) return EPD_SEEK_ERROR;
    if (0 < rest) {
      if (epd_debug_info(self)) {
        fprintf(stderr, "  pos: %u\n", (unsigned)pos);
      }
      nwrite = write(self->fd, data+sum_write, wsize);
      if (nwrite != wsize) return EPD_WRITE_ERROR;
      sum_write += wsize;
      rest -= wsize;
    }
    if (h->next == EPD_EMPTY_INDEX || h->index == h->next) break;
    err = epd_read_chunk_head(self, h->next, &head);
    if (EPD_NO_ERROR != err) return err;
    h = &head;
  }
  if (self->write_sync) {
    wrap_fdatasync(self->fd);
  }
  return EPD_NO_ERROR;
}

static epd_status epd_expand(epd* self, epd_chunk_head* chead, size_t size)
{
  epd_chunk_head* h;
  epd_chunk_head ch;
  size_t psize;
  size_t capacity;
  epd_status err;
  assert(0 < self->fd);

  h = chead;
  capacity = 0;
  while (1) {
    psize = epd_chunk_part_size(self, h);
    capacity += psize;
    if (size <= capacity) {
      return EPD_NO_ERROR;
    }
    if (EPD_EMPTY_INDEX == h->next) break;

    err = epd_read_chunk_head(self, h->next, &ch);
    if (EPD_NO_ERROR != err);
  }
  if (capacity < size) {
    epd_reserve(self, capacity - size, &ch);
    h->next = ch.index;
    ch.type = EPD_CHUNK_PART;
    ch.previous = h->index;
    ch.offset = h->offset + psize;
    ch.start = h->start;

    err = epd_write_chunk_head(self, h);
    if (EPD_NO_ERROR != err) return err;

    err = epd_write_chunk_head(self, &ch);
    if (EPD_NO_ERROR != err) return err;
  }
  if (self->write_sync) {
    wrap_fdatasync(self->fd);
  }
  return EPD_NO_ERROR;
}

size_t epd_pread(epd* self, epd_chunk_head* chead, size_t offset, size_t size, void* data)
{
  off_t pos, pos2;
  epd_chunk_head* h;
  epd_chunk_head ch;
  size_t skip;
  size_t nread;
  size_t csize;
  size_t rsize;
  size_t rest;
  size_t sum_read = 0;
  epd_status err;
  assert(0 < self->fd);

  if (epd_debug_info(self)) {
    fprintf(stderr, "epd_pread: index: %u  offset: %zd  size: %zd\n",
            chead->index, offset, size);
  }

  if (chead->size < offset) {
    return sum_read;
  }
  if (chead->size < offset+size) {
    size = chead->size - offset;
  }
  skip = offset;
  h = chead;
  while (1) {
    csize = epd_chunk_part_size(self, h);
    if (skip < csize) {
      break;
    }
    skip -= csize;
    err = epd_read_chunk_head(self, h->next, &ch);
    if (EPD_NO_ERROR != err) {
      return (size_t)-1;
    }
    h = &ch;
  }
  rest = size;
  while (1) {
    csize = (h->next == EPD_EMPTY_INDEX) ?
            h->size - h->offset : epd_chunk_part_size(self, h);
    pos = epd_chunk_data_pos(self, h->index) + skip;
    pos2 = lseek(self->fd, pos, SEEK_SET);
    if (pos2 == (off_t)-1) {
      return (size_t)-1;
    }
    rsize = csize - skip;
    rsize = (rest < rsize) ? rest : rsize;
    nread = read(self->fd, data+sum_read, rsize);
    if (nread != rsize) {
      return (size_t)-1;
    }
    rest -= rsize;
    sum_read += rsize;
    skip = 0;
    if (0 == rest) {
      break;
    }
    err = epd_read_chunk_head(self, h->next, &ch);
    if (EPD_NO_ERROR != err) {
      return (size_t)-1;
    }
    h = &ch;
  }

  if (epd_debug_info(self)) {
    int len = (16 < size) ? 16 : size;
    int i;
    const unsigned char* p = data;
    fprintf(stderr, "  data:");
    for (i = 0; i < len; i++) fprintf(stderr, " %02x", p[i]);
    if (len < size) fprintf(stderr, " ...");
    fprintf(stderr, "\n");
  }

  return sum_read;
}

epd_status epd_pwrite(epd* self, epd_chunk_head* chead, size_t offset, size_t size, void* data)
{
  off_t pos, pos2;
  epd_chunk_head* h;
  epd_chunk_head ch;
  size_t skip;
  size_t csize;
  size_t wsize;
  size_t nwrite;
  size_t rest;
  size_t sum_write = 0;
  epd_status err;
  int update_size = EPD_FALSE;
  uint32_t new_size;
  assert(0 < self->fd);

  if (epd_debug_info(self)) {
    int len = (16 < size) ? 16 : size;
    int i;
    const unsigned char* p = data;
    fprintf(stderr, "epd_pwrite: index: %u  offset: %zd  size: %zd\n",
            chead->index, offset, size);
    fprintf(stderr, "  data:");
    for (i = 0; i < len; i++) fprintf(stderr, " %02x", p[i]);
    if (len < size) fprintf(stderr, " ...");
    fprintf(stderr, "\n");
  }

  err = epd_expand(self, chead, offset + size);
  if (err != EPD_NO_ERROR) {
    return err;
  }
  if (chead->size < offset + size) {
    update_size = EPD_TRUE;
    new_size = offset + size;
  }

  h = chead;
  skip = offset;
  while (1) {
    csize = epd_chunk_part_size(self, h);
    if (update_size) {
      h->size = new_size;
      err = epd_write_chunk_head(self, h);
      if (EPD_NO_ERROR != err) return err;
    }
    if (skip < csize) break;
    skip -= csize;
    err = epd_read_chunk_head(self, h->next, &ch);
    if (EPD_NO_ERROR != err) return err;
    h = &ch;
  }
  rest = size;
  sum_write = 0;
  while (1) {
    csize = epd_chunk_part_size(self, h);
    pos = epd_chunk_data_pos(self, h->index) + skip;
    pos2 = lseek(self->fd, pos, SEEK_SET);
    if ((off_t)-1 == pos2) return EPD_SEEK_ERROR;
    wsize = csize - skip;
    wsize = (rest < wsize) ? rest : wsize;
    nwrite = write(self->fd, data+sum_write, wsize);
    if (nwrite != wsize) return EPD_WRITE_ERROR;
    rest -= wsize;
    sum_write += wsize;
    skip = 0;
    if (0 == rest) break;
    err = epd_read_chunk_head(self, h->next, &ch);
    if (EPD_NO_ERROR != err) return err;
    h = &ch;
  }
  h = chead;
  if (update_size) {
    while (1) {
      h->size = new_size;
      err = epd_write_chunk_head(self, h);
      if (EPD_NO_ERROR != err) return err;
      if (h->next == EPD_EMPTY_INDEX) break;
      err = epd_read_chunk_head(self, h->next, &ch);
      if (EPD_NO_ERROR != err) return err;
      h = &ch;
    }
  }
  if (self->write_sync) {
    wrap_fdatasync(self->fd);
  }
  return EPD_NO_ERROR;
}

epd_status epd_add(epd* self, size_t size, void* data, epd_chunk_head* chead)
{
  off_t pos;
  uint32_t nblock;
  size_t nwrite;
  uint32_t njoin;
  uint32_t rest;
  epd_status err;
  epd_chunk_head ch;
  assert(0 < self->fd);

  if (epd_debug_info(self)) {
    fprintf(stderr, "epd_add: size: %zd\n", size);
  }

  pos = lseek(self->fd, 0, SEEK_END);
  if (pos == (off_t)-1) {
    return EPD_SEEK_ERROR;
  }
  nblock = (pos - self->block_start) / self->block_size;
  njoin = 1;
  rest = size - (self->block_size - sizeof(epd_chunk_head_store));
  if (0 < rest) {
    njoin += rest / self->block_size;
    if (0 != rest % self->block_size) njoin += 1;
  }
  ch.type      = EPD_CHUNK_START;
  ch.used      = EPD_TRUE;
  ch.index     = nblock;
  ch.size      = size;
  ch.njoin     = njoin;
  ch.offset    = 0;
  ch.start     = nblock;
  ch.next      = EPD_EMPTY_INDEX;
  ch.previous  = EPD_EMPTY_INDEX;
  err = epd_write_chunk_head(self, &ch);
  pos = epd_chunk_data_pos(self, ch.index);
  if ((off_t)-1 == lseek(self->fd, pos, SEEK_SET)) return EPD_SEEK_ERROR;
  nwrite = write(self->fd, data, size);
  if (nwrite != size) {
    ftruncate(self->fd, pos);
    return EPD_WRITE_ERROR;
  }
  *chead = ch;
  if (self->write_sync) {
    wrap_fdatasync(self->fd);
  }
  return EPD_NO_ERROR;
}

void epd_dump_header(epd* self)
{
  epd_chunk_head head;
  struct stat st;
  size_t size;
  size_t nblock;
  size_t i;
  epd_status err;

  if (0 != fstat(self->fd, &st)) {
    return;
  }
  size = st.st_size;
  nblock = (size - self->block_start) / self->block_size;

  for (i = 0; i < nblock; i += head.njoin) {
    err = epd_read_chunk_head(self, i, &head);
    if (EPD_NO_ERROR != err) return;
    printf("%8u: t:%d u:%d sz:%8u o:%u j:%6u st:%8u pv:%8d nx:%8d\n",
           head.index,
           head.type,
           head.used,
           head.size,
           head.offset,
           head.njoin,
           head.start,
           head.previous,
           head.next);

  }
}
