/**
 * @file      epdb.c
 * @brief     EPD based hash table.
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2010-2011 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#if defined(__MINGW32__)
#include <winsock.h>
#else
#include <arpa/inet.h>
#endif
#include "epdb.h"

#define EPDB_MAGIC      "EPDB"
#define EPDB_MAGIC_SIZE      4

#define EPDB_HEAD_INDEX   0

typedef struct _epdb_head_store {
  char magic[4];                       /* 4 byte magic: "EPDB" */
  uint32_t table_size;                 /* table size */
  uint32_t hash;                       /* "EPDB" hash result */
} epdb_head_store;

typedef struct _epdb_table_array {
  int32_t ki,vi;
} epdb_table_array;

#ifdef EPDB_MALLOC_DEBUG
static void* epdb_malloc(size_t n)
{
  void* p;
  fprintf(stderr, "malloc: size: %u ", n);
  p = malloc(n);
  fprintf(stderr, "(0x%08x)\n", (unsigned)p);
  return p;
}
#else
#  define epdb_malloc(p)  malloc(p)
#endif


static epdb_status epdb_read_head(epdb* self, epd_chunk_head* c_head);
static epdb_status epdb_create_head(epdb* self);
static epdb_pair* epdb_pair_create(epdb_data* key, epdb_data* value);
static epdb_status epdb_get_table_array_head(epdb* self, epdb_data* key, epd_chunk_head* head);
static epdb_status epdb_get_table_array(epdb* self, epd_chunk_head* head, epdb_table_array* array);
static epdb_status epdb_search_table_array(epdb* self, epdb_data* key, const epdb_table_array* array, uint32_t narray, uint32_t* index);



const epdb_config* epdb_config_default(void)
{
  static epdb_config conf = {
    EPDB_TABLE_MEDIUM,
    epdb_default_hash_function,
    (EPD_RDWR|EPD_CREAT),
    EPD_DEFAULT_BLOCK_SIZE,
    EPD_TRUE,
    EPD_DEBUG_LEVEL_NONE,
  };
  return &conf;
}

void epdb_config_set_default(epdb_config* conf)
{
  *conf = *(epdb_config_default());
}

epdb* epdb_open(const char* path, const epdb_config* conf)
{
  epdb* self;
  epd_chunk_head c_head;
  epd_status err;
  epd_config fconf;

  self = epdb_malloc(sizeof(*self));
  if (!self) {
    return NULL;
  }
  memset(self, 0, sizeof(*self));
  fconf.write_sync = conf->write_sync;
  fconf.block_size = conf->block_size;
  fconf.debug_level = conf->debug_level;
  self->db = epd_open(path, conf->open_mode, &fconf);
  if (!self->db) {
    free(self);
    return NULL;
  }
  self->hash = (conf->hash) ? conf->hash : epdb_default_hash_function;
  self->table_size = conf->table_size;
  err = epd_read_chunk_head(self->db, EPDB_HEAD_INDEX, &c_head);
  if (err == EPD_NO_ERROR) {
    /* existed file */
    err = epdb_read_head(self, &c_head);
    if (EPDB_NO_ERROR != err) goto epdb_open_error;
  }
  else if (err == EPD_CHUNK_NOT_FOUND) {
    /* new file */
    err = epdb_create_head(self);
    if (EPDB_NO_ERROR != err) goto epdb_open_error;
  }
  else {
    goto epdb_open_error;
  }
  return self;

epdb_open_error:
  if (self) epdb_close(self);
  return NULL;
}

epdb_status epdb_close(epdb* self)
{
  epdb_status err;
  if (self) {
    epdb_flush(self);
    if (self->table) {
      free(self->table);
      self->table = NULL;
    }
    err = epd_close(self->db);
    if (!err) {
      return err;
    }
    self->db = NULL;
    free(self);
  }
  return EPDB_NO_ERROR;
}

epdb_status epdb_flush(epdb* self)
{
  epdb_status err;
  epd_chunk_head head;
  size_t tsize;
  int i;

  if (self->db && self->table) {
    tsize = self->table_size * sizeof(int32_t);
    err = epd_read_chunk_head(self->db, EPDB_HEAD_INDEX, &head);
    if (EPD_NO_ERROR != err) return err;
    for (i = 0; i < self->table_size; i++) self->table[i] = htonl(self->table[i]);
    err = epd_pwrite(self->db, &head, sizeof(epdb_head_store), tsize, self->table);
    for (i = 0; i < self->table_size; i++) self->table[i] = ntohl(self->table[i]);
    if (EPD_NO_ERROR != err) return err;
  }
  return EPDB_NO_ERROR;
}

static epdb_status epdb_read_head(epdb* self, epd_chunk_head* c_head)
{
  epdb_head_store db_head;
  uint32_t hash1;
  uint32_t hash2;
  size_t tsize;
  size_t nread;
  size_t i;
  epdb_status err;

  if (c_head->size < sizeof(db_head)) goto epdb_read_head_error;       /* format error */
  nread = epd_pread(self->db, c_head, 0, sizeof(db_head), &db_head);
  if ((size_t)-1 == nread) {
    err = EPDB_READ_ERROR;
    goto epdb_read_head_error;             /* IO error */
  }
  if (0 != strncmp(db_head.magic, EPDB_MAGIC, EPDB_MAGIC_SIZE))
    goto epdb_read_head_error;             /* format error */
  hash1 = epdb_hash(self, EPDB_MAGIC_SIZE, EPDB_MAGIC);
  hash2 = ntohl(db_head.hash);
  if (hash1 != hash2) goto epdb_read_head_error;       /* hash error */
  self->table_size = ntohl(db_head.table_size);
  tsize = self->table_size * sizeof(uint32_t);
  if (c_head->size != sizeof(db_head) + tsize)
    goto epdb_read_head_error;             /* table size error */
  self->table = epdb_malloc(tsize);
  if (NULL == self->table)
    goto epdb_read_head_error;             /* malloc error */
  err = epd_pread(self->db, c_head, sizeof(db_head), tsize, self->table);
  if ((size_t)-1 == nread) {
    err = EPDB_READ_ERROR;
    goto epdb_read_head_error;             /* table not found */
  }
  for (i = 0; i < self->table_size; i++) self->table[i] = ntohl(self->table[i]);
  return EPD_NO_ERROR;

epdb_read_head_error:
  return err;
}

static epdb_status epdb_create_head(epdb* self)
{
  uint32_t tsize;
  epdb_status err;
  epd_chunk_head head;
  epdb_head_store db_head;
  size_t i;

  tsize = sizeof(uint32_t) * self->table_size;
  err = epd_reserve(self->db, sizeof(db_head)+tsize, &head);
  if (EPD_NO_ERROR != err)
    goto epdb_write_head_error;
  strncpy(db_head.magic, EPDB_MAGIC, EPDB_MAGIC_SIZE);
  db_head.table_size = htonl(self->table_size);
  db_head.hash = htonl(epdb_hash(self, EPDB_MAGIC_SIZE, EPDB_MAGIC));
  err = epd_write(self->db, &head, sizeof(db_head), &db_head);
  if (EPD_NO_ERROR != err)
    goto epdb_write_head_error;
  self->table = malloc(tsize);
  if (NULL == self->table) {
    err = EPDB_MEMORY_FULL;
    goto epdb_write_head_error;
  }
  for (i = 0; i < self->table_size; i++) self->table[i] = EPD_EMPTY_INDEX;
  err = epdb_flush(self);
  if (EPDB_NO_ERROR != err) goto epdb_write_head_error;
  return EPDB_NO_ERROR;

epdb_write_head_error:
  return err;
}


epdb_status epdb_set(epdb* self, epdb_data* key, epdb_data* value)
{
  uint32_t hash = epdb_hash(self, key->size, key->data);
  uint32_t index = self->table[hash];
  epd_chunk_head a_head;
  epd_chunk_head k_head;
  epd_chunk_head v_head;
  epd_status err;
  size_t size;
  int i;
  uint32_t match_index;
  int new_key   = EPDB_FALSE;
  int new_table = EPDB_FALSE;
  epdb_table_array* kv_index;
  void* ptr = NULL;

  if (index == (uint32_t)-1) {
    /* new array */
    new_table = EPDB_TRUE;
    new_key = EPDB_TRUE;
    err = epd_reserve(self->db, sizeof(*kv_index)*20, &a_head);
    if (EPD_NO_ERROR != err) {
      goto epdb_set_error;
    }
    err = epd_reserve(self->db, key->size, &k_head);
    if (EPD_NO_ERROR != err) {
      goto epdb_set_error;
    }
    err = epd_reserve(self->db, value->size, &v_head);
    if (EPD_NO_ERROR != err) {
      goto epdb_set_error;
    }
    size = 0;
    kv_index = epdb_malloc(sizeof(*kv_index));
    if (!kv_index) {
      err = EPDB_MEMORY_FULL;
      goto epdb_set_error;
    }
  }
  else {
    /* array exists */
    err = epd_read_chunk_head(self->db, index, &a_head);
    if (EPD_NO_ERROR != err) goto epdb_set_error;
    size = a_head.size / sizeof(*kv_index);
    kv_index = epdb_malloc(a_head.size + sizeof(*kv_index));
    if (!kv_index) {
      err = EPDB_MEMORY_FULL;
      goto epdb_set_error;
    }
    match_index = -1;
    if (0 < size) {
      err = epd_read(self->db, &a_head, kv_index);
      if (err) goto epdb_set_error;
      for (i = 0; i < size; i++) {
        kv_index[i].ki = ntohl(kv_index[i].ki);
        kv_index[i].vi = ntohl(kv_index[i].vi);
      }
      err = epdb_search_table_array(self, key, kv_index, size, &match_index);
      if (err == EPDB_NOT_FOUND) {
        match_index = -1;
        new_key = EPDB_TRUE;
      }
      else if (err == EPDB_NO_ERROR) {
      }
      else {
        goto epdb_set_error;
      }
    }
    if (new_key) {
      err = epd_reserve(self->db, key->size, &k_head);
      if (EPD_NO_ERROR != err) goto epdb_set_error;
      err = epd_reserve(self->db, value->size, &v_head);
      if (EPD_NO_ERROR != err) goto epdb_set_error;
    }
    else {
      err = epd_read_chunk_head(self->db, kv_index[match_index].vi, &v_head);
      if (EPD_NO_ERROR != err) goto epdb_set_error;
    }
  }
  if (new_key) {
    err = epd_write(self->db, &k_head, key->size, key->data);
    if (EPD_NO_ERROR != err) goto epdb_set_error;
  }
  err = epd_write(self->db, &v_head, value->size, value->data);
  if (EPD_NO_ERROR != err) goto epdb_set_error;

  if (new_key) {
    kv_index[size].ki = k_head.index;
    kv_index[size].vi = v_head.index;
    for (i = 0; i < size+1; i++) {
      kv_index[i].ki = htonl(kv_index[i].ki);
      kv_index[i].vi = htonl(kv_index[i].vi);
    }
    err = epd_write(self->db, &a_head, (size+1)*sizeof(*kv_index), kv_index);
    for (i = 0; i < size+1; i++) {
      kv_index[i].ki = ntohl(kv_index[i].ki);
      kv_index[i].vi = ntohl(kv_index[i].vi);
    }
    if (err) goto epdb_set_error;
  }
  if (new_table) {
    self->table[hash] = a_head.index;
    if (err) goto epdb_set_error;
  }

  if (kv_index) free(kv_index);
  if (ptr) free(ptr);
  return EPDB_NO_ERROR;

epdb_set_error:
  if (ptr) free(ptr);
  if (kv_index) free(kv_index);
  return err;
}

epdb_status epdb_setsv(epdb* self, const char* key, epdb_data* value)
{
  epdb_status err;
  epdb_data* data;

  data = epdb_data_create(strlen(key)+1, (void *)key, EPDB_NO_COPY|EPDB_READ_ONLY);
  if (!data) {
    return EPDB_MEMORY_FULL;
  }
  err = epdb_set(self, data, value);
  epdb_data_release(data);
  return err;
}

epdb_status epdb_setss(epdb* self, const char* key, const char* value)
{
  epdb_status err;
  epdb_data* data_key;
  epdb_data* data_value;

  data_key = epdb_data_create(strlen(key)+1, (void *)key, EPDB_NO_COPY|EPDB_READ_ONLY);
  if (!data_key) {
    return EPDB_MEMORY_FULL;
  }
  data_value = epdb_data_create(strlen(value)+1, (void *)value, EPDB_NO_COPY|EPDB_READ_ONLY);
  if (!data_value) {
    epdb_data_release(data_key);
    return EPDB_MEMORY_FULL;
  }
  err = epdb_set(self, data_key, data_value);
  epdb_data_release(data_key);
  epdb_data_release(data_value);
  return err;
}

epdb_status epdb_get(epdb* self, epdb_data* key, epdb_data** value)
{
  epdb_data* data = NULL;
  epdb_table_array* kv_index = NULL;
  void* ptr = NULL;
  epdb_status err;
  epd_chunk_head a_head;
  epd_chunk_head v_head;
  size_t size;
  uint32_t match_index;

  err = epdb_get_table_array_head(self, key, &a_head);
  if (EPDB_NO_ERROR != err) goto epdb_get_error;
  size = a_head.size / sizeof(*kv_index);
  if (0 == size) {
    err = EPDB_NOT_FOUND;
    goto epdb_get_not_found;
  }
  kv_index = epdb_malloc(a_head.size);
  if (!kv_index) {
    err = EPDB_MEMORY_FULL;
    goto epdb_get_error;
  }
  err = epdb_get_table_array(self, &a_head, kv_index);
  if (EPDB_NO_ERROR != err) goto epdb_get_error;
  err = epdb_search_table_array(self, key, kv_index, size, &match_index);
  if (EPDB_NO_ERROR != err) goto epdb_get_error;
  err = epd_read_chunk_head(self->db, kv_index[match_index].vi, &v_head);
  ptr = epdb_malloc(v_head.size);
  if (!ptr) {
    err = EPDB_MEMORY_FULL;
    goto epdb_get_error;
  }
  err = epd_read(self->db, &v_head, ptr);
  if (EPDB_NO_ERROR != err) goto epdb_get_error;
  data = epdb_data_create(v_head.size, ptr, 0);
  if (!data) {
    err = EPDB_MEMORY_FULL;
    goto epdb_get_error;
  }
  free(kv_index);
  free(ptr);
  *value = data;

  return EPDB_NO_ERROR;

epdb_get_not_found:
epdb_get_error:
  if (data) epdb_data_release(data);
  if (kv_index) free(kv_index);
  if (ptr) free(ptr);
  return err;
}

epdb_status epdb_getsv(epdb* self, const char* key, epdb_data** value)
{
  epdb_status err;
  epdb_data data_key;

  err = epdb_data_init(&data_key, strlen(key)+1, (void *)key, (EPDB_NO_COPY|EPDB_READ_ONLY));
  if (err != EPDB_NO_ERROR) {
    return err;
  }
  err = epdb_get(self, &data_key, value);
  return err;
}

epdb_status epdb_getss(epdb* self, const char* key, char** value)
{
  epdb_status err;
  epdb_data* data_value = NULL;
  char* v = NULL;

  err = epdb_getsv(self, key, &data_value);
  if (err != EPDB_NO_ERROR) {
    return err;
  }
  v = epdb_malloc(data_value->size);
  if (!v) {
    return EPDB_MEMORY_FULL;
  }
  memcpy(v, epdb_data_get(data_value), epdb_data_size(data_value));
  *value = v;
  return EPDB_NO_ERROR;
}

static epdb_status epdb_get_table_array_head(epdb* self, epdb_data* key, epd_chunk_head* head)
{
  epdb_status err;
  epd_chunk_head a_head;
  uint32_t hash;
  int index;

  hash = epdb_hash(self, key->size, key->data);
  index = self->table[hash];
  if (index < 0) {
    return EPDB_NOT_FOUND;
  }
  err = epd_read_chunk_head(self->db, index, &a_head);
  if (err == EPD_CHUNK_NOT_FOUND) {
    return EPDB_NOT_FOUND;
  }
  else if (EPD_NO_ERROR != err) {
    return err;
  }
  *head = a_head;
  return EPDB_NO_ERROR;
}

static epdb_status epdb_get_table_array(epdb* self, epd_chunk_head* head, epdb_table_array* array)
{
  epdb_status err;
  size_t size;
  int i;
  err = epd_read(self->db, head, array);
  if (EPD_NO_ERROR != err) return err;
  size = head->size / sizeof(*array);
  for (i = 0; i < size; i++) {
    array[i].ki = ntohl(array[i].ki);
    array[i].vi = ntohl(array[i].vi);
  }
  return EPDB_NO_ERROR;
}

static epdb_status epdb_search_table_array(epdb* self, epdb_data* key, const epdb_table_array* array, uint32_t narray, uint32_t* index)
{
  uint32_t i;
  epd_chunk_head head;
  void* ptr;
  epdb_status err;

  for (i = 0; i < narray; i++) {
    err = epd_read_chunk_head(self->db, array[i].ki, &head);
    if (EPD_NO_ERROR != err) return err;
    if (head.size == key->size) {
      ptr = epdb_malloc(head.size);
      if (!ptr) return EPDB_MEMORY_FULL;
      err = epd_read(self->db, &head, ptr);
      if (EPD_NO_ERROR != err) {
        free(ptr);
        return err;
      }
      if (0 == memcmp(ptr, key->data, key->size)) {                   /* match! */
        free(ptr);
        *index = i;
        return EPDB_NO_ERROR;
      }
      free(ptr);
      ptr = NULL;
    }
  }
  return EPDB_NOT_FOUND;
}

epdb_status epdb_delete(epdb* self, epdb_data* key)
{
  size_t size;
  uint32_t index;
  epdb_status err;
  epd_chunk_head a_head;
  epd_chunk_head k_head;
  epd_chunk_head v_head;
  epdb_table_array* array;
  epdb_table_array match;

  err = epdb_get_table_array_head(self, key, &a_head);
  if (EPDB_NO_ERROR != err) return err;
  size = a_head.size / sizeof(*array);
  if (0 == size) return EPDB_NOT_FOUND;
  array = malloc(a_head.size);
  if (!array) return EPDB_MEMORY_FULL;
  err = epdb_get_table_array(self, &a_head, array);
  if (!err) {free(array); return err; }
  err = epdb_search_table_array(self, key, array, size, &index);
  if (err != EPD_NO_ERROR) {free(array); return err; }
  match = array[index];
  if (index != size-1) array[index] = array[size-1];
  size--;
  err = epd_write(self->db, &a_head, sizeof(*array)*size, array);
  if (EPD_NO_ERROR != err) {free(array); return err; }
  free(array);
  err = epd_read_chunk_head(self->db, match.ki, &k_head);
  if (EPD_NO_ERROR != err) return err;
  err = epd_unlink(self->db, &k_head);
  if (EPD_NO_ERROR != err) return err;
  err = epd_read_chunk_head(self->db, match.vi, &v_head);
  if (EPD_NO_ERROR != err) return err;
  err = epd_unlink(self->db, &v_head);
  if (EPD_NO_ERROR != err) return err;
  return EPDB_NO_ERROR;
}

epdb_status epdb_deletes(epdb* self, const char* key)
{
  epdb_data* data = epdb_data_create(strlen(key)+1, (void *)key,
                                     EPDB_NO_COPY|EPDB_READ_ONLY);
  epdb_status err;
  err = epdb_delete(self, data);
  epdb_data_release(data);
  return err;
}

epdb_status epdb_rewind(epdb* self)
{
  self->table_index = 0;
  self->array_index = -1;
  return EPDB_NO_ERROR;
}

epdb_pair* epdb_next(epdb* self)
{
  uint32_t ti;
  uint32_t ai;
  epd_chunk_head a_head;
  epd_chunk_head k_head;
  epd_chunk_head v_head;
  epd_status err;
  size_t a_size;
  int32_t array_index = self->table_index;
  int32_t table_index = self->array_index;
  epdb_data* key = NULL;
  epdb_data* value = NULL;
  epdb_pair* pair = NULL;
  epdb_table_array* t_array = NULL;
  void* ptr;

  for (ti = table_index; ti < self->table_size; ti++) {
    if ((uint32_t)-1 == self->table[ti]) {
      array_index = -1;
      continue;
    }
    err = epd_read_chunk_head(self->db, ti, &a_head);
    if (EPD_NO_ERROR != err) goto epdb_next_error;
    a_size = a_head.size / sizeof(epdb_table_array);
    ai = array_index+1;
    if (a_size <= ai) {
      array_index = -1;
      continue;
    }
    t_array = epdb_malloc(a_head.size);
    if (!t_array) {
      err = EPDB_MEMORY_FULL;
      goto epdb_next_error;
    }
    err = epd_read_chunk_head(self->db, t_array[ai].ki, &k_head);
    if (EPD_NO_ERROR != err) goto epdb_next_error;
    err = epd_read_chunk_head(self->db, t_array[ai].vi, &v_head);
    if (EPD_NO_ERROR != err) goto epdb_next_error;
    ptr = epdb_malloc(k_head.size);
    if (!ptr) {
      err = EPDB_MEMORY_FULL;
      goto epdb_next_error;
    }
    err = epd_read(self->db, &k_head, ptr);
    if (EPDB_NO_ERROR != err) goto epdb_next_error;
    key = epdb_data_create(k_head.size, ptr, 0);
    if (!key) goto epdb_next_error;
    free(ptr);
    ptr = NULL;
    err = epdb_get(self, key, &value);
    if (!value) goto epdb_next_error;
    pair = epdb_pair_create(key, value);
    if (!pair) {
      err = EPDB_MEMORY_FULL;
      goto epdb_next_error;
    }
    free(t_array);
    t_array = NULL;
    break;
  }
  if (self->table_size <= ti) {
    self->table_index = 0;
    self->array_index = -1;
    goto epdb_next_reach_end;
  }
  pair = epdb_pair_create(key, value);
  if (!pair) {
    err = EPDB_MEMORY_FULL;
    goto epdb_next_error;
  }
  epdb_data_release(key);
  epdb_data_release(value);
  return pair;

epdb_next_reach_end:
epdb_next_error:
  if (t_array) free(t_array);
  if (ptr) free(ptr);
  if (key) epdb_data_release(key);
  if (value) epdb_data_release(value);
  if (pair) epdb_pair_free(pair);
  return NULL;
}

uint32_t epdb_hash(epdb* self, size_t size, void* data)
{
  return self->hash(size, data) % self->table_size;
}


/*
 * epdb_data
 */

epdb_data* epdb_data_create(size_t size, void* data, epdb_data_attr attr)
{
  epdb_data* self;
  epdb_status err;
  self = epdb_malloc(sizeof(*self));
  if (!self) return NULL;
  memset(self, 0, sizeof(*self));
  err = epdb_data_init(self, size, data, (attr|EPDB_FREE_SELF));
  if (err) {
    free(self);
    return NULL;
  }
  return self;
}

epdb_status epdb_data_init(epdb_data* self, size_t size, void* data, epdb_data_attr attr)
{
  self->attr = attr;
  self->ref_count = 1;
  self->size = size;
  self->free_func = free;
  if (0 != (attr & EPDB_NO_COPY)) {
    self->data = data;
  }
  else {
    self->attr |= EPDB_FREE_DATA;
    self->data = epdb_malloc(size);
    if (!self->data) return EPDB_MEMORY_FULL;
    memcpy(self->data, data, size);
  }
  return EPDB_NO_ERROR;
}

void epdb_data_set_free_func(epdb_data* self, epdb_data_free_function func)
{
  self->free_func = (NULL != func) ? func : free;
}

epdb_data* epdb_data_copy(epdb_data* self)
{
  return epdb_data_create(epdb_data_size(self), epdb_data_get(self), 0);
}

void epdb_data_release(epdb_data* self)
{
  self->ref_count--;
  if (self->ref_count < 1) {
    if (0 != (self->attr&EPDB_FREE_DATA)) {
      self->free_func(self->data);
      self->data = NULL;
    }
    if (0 != (self->attr&EPDB_FREE_SELF)) {
      free(self);
    }
  }
}

epdb_data* epdb_data_retain(epdb_data* self)
{
  self->ref_count++;
  return self;
}

void* epdb_data_get(epdb_data* self)
{
  return self->data;
}

size_t epdb_data_size(epdb_data* self)
{
  return self->size;
}

/*
 * epdb_pair
 */
static epdb_pair* epdb_pair_create(epdb_data* key, epdb_data* value)
{
  epdb_pair* self;
  self = epdb_malloc(sizeof(*self));
  if (!self) return NULL;
  self->key = epdb_data_retain(key);
  self->value = epdb_data_retain(value);
  return self;
}

epdb_data* epdb_pair_key(epdb_pair* self)
{
  return self->key;
}

epdb_data* epdb_pair_value(epdb_pair* self)
{
  return self->value;
}

void epdb_pair_free(epdb_pair* self)
{
  epdb_data_release(self->key);
  epdb_data_release(self->value);
  free(self);
}

/**
 * This function use Bernstein hash algorithm.
 * @see: http://www.eternallyconfuzzled.com/tuts/algorithms/jsw_tut_hashing.aspx
 */
uint32_t epdb_default_hash_function(size_t size, const void* ptr)
{
  const unsigned char* p = ptr;
  unsigned h;
  size_t i;
  h = 0;
  for (i = 0; i < size; i++) {
    h = 33 * h + p[i];
  }
  return h;
}
