/**
 * @file      epdb.h
 * @brief     EPD based hash table.
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2010-2011 Japan Aerospace Exploration Agency
 *
 */

#ifndef _EPDB_H
#define _EPDB_H

#include <stdint.h>
#include "epd.h"

#define EPDB_FALSE  0
#define EPDB_TRUE   1

#define EPDB_TABLE_SMALL      8192
#define EPDB_TABLE_MEDIUM    65536
#define EPDB_TABLE_LARGE    524288

typedef int epdb_status;
#define EPDB_NO_ERROR        EPD_NO_ERROR
#define EPDB_MEMORY_FULL     EPD_MEMORY_FULL
#define EPDB_READ_ERROR      EPD_READ_ERROR
#define EPDB_NOT_FOUND       100
#define EPDB_INVALID_FORMAT  101

typedef int epdb_data_attr;
#define EPDB_NO_COPY       1
#define EPDB_READ_ONLY     2
#define EPDB_FREE_DATA     4
#define EPDB_FREE_SELF     8

typedef uint32_t (* epdb_hash_function)(size_t, const void *);
typedef void (* epdb_data_free_function)(void *);

typedef struct _epdb_config {
  size_t table_size;
  epdb_hash_function hash;
  epd_opmode open_mode;
  uint32_t block_size;
  epd_debug_level debug_level;
  epd_bool write_sync;
} epdb_config;

typedef struct _epdb {
  epd* db;
  uint32_t* table;
  uint32_t table_size;
  epdb_hash_function hash;
  ssize_t table_index;
  ssize_t array_index;
} epdb;

typedef struct _epdb_data {
  size_t size;
  void* data;
  epdb_data_free_function free_func;
  int ref_count;
  epdb_data_attr attr;
} epdb_data;

typedef struct _epdb_pair {
  epdb_data* key;
  epdb_data* value;
} epdb_pair;

__BEGIN_DECLS
const epdb_config* epdb_config_default(void);
void epdb_config_set_default(epdb_config* conf);

epdb* epdb_open(const char* path, const epdb_config* conf);
epdb_status epdb_close(epdb* self);
epdb_status epdb_flush(epdb* self);
epdb_status epdb_set(epdb* self, epdb_data* key, epdb_data* value);
epdb_status epdb_setsv(epdb* self, const char* key, epdb_data* value);
epdb_status epdb_setss(epdb* self, const char* key, const char* value);
epdb_status epdb_get(epdb* self, epdb_data* key, epdb_data** value);
epdb_status epdb_getsv(epdb* self, const char* key, epdb_data** value);
epdb_status epdb_getss(epdb* self, const char* key, char** value);
epdb_status epdb_delete(epdb* self, epdb_data* key);
epdb_status epdb_deletes(epdb* self, const char* key);
epdb_status epdb_rewind(epdb* self);
epdb_pair* epdb_next(epdb* self);
uint32_t epdb_hash(epdb* self, size_t size, void* data);

epdb_data* epdb_data_create(size_t size, void* data, epdb_data_attr attr);
epdb_status epdb_data_init(epdb_data* self, size_t size, void* data, epdb_data_attr attr);
void epdb_data_set_free_func(epdb_data* self, epdb_data_free_function func);
epdb_data* epdb_data_copy(epdb_data* self);
void epdb_data_release(epdb_data* self);
/**
 * @result "self" itself.
 */
epdb_data* epdb_data_retain(epdb_data* self);
void* epdb_data_get(epdb_data* self);
size_t epdb_data_size(epdb_data* self);

epdb_data* epdb_pair_key(epdb_pair* self);
epdb_data* epdb_pair_value(epdb_pair* self);
void epdb_pair_free(epdb_pair* self);

uint32_t epdb_default_hash_function(size_t size, const void* ptr);
__END_DECLS

#endif /* _EPDB_H */
