/**
 * @file      float_util.c
 * @brief     Floating point utilities
 * @date      2012/08/09
 *
 * @copyright
 * Copyright 2005, Google Inc. All rights reserved.
 * @sa http://code.google.com/p/spectral-methods/source/browse/FloatingPoint.h?spec=svn9&r=9
 * @sa http://www.cygnus-software.com/papers/comparingfloats/comparingfloats.htm
 *
 * @par       History
 * - 2012/08/09 Y. Yamamoto (Japan Aerospace Exploration Agency)
 *   -# Porting from C++ to C.
 */

// Copyright 2005, Google Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Authors: Zhanyong Wan, Sean Mcafee
//
// Taken from The Google C++ Testing Framework (Google Test).
// Modified for this discussion by Fred Richards.

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdlib.h>
#include <assert.h>
#include <stdint.h>
#include <math.h>
#include "float_util.h"

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

typedef union {
  float f;
  uint32_t bits_;
} Float_Bit;

typedef union {
  double d;
  uint64_t bits_;
} Double_Bit;


static int isFloatNan(float a) {
  uint32_t kFracBitMask = UINT32_C(0x007fffff);       // 0xffffffff >> (8+1)
  uint32_t kExpBitMask  = UINT32_C(0x7f800000);       // ~(kSignBitMask | kFracBitMask)
  Float_Bit val;

  val.f = a;
  return ((kExpBitMask & val.bits_) == kExpBitMask) &&
         ((kFracBitMask & val.bits_) != 0);
}

static int isDoubleNan(double a) {
  uint64_t kFracBitMask = UINT64_C(0x000fffffffffffff);       // 0xffffffffffffffff >> (11+1)
  uint64_t kExpBitMask  = UINT64_C(0x7ff0000000000000);       // ~(kSignBitMask | kFracBitMask)
  Double_Bit val;

  val.d = a;
  return ((kExpBitMask & val.bits_) == kExpBitMask) &&
         ((kFracBitMask & val.bits_) != 0);
}

static uint32_t float_SignAndMagnitudeToBiased(uint32_t sam) {
  uint32_t kSignBitMask = UINT32_C(0x80000000);       // 1 << (32-1)
  if (kSignBitMask & sam) {
    return ~sam + 1;      // two's complement
  } else {
    return kSignBitMask | sam;      // * 2
  }
}

static uint64_t double_SignAndMagnitudeToBiased(uint64_t sam) {
  uint64_t kSignBitMask=UINT64_C(0x8000000000000000);       // 1 << (64-1)
  if (kSignBitMask & sam) {
    return ~sam + 1;      // two's complement
  } else {
    return kSignBitMask | sam;      // * 2
  }
}

uint32_t float_ULP_diff(uint32_t sam1, uint32_t sam2) {
  uint32_t biased1 = float_SignAndMagnitudeToBiased(sam1);
  uint32_t biased2 = float_SignAndMagnitudeToBiased(sam2);
  return (biased1 >= biased2) ? (biased1 - biased2) : (biased2 - biased1);
}

uint64_t double_ULP_diff(uint64_t sam1, uint64_t sam2) {
  uint64_t biased1 = double_SignAndMagnitudeToBiased(sam1);
  uint64_t biased2 = double_SignAndMagnitudeToBiased(sam2);
  return (biased1 >= biased2) ? (biased1 - biased2) : (biased2 - biased1);
}

int isFloatAlmostEquals(float A, float B) {
  size_t kMaxUlps = 4;

  assert(sizeof(float) == sizeof(int32_t));

  if (isnan(A) || isnan(B)) {
    return FALSE;
  }

  Float_Bit fA, fB;
  fA.f = A;
  fB.f = B;

  return float_ULP_diff(fA.bits_, fB.bits_) <= kMaxUlps;
}

int isDoubleAlmostEquals(double A, double B) {
  size_t kMaxUlps = 4;

  assert(sizeof(double) == sizeof(int64_t));

  if (isnan(A) || isnan(B)) {
    return FALSE;
  }

  Double_Bit dA, dB;
  dA.d = A;
  dB.d = B;

  return double_ULP_diff(dA.bits_, dB.bits_) <= kMaxUlps;
}
