/**
 * @file      float_util.h
 * @brief     Header file of floating point utilities
 * @date      2012/08/09
 *
 * @copyright
 * Copyright 2005, Google Inc. All rights reserved.
 * @sa http://code.google.com/p/spectral-methods/source/browse/FloatingPoint.h?spec=svn9&r=9
 * @sa http://www.cygnus-software.com/papers/comparingfloats/comparingfloats.htm
 *
 * @par       History
 * - 2012/08/09 Y. Yamamoto (Japan Aerospace Exploration Agency)
 *   -# Porting from C++ to C.
 */

// Copyright 2005, Google Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Authors: Zhanyong Wan, Sean Mcafee
//
// Taken from The Google C++ Testing Framework (Google Test).
// Modified for this discussion by Fred Richards.

#ifndef FLOAT_UTIL_H_
#define FLOAT_UTIL_H_

__BEGIN_DECLS

/**
 * @brief Check if the two single precision floating-point numbers are close to each other
 *
 * @param[in]   A       single precision floating-point number (32 bits)
 * @param[in]   B       single precision floating-point number (32 bits)
 * @return      isFloatAlmostEquals() returns TRUE if the two single precision
 *              floating-point numbers are close to each other. Otherwise,
 *              returns FALSE.
 */
int isFloatAlmostEquals(float A, float B);

/**
 * @brief Check if the two double precision floating-point numbers are close to each other
 *
 * @param[in]   A       double precision floating-point number (64 bits)
 * @param[in]   B       double precision floating-point number (64 bits)
 * @return      isDoubleAlmostEquals() returns TRUE if the two double precision
 *              floating-point numbers are close to each other. Otherwise,
 *              returns FALSE.
 */
int isDoubleAlmostEquals(double A, double B);

/**
 * @brief Return the differences between two floating-point numbers using integers
 *
 * @param[in]   sam1    integer corresponding to IEEE 754 floating-point number (32 bits)
 * @param[in]   sam2    integer corresponding to IEEE 754 floating-point number (32 bits)
 * @return      float_ULP_diff() returns the differences between sam1 and sam2.
 */
uint32_t float_ULP_diff(uint32_t sam1, uint32_t sam2);

/**
 * @brief Return the differences between two floating-point numbers using integers
 *
 * @param[in]   sam1    integer corresponding to IEEE 754 floating-point number (64 bits)
 * @param[in]   sam2    integer corresponding to IEEE 754 floating-point number (64 bits)
 * @return      double_ULP_diff() returns the differences between sam1 and sam2.
 */
uint64_t double_ULP_diff(uint64_t sam1, uint64_t sam2);

__END_DECLS

#endif /* FLOAT_UTIL_H_ */
