/**
 * @file      flow_common.c
 * @brief     flow_se and flow_ig common codes
 * @date      2017-05-21
 *
 * @copyright
 * Copyright 2010-2011 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <libgen.h>
#include <errno.h>
#include <math.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <regex.h>
#include "sv_doc.h"
#include "flow_common.h"
#include "float_util.h"

void vec_middle(const SpiceDouble* vec1, SpiceDouble* vec2, SpiceDouble* dst)
{
  SpiceDouble vec3[3];
  vadd_c(vec1, vec2, vec3);
  vscl_c(0.5, vec3, dst);
}

void vec_padist(const SpiceDouble* vec1, const SpiceDouble* vec2, SpiceDouble* _pa, SpiceDouble* _dist)
{
  double pa, dist;
  double sda, sdd;
  SpiceDouble range;
  SpiceDouble ad1[2];
  SpiceDouble ad2[2];
  SpiceDouble pi2 = twopi_c();

  recrad_c(vec1, &range, &ad1[0], &ad1[1]);
  recrad_c(vec2, &range, &ad2[0], &ad2[1]);
  pa = atan2(cos(ad2[1]) * sin(ad2[0] - ad1[0]),
             cos(ad1[1]) * sin(ad2[1]) -
             sin(ad1[1]) * cos(ad2[1]) * cos(ad2[0] - ad1[0]));
  pa = (pa > 0.0) ? pa : pi2 + pa;
  sda = sin((ad1[0] - ad2[0]) / 2);
  sdd = sin((ad1[1] - ad2[1]) / 2);
  dist = sdd * sdd + cos(ad1[1]) * cos(ad2[1]) * sda * sda;
  dist = asin(sqrt(dist))*2;
  if (NULL != _pa) *_pa = pa;
  if (NULL != _dist) *_dist = dist;
}

void vec_plotxy(const SpiceDouble* bv, const SpiceDouble* ltv, const SpiceDouble* tv, double r, double* _x, double* _y)
{
  double ltpa, ltdist;
  double tpa, tdist;
  double x;
  double y;

  vec_padist(bv, ltv, &ltpa, &ltdist);
  vec_padist(bv, tv, &tpa, &tdist);
  x = (tdist / ltdist) * r * sin(tpa-ltpa);
  y = -((tdist / ltdist) * r * cos(tpa-ltpa));
  *_x = x;
  *_y = y;
}

double* bounds_top_left(int nbounds, double* bounds)
{
  int i;
  double* bound_tl = NULL;
  for (i = 0; i < nbounds; i++) {
    if (!bound_tl || (bounds[i*3] <= bound_tl[0] &&
                      bounds[i*3+1] >= bound_tl[1])) {
      bound_tl = &bounds[i*3];
    }
  }
  return bound_tl;
}

double* bounds_bottom_right(int nbounds, double* bounds)
{
  int i;
  double* bound_br;
  for (i = 0; i < nbounds; i++) {
    if (!bound_br || (bound_br[0] <= bounds[i*3] &&
                      bound_br[1] >= bounds[i*3+1])) {
      bound_br = &bounds[i*3];
    }
  }
  return bound_br;
}

void mpc_get_bounds_rect(const char* shape, const double* boresight, int nbounds, const double* bounds, double* rect)
{
  const double* p;
  double* v;
  double top;
  double left;
  double bottom;
  double right;
  int i;
  if (0 == strcmp(shape, "RECTANGLE") || 0 == strcmp(shape, "POLYGON")) {
    right = left = bounds[0];
    bottom = top = bounds[1];
    for (i = 1; i < nbounds; i++) {
      p = bounds + i*3;
      left   = min(p[0], left);
      top    = min(p[1], top);
      right  = max(p[0], right);
      bottom = max(p[1], bottom);
    }
  }
  else if (0 == strcmp(shape, "CIRCLE")) {
    double dx = bounds[0] - boresight[0];
    double dy = bounds[1] - boresight[1];
    double r = sqrt(dx*dx+dy*dy);
    p = boresight;
    left   = p[0] - r;
    top    = p[1] - r;
    right  = p[0] + r;
    bottom = p[1] + r;
  }
  else if (0 == strcmp(shape, "ELLIPSE")) {
    double dx;
    double dy;
    double a,b;
    double t;
    double sint, cost;
    int i;
    double rad;
    double x, y, tx, ty, cx, cy;
    cx = boresight[0];
    cy = boresight[1];
    left = right = cx;
    top = bottom = cy;
    dx = bounds[0] - cx;
    dy = bounds[1] - cy;
    a = sqrt(dx*dx+dy*dy);
    t = atan2(dy,dx);
    sint = sin(t);
    cost = cos(t);
    dx = bounds[3] - cx;
    dy = bounds[4] - cy;
    b = sqrt(dx*dx+dy*dy);
    for (i = 0; i < 90; i++) {
      rad = deg2rad((double)i);
      tx = a * cos(rad);
      ty = b * sin(rad);
      x = tx*cost - ty*sint + cx;
      y = tx*sint + ty*cost + cy;
      right  = max(x, right);
      bottom = max(y, bottom);
    }
    left = right - (right - cx) * 2;
    top  = bottom - (bottom - cy) * 2;
  }
  else {
    fprintf(stderr, "%s: unknown instrument shape %s.\n", app_name, shape);
    abort();
  }

  v = rect;
  v[0] = left;
  v[1] = top;
  v[2] = bounds[2];
  v[3] = right;
  v[4] = bottom;
  v[5] = bounds[2];
}

void mpc_get_projection_rect(enum FLOWProjectionMode mode, double w, double h, double left, double top, double right, double bottom, double* rect)
{
  double pw = right-left;
  double ph = bottom-top;
  double vasp = w/h;
  double pasp = pw/ph;
  double cx, cy, nw_2, nh_2;
  switch (mode) {
  case FLOWProjectionModeFill:
    rect[0] = left;
    rect[1] = top;
    rect[2] = right;
    rect[3] = bottom;
    break;
  case FLOWProjectionModeScaleAspectFit:
    if (vasp < pasp) {
      cy = top + ph / 2.0;
      nh_2 = pw * h / w / 2.0;
      rect[0] = left;
      rect[1] = cy - nh_2;
      rect[2] = right;
      rect[3] = cy + nh_2;
    }
    else if (vasp > pasp) {
      cx = left + pw / 2.0;
      nw_2 = ph * vasp / 2.0;
      rect[0] = cx - nw_2;
      rect[1] = top;
      rect[2] = cx + nw_2;
      rect[3] = bottom;
    }
    else {
      rect[0] = left;
      rect[1] = top;
      rect[2] = right;
      rect[3] = bottom;
    }
    break;
  case FLOWProjectionModeScaleAspectFill:
    if (vasp < pasp) {
      cx = left + pw / 2.0;
      nw_2 = ph * vasp / 2.0;
      rect[0] = cx - nw_2;
      rect[1] = top;
      rect[2] = cx + nw_2;
      rect[3] = bottom;
    }
    else if (vasp > pasp) {
      cy = top + ph / 2.0;
      nh_2 = pw * h / w / 2.0;
      rect[0] = left;
      rect[1] = cy - nh_2;
      rect[2] = right;
      rect[3] = cy + nh_2;
    }
    else {
      rect[0] = left;
      rect[1] = top;
      rect[2] = right;
      rect[3] = bottom;
    }
    break;
  }
}

int get_object_type(int object_id)
{
  //
  // http://naif.jpl.nasa.gov/pub/naif/toolkit_docs/C/req/naif_ids.html
  //
  if (object_id < -100000) {
    // artificial satellite (earth orbitting spacecraft)
    return OBJECT_TYPE_ASAT;
  }
  else if (object_id < 0) {
    // spacecraft or instrument
    return OBJECT_TYPE_SPACECRAFT;
  }
  else if (object_id < 10) {
    // Required size for System barycenter
    return OBJECT_TYPE_BARYCENTER;
  }
  else if (object_id == 10) {
    // sun
    return OBJECT_TYPE_SUN;
  }
  else if (object_id < 100) {
    // invalid (11...99)
    return OBJECT_TYPE_INVALID;
  }
  else if ((object_id % 100) == 99) {
    // planet (199, 299, ...)
    return OBJECT_TYPE_PLANET;
  }
  else if ((object_id % 100) == 0) {
    // invalid (100, 200, ...)
    return OBJECT_TYPE_INVALID;
  }
  else if (object_id < 100000) {
    // satellite (PXX, PNXXX, 1<=P<=9, N=0,5)
    return OBJECT_TYPE_SATELLITE;
  }
  else {
    // comets, asteroids or other objects
    return OBJECT_TYPE_SMALL_BODY;
  }
}

enum FLOWImageFormat predict_image_format_from_filename(const char* path)
{
  regex_t reg_png;
  regex_t reg_fts;
  regex_t reg_jpg;
  regmatch_t m[1];
  int err;
  enum FLOWImageFormat format = FLOWImageFormatAuto;

  if ((err = regcomp(&reg_png, "\\.png$", REG_EXTENDED|REG_ICASE)) != 0) {
    perror("error in function make_image > regcomp.");
    return FLOWImageFormatError;
  }
  if ((err = regcomp(&reg_jpg, "\\.jpe?g$", REG_EXTENDED|REG_ICASE)) != 0) {
    perror("error in function make_image > regcomp.");
    regfree(&reg_png);
    return FLOWImageFormatError;
  }
  if ((err = regcomp(&reg_fts, "\\.fi?ts$", REG_EXTENDED|REG_ICASE)) != 0) {
    perror("error in function make_image > regcomp.");
    regfree(&reg_png);
    regfree(&reg_jpg);
    return FLOWImageFormatError;
  }
  if (0 == (err = regexec(&reg_png, path, 1, m, 0))) {
    format = FLOWImageFormatPNG;
  }
  else if (0 == (err = regexec(&reg_fts, path, 1, m, 0))) {
    format = FLOWImageFormatFITS;
  }
  else if (0 == (err = regexec(&reg_jpg, path, 1, m, 0))) {
    format = FLOWImageFormatJPEG;
  }
  regfree(&reg_png);
  regfree(&reg_fts);
  regfree(&reg_jpg);

  return format;
}

double mpc_degnorm(double deg, double max)
{
  deg = (0 < deg) ? fmod(deg, 360.0) : 360.0 - fmod(-deg, 360.0);
  if (max < deg) deg -= 360.0;
  return deg;
}

int mpc_mkdir_p(const char* path, int mode)
{
  struct stat st;
  char* s;
  char* s2;
  char* parent;
  int err;

  s = strdup(path);
  s2 = dirname(s);
  parent = strdup(s2);
  free(s);
  if (0 != (err = stat(parent, &st))) {
    if (errno == ENOENT) {
      mpc_mkdir_p(parent, mode);
    }
    else {
      free(parent);
      return err;
    }
  }
  free(parent);
  if (0 != (err = mkdir(path, mode))) {
    return err;
  }
  return 0;
}

char* mpc_get_config_path(const char* path, char* buf, size_t nbuf, size_t* nlen)
{
  char* home;
  int len = 0;

  home = getenv("HOME");
  if (!home) {
    *nlen = len;
    return NULL;
  }
  len = strlen(home) + strlen(FLOW_CONFIG_DIR) + strlen(path) + 3;
  *nlen = len;
  if (!buf) {
    buf = malloc(len);
  }
  else if (nbuf < len) {
    return NULL;
  }
  strcpy(buf, home);
  strcat(buf, "/");
  strcat(buf, FLOW_CONFIG_DIR);
  strcat(buf, "/");
  strcat(buf, path);
  return buf;
}

int mpc_create_config_dir(void)
{
  char* home;
  char* path;
  int len;
  int err;
  struct stat st;

  home = getenv("HOME");
  if (!home) {
    return 1;
  }
  len = strlen(home) + strlen(FLOW_CONFIG_DIR) + 2;
  path = malloc(len);
  strcpy(path, home);
  strcat(path, "/");
  strcat(path, FLOW_CONFIG_DIR);
  if (0 == stat(path, &st)) {
    if (S_ISDIR(st.st_mode)) {
      free(path);
      return 0;
    }
  }
  if (0 != (err = mpc_mkdir_p(path, 0777))) {
    free(path);
    return err;
  }
  free(path);
  return 0;
}

void get_appname(char* arg0, char* name, char* path)
{
  char* p;

  if ((p = strrchr(arg0, '/')) == NULL) {
    strcpy(path, "");
    strcpy(name, arg0);
  } else {
    int path_len = (p - arg0) + 1;
    strncpy(path, arg0, path_len);
    path[path_len] = '\0';
    strcpy(name, arg0 + path_len);
  }
}

void mpc_get_boresight_rotation_matrix(SpiceDouble boresight[3], SpiceDouble r[3][3]) {
  SpiceDouble z_axis[] = {0.0, 0.0, 1.0};
  SpiceDouble theta, a[3], q[4], c, s;

  if( (boresight[0] == 0 &&
       boresight[1] == 0 &&
       boresight[2] > 0)) {
    ident_c(r);
    return;
  }

  theta = vsep_c(boresight, z_axis);
  vcrss_c(boresight, z_axis, a);
  if (isDoubleAlmostEquals(vnorm_c(a), 0.0)) {
    r[0][0] =-1.0; r[0][1] =  0.0; r[0][2] =  0.0;
    r[1][0] = 0.0; r[1][1] =  1.0; r[1][2] =  0.0;
    r[2][0] = 0.0; r[2][1] =  0.0; r[2][2] = -1.0;
  } else {
    c = cos(theta/2.0);
    s = sin(theta/2.0);
    q[0] = c;
    q[1] = s * a[0];
    q[2] = s * a[1];
    q[3] = s * a[2];
    q2m_c(q,r);
  }
}

void mpc_rotate_frame(SV_FRAME* frame) {
  int i;
  SV_VIEW* view = frame->view;
  SpiceDouble r[3][3];

  mpc_get_boresight_rotation_matrix(view->boresight, r);
  mxv_c(r, view->boresight, view->boresight);
  for(i=0; i<view->nbounds; ++i) {
    SpiceDouble* b = &view->bounds[i*3];
    mxv_c(r, b, b);
  }
}
