/**
 * @file      flow_common.h
 * @brief     flow_se and flow_ig common codes
 * @date      2017-05-21
 *
 * @copyright
 * Copyright 2011 Japan Aerospace Exploration Agency
 *
 */

#ifndef _FLOW_COMMON_H
#define _FLOW_COMMON_H

#include <SpiceUsr.h>

//
// defined in main
//
extern char app_path[], app_name[];

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

#define rad2deg(x)    ((x)*180.0/M_PI)
#define deg2rad(x)    ((x)*M_PI/180.0)
#define max(a,b)      (((a) < (b)) ? (b) : (a))
#define min(a,b)      (((a) < (b)) ? (a) : (b))

//
// object IDs defined in:
//   http://naif.jpl.nasa.gov/pub/naif/toolkit_docs/C/req/naif_ids.html
//
#define OBJECT_ID_SUN           10
#define OBJECT_ID_MOON          301
#define OBJECT_ID_MERCURY       199
#define OBJECT_ID_VENUS         299
#define OBJECT_ID_EARTH         399
#define OBJECT_ID_MARS          499
#define OBJECT_ID_JUPITER       599
#define OBJECT_ID_SATURN        699
#define OBJECT_ID_URANUS        799
#define OBJECT_ID_NEPTUNE       899
#define OBJECT_ID_PLUTO         999

//
// object types
//
#define OBJECT_TYPE_INVALID     0
#define OBJECT_TYPE_ASAT        1 // artificial sat. (earth orbitting spacecraft)
#define OBJECT_TYPE_SPACECRAFT  2 // other spacecraft or it's instrument
#define OBJECT_TYPE_BARYCENTER  3 // system barycenter
#define OBJECT_TYPE_SUN         4
#define OBJECT_TYPE_PLANET      5 // including pluto
#define OBJECT_TYPE_SATELLITE   6
#define OBJECT_TYPE_SMALL_BODY  7 // comet, asteroid or other object

enum FLOWImageFormat {
  FLOWImageFormatAuto,
  FLOWImageFormatFITS,
  FLOWImageFormatJPEG,
  FLOWImageFormatPNG,
  FLOWImageFormatUnkown,
  FLOWImageFormatError
};

enum FLOWProjectionMode {
  FLOWProjectionModeFill = 1,       /** Scale so that the vertical and horizontal fit on the screen */
  FLOWProjectionModeScaleAspectFit, /** Scale so that the whole will fit while keeping the camera ratio */
  FLOWProjectionModeScaleAspectFill,/** Scale to be displayed on the whole screen while maintaining the ratio of the camera */
};

#define FLOW_CONFIG_DIR                 ".flow"

#ifdef ENABLE_WMS
#define FLOW_WMS_CACHE_FILE             "/wms_cache.edb"
#endif

__BEGIN_DECLS
double vec2d_cross_product(const double* v1, const double* v2);
double vec2d_dot_product(const double* v1, const double* v2);

/**
 * @brief Create a vector indicating the middle of the two vectors.
 * @param[in] vec1   input vector1
 * @param[in] vec2   input vector2
 * @param[out] dst   the middle of the two vectors
 */
void vec_middle(const SpiceDouble* vec1, SpiceDouble* vec2, SpiceDouble* dst);

/**
 * Calculate the position angle and angular distance of two vectors.
 * @param[in] vec1   input vector1
 * @param[in] vec2   input vector2
 * @param[out] _pa   position angle [rad]
 * @param[out] _dist  angular distance [rad]
 */
void vec_padist(const SpiceDouble* vec1, const SpiceDouble* vec2, SpiceDouble* _pa, SpiceDouble* _dist);

/**
 * @brief Calculate the projected coordinates of the target coordinates
 * by passing the center coordinates and the upper left coordinates.
 *
 * @param[in]  bv   boresight vector
 * @param[in]  ltv  left upper reference vector
 * @param[in]  tv   target vector to an astronomical object
 * @param[in]  r    drawing radius [px]
 * @param[out] _x   X coordinate
 * @param[out] _y   Y coordinate
 */
void vec_plotxy(const SpiceDouble* bv, const SpiceDouble* ltv, const SpiceDouble* tv, double r, double* _x, double* _y);

/**
 * @brief Returns the one in the upper left from one of the camera boundary coordinates.
 * @param[in] nbounds  Number of points in the boundary
 * @param[in] bounds   array of boundary points
 *                     1D-dimensional array and 3 elements x number of vertices.
 * @return Pointer to top top-left vertex coordinates.
 */
double* bounds_top_left(int nbounds, double* bounds);

/**
 * @brief Returns the one in the lower right from a list of camera boundary coordinates.
 * @param[in] nbounds  Number of points in the boundary
 * @param[in] bounds   array of boundary points
 *                     1D-dimensional array and 3 elements x number of vertices.
 * @return Pointer to top bottom-right vertex coordinates.
 */
double* bounds_bottom_right(int nbounds, double* bounds);

/**
 * @brief Return the type of object
 *
 * @param[in]   object_id  Object ID
 * @return      get_object_type() returns the object ID (OBJECT_TYPE_XXX).
 *              If an error occurs, returns -1.
 */
int get_object_type(int object_id);

/**
 * @brief Returns top / left, bottom / right of rect circumscribing shape, bounds.
 * @param[in] shape      shape {RECT | CIRCLE | ELLIPSE | POLYGON}
 * @param[in] boresight  center vector (XYZ)
 * @param[in] nbounds    Number of vertexes in a boundary
 * @param[in] bounds     vertex data (XYZ x nbounds)。
 * @param[out] rect      vertex data (XYZ x 2)
 */
void mpc_get_bounds_rect(const char* shape, const double* boresight, int nbounds, const double* bounds, double* rect);

/**
 * @brief Convert bounds to the ratio which is included in view (w, h)
 *
 * @param[in]   mode    projection mode
 * @param[in]   w       view width.
 * @param[in]   h       view height.
 * @param[in]   left    top of bounds.
 * @param[in]   top     top of bounds.
 * @param[in]   right   top of bounds.
 * @param[in]   bottom  top of bounds.
 * @param[out]  rect    result of the calculation {left, top, right, bottom}
 */
void mpc_get_projection_rect(enum FLOWProjectionMode mode, double w, double h, double left, double top, double right, double bottom, double* rect);
void mpc_get_boresight_rotation_matrix(SpiceDouble boresight[3], SpiceDouble r[3][3]);

/**
 * Rotate frame following boresight vector
 * @param frame     [in] target frame
 */
void mpc_rotate_frame(SV_FRAME* frame);

/**
 * @brief Return the absolute path under the config directory
 * At least buf should be allocated at least nbuf bytes or NULL.
 * If it is NULL, the result of internally malloc is returned.
 * If the created path exceeds nbuf, NULL is returned.
 *
 * @param[in]  path  path under configuration directory
 * @param[in]  buf   buffer to store absolute path
 * @param[in]  nbuf  size of buffer
 * @param[out] nlen  actual length
 * @return  mpc_get_config_path() returns the absolute path if buf is NULL or
 * memory is allocated enough, NULL otherwise.
 */
char* mpc_get_config_path(const char* path, char* buf, size_t nbuf, size_t* nlen);

/**
 * @brief Create configuration directory recursively.
 * @return return 0 if successful, error code otherwise.
 */
int mpc_create_config_dir(void);

/**
 * @brief If the parent directory does not exist, mkdir creates a directory backwards.
 * @param[in] path directory path to create
 * @param[in] mode creation mode
 * @return mpc_mkdir_p() returns 0 if successful.
 * Otherwise, return value of the last command.
 */
int mpc_mkdir_p(const char* path, int mode);

/**
 * @brief Normalize the angle to max - 360.0 <= x <= max.
 * @param[in] deg  angle (degree)
 * @param[in] max  maximum angle (degree)
 * @return mpc_degnorm() returns normalized angle.
 */
double mpc_degnorm(double deg, double max);

/**
 * @brief Determine the image format from the extension.
 * @param[in] path filepath
 * @return predict_image_format_from_filename() returns predicted format listed
 * in the enumaration.
 */
enum FLOWImageFormat predict_image_format_from_filename(const char* path);

/**
 * Get execution file name and path name
 * @param[in] arg0  program name
 * @param[out] name
 * @param[out] path
 */
void get_appname(char* arg0, char* name, char* path);
__END_DECLS

#endif // _FLOW_COMMON_H
