/**
 * @file      flow_cov.c
 * @brief     coverage utility
 * @date      2015-05-14
 *
 * @copyright
 * Copyright 2010-2015 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <unistd.h>
#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>
#include <SpiceUsr.h>
#include "sv_doc.h"
#include "flow_common.h"
#include "flowse_lua_bind.h"
#include "timepair.h"

#define DETAIL_TRUE  1
#define DETAIL_FALSE 0

#define MAXOBJ 1000
#define MAXIV  5000
#define WINSIZ ( 2 * MAXIV )
#define TIMLEN  51
#define TIMFMT "YYYY-MM-DDTHR:MN:SC.###Z ::UTC"

#define FILLEN 128
#define TYPLEN 32
#define SRCLEN 128

#define CHECK_OFFSET_CK  (1.0e-6)
#define CHECK_OFFSET_SPK (1.0)

char app_path[256], app_name[256];
static lua_State* Lua = NULL;

typedef void (* SPICE_FUNC)(ConstSpiceChar* filename);
typedef void (* FUNCPTR)(const char* filename, SpiceChar* frame, time_pair_list* list);

#define MAXBND 10
#define WDSIZE 32
SpiceChar* get_inst_frame_name(int instid, SpiceChar* frame) {
  SpiceChar shape[WDSIZE];
  SpiceDouble bounds[MAXBND][3];
  SpiceDouble bsight[3];
  SpiceInt n;
  getfov_c(instid, MAXBND, WDSIZE, WDSIZE, shape, frame, bsight, &n, bounds);
  return frame;
}

void add_spk_coverage(const char* file, SpiceChar* obs, time_pair_list* list) {
  SpiceInt i, j;
  SpiceInt niv;
  SpiceInt obj;
  SpiceDouble b, e;
  SPICEINT_CELL(ids, MAXOBJ);
  SPICEDOUBLE_CELL(cover, WINSIZ);
  SpiceDouble state[6], lt;
  SpiceChar comment[SIZE_COMMENT];

  spkobj_c(file, &ids);

  for(i=0; i<card_c(&ids); ++i) {
    obj  =  SPICE_CELL_ELEM_I(&ids, i);
    scard_c(0, &cover);
    spkcov_c(file, obj, &cover);
    niv = wncard_c(&cover);
    if (niv > 0) {
      for(j=0; j<niv; ++j) {
        wnfetd_c(&cover, j, &b, &e);

        spkezr_c("EARTH", b+CHECK_OFFSET_SPK, "J2000", "LT+S", obs, state, &lt);
        if (failed_c()) {
          reset_c();
          continue;
        }

        spkezr_c("EARTH", e-CHECK_OFFSET_SPK, "J2000", "LT+S", obs, state, &lt);
        if (failed_c()) {
          reset_c();
          continue;
        }

        sprintf(comment, "%s", file);
        add_time_pair_with_comment(list, b, e, comment);
      }
    }
  }
}

void add_ck_coverage(const char* file, SpiceChar* frame, time_pair_list* list) {
  SpiceInt i, j;
  SpiceInt niv;
  SpiceInt obj;
  SpiceDouble b, e;
  SPICEINT_CELL(ids, MAXOBJ);
  SPICEDOUBLE_CELL(cover, WINSIZ);
  SpiceDouble rotate[3][3];
  SpiceChar comment[SIZE_COMMENT];

  ckobj_c(file, &ids);
  for(i=0; i<card_c(&ids); ++i) {
    obj = SPICE_CELL_ELEM_I(&ids, i);
    scard_c(0, &cover);
    ckcov_c(file, obj, SPICEFALSE, "INTERVAL", 0.0, "TDB", &cover);
    niv = wncard_c ( &cover );
    if (niv > 0) {
      for(j=0; j<niv; ++j) {
        wnfetd_c(&cover, j, &b, &e);
        pxform_c(frame, "J2000", b+CHECK_OFFSET_CK, rotate);
        if (failed_c()) {
          reset_c();
          continue;
        }
        pxform_c(frame, "J2000", e-CHECK_OFFSET_CK, rotate);
        if (failed_c()) {
          reset_c();
          continue;
        }
        sprintf(comment, "%s", file);
        add_time_pair_with_comment(list, b, e, comment);
      }
    }
  }
}

void add_coverage_into_list(const char* type, time_pair_list* list) {
  FUNCPTR add_coverage = NULL;
  SpiceInt i, count;
  SpiceChar file[FILLEN];
  SpiceChar filetype[TYPLEN];
  SpiceChar source[SRCLEN];
  SpiceInt handle;
  SpiceBoolean found;
  SpiceChar* target = NULL;

  if (strcmp(type, "CK") == 0) {
    int inst;
    if ( LUA_TNUMBER != flowse_lua_get_type("basis_instrument_id") ) {
      fprintf(stderr, "%s: cannot parse `basis_instrument_id' value.\n", app_name);
      return;
    }
    inst = flowse_lua_get_integer("basis_instrument_id");

    target = (SpiceChar *)malloc(sizeof(SpiceChar)*WDSIZE);
    get_inst_frame_name(inst, target);
    add_coverage = add_ck_coverage;
  }
  else if(strcmp(type, "SPK") == 0) {
    if ( LUA_TSTRING != flowse_lua_get_type("basis_object") ) {
      fprintf(stderr, "%s: cannot parse `basis_object' value.\n", app_name);
      return;
    }
    target = flowse_lua_copy_string("basis_object");
    add_coverage = add_spk_coverage;
  }

  ktotal_c(type, &count);
  for(i=0; i<count; ++i) {
    kdata_c(i, type, FILLEN, TYPLEN, SRCLEN,
            file, filetype, source, &handle, &found);
    add_coverage(file, target, list);
  }

  if(target) {
    free(target);
  }
}

void loop_spice_kernels(SPICE_FUNC func) {
  int i;
  char* str;
  size_t max = flowse_lua_get_array_length("spice_kernels");
  if (max == (size_t)-1) {
    max = 0;
  }
  for (i = 0; i < max; i++) {
    str = flowse_lua_copy_array_item_as_string("spice_kernels", i);
    func(str);
    free(str);
  }
}

void load_spice_kernels() {
  loop_spice_kernels(furnsh_c);
}

void unload_spice_kernels() {
  loop_spice_kernels(unload_c);
}

void show_coverage(int flag_detail) {
  int i;
  SpiceChar timstr1[TIMLEN];
  SpiceChar timstr2[TIMLEN];

  time_pair_list* list_ck = create_time_pair_list();
  time_pair_list* list_spk = create_time_pair_list();
  time_pair_list* list_cov;

  add_coverage_into_list("CK", list_ck);
  add_coverage_into_list("SPK", list_spk);

  if (flag_detail == DETAIL_TRUE) {
    printf("--- list ck ---\n");
    dump_time_pair_list(list_ck);
    printf("--- list spk ---\n");
    dump_time_pair_list(list_spk);
  }

  reduce_time_pair_list(list_ck);
  reduce_time_pair_list(list_spk);
  list_cov = logical_and_time_pair_list(list_ck, list_spk);

  if (flag_detail == DETAIL_TRUE) {
    printf("--- list ck (after reduced) ---\n");
    dump_time_pair_list(list_ck);

    printf("--- list spk (after reduced) ---\n");
    dump_time_pair_list(list_spk);

    printf("--- list cov ---\n");
    dump_time_pair_list(list_cov);

    printf("--- list coverage ---\n");
  }

  for(i=0; i<list_cov->used; ++i) {
    timout_c (list_cov->pairs[i].start, TIMFMT, TIMLEN, timstr1);
    timout_c (list_cov->pairs[i].end, TIMFMT, TIMLEN, timstr2);
    printf( "%s   to   %s\n", timstr1, timstr2);
  }

  destroy_time_pair_list(list_ck);
  destroy_time_pair_list(list_spk);
  destroy_time_pair_list(list_cov);
}

void usage(const char* cmd) {
  fprintf(stderr, "usage: %s [-d] lua-file\n", cmd);
}

int main(int argc, char** argv) {
  const char* cmd = argv[0];
  const char* lua_file = argv[1];
  int ch;
  int flag_detail = DETAIL_FALSE;

  get_appname(argv[0], app_name, app_path);
  if (!flowse_lua_init(app_name)) {
    return EXIT_FAILURE;
  }

  // parse option
  while((ch = getopt(argc, argv, "d")) != -1) {
    switch (ch) {
    case 'd':
      flag_detail = DETAIL_TRUE;
      break;
    case '?':
      usage(app_name);
      return EXIT_FAILURE;
    }
  }
  argc -= optind;
  argv += optind;
  if (argc != 1) {
    usage(app_name);
    return EXIT_FAILURE;
  }

  // initialize SPICE
  errdev_c("SET", 5, "NULL");
  erract_c("SET", 0, "RETURN");

  if (!flowse_lua_parse(argv[0])) {
    flowse_lua_end();
    return EXIT_FAILURE;
  }

  load_spice_kernels();

  show_coverage(flag_detail);

  unload_spice_kernels();

  flowse_lua_end();

  return EXIT_SUCCESS;
}
