/**
 * @file      flow_ig.c
 * @brief     image generator
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2010-2015 Japan Aerospace Exploration Agency
 *
 */

/*
 * Function description
 * * make_image() creates a canvas and an fitsfile structure, then write them
 *   in the fits file.
 * * set_fits_header_* set fits header.
 * * render_image() draw on canvas.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <libgen.h>
#include <math.h>
#include <sys/types.h>
#include <regex.h>
#include "sv_doc.h"

#ifdef HAVE_CFITSIO
#include <fitsio.h>
#include "canvas_fitsio.h"
#endif

#ifdef HAVE_LIBPNG
#include "canvas_pngio.h"
#endif

#ifdef HAVE_JPEGLIB_H
#include "canvas_jpegio.h"
#endif

#include "flowig_config.h"
#include "flow_common.h"
#include "tge.h"
#include "stl.h"
#include "render.h"

//static char *VERSION = "2.0.0 (2011-02-17)";
char app_path[256], app_name[256];

// Instead of normal image output, it outputs an image for debugging.
//#define ENABLE_IMAGE_DEBUG

#define FRAME_JOIN_VERTICAL    1
#define FRAME_JOIN_HORIZONTAL  2

#ifdef HAVE_CFITSIO
BOOL set_fits_header_primary(FLOWIGConfig* config, SV_DOC* doc, fitsfile* fits);
#endif

BOOL make_image(FLOWIGConfig* config, SV_DOC* doc);
void make_lookdown_image(FLOWIGConfig* config, SV_DOC* doc);
void usage(void);
canvas* load_image(FLOWIGConfig* config, const char* path);

/* Maximum and minimum values of canvas size */
#define MIN_WIDTH   1
#define MAX_WIDTH   100000
#define MIN_HEIGHT  1
#define MAX_HEIGHT  100000

typedef struct _canvasCallbackData {
  FLOWIGConfig* config;
  SV_DOC* doc;
} canvasCallbackData;


//////////////////////////////////////////////////////////////////////
// Obtain image size
//////////////////////////////////////////////////////////////////////
static int get_image_width(SV_DOC* doc, int frame_no, double scale)
{
  return (int)floor((double)doc->frames[frame_no]
                    ->view->image_size[0] * scale + 0.5);
}
static int get_image_height(SV_DOC* doc, int frame_no, double scale)
{
  return (int)floor((double)doc->frames[frame_no]
                    ->view->image_size[1] * scale + 0.5);
}

#ifdef HAVE_CFITSIO
//////////////////////////////////////////////////////////////////////
// Set fits headers
// TODO: Output header as specified
//////////////////////////////////////////////////////////////////////
BOOL set_fits_header_primary(FLOWIGConfig* config, SV_DOC* doc, fitsfile* fits)
{
  int status = 0;
  int i;
  char keyname[32];
  char index[8];

  if (0 != fits_update_key(fits, TSTRING, "GENERATE", app_name, "Application Name", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }
  for (i = 0; i < doc->frame_count; i++) {
    if (doc->frame_count > 1) {
      sprintf(index, "%d", i + 1);
    } else {
      index[0] = '\0';
    }
    SV_VIEW* view = doc->frames[i]->view;
    //
    // DATE
    //
    strcpy(keyname, "DATE");
    strcat(keyname, index);
    if (0 != fits_update_key(fits, TSTRING, keyname, view->date, "UTC", &status)) {
      fits_report_error(stderr, status);
      fits_close_file(fits, &status);
      return FALSE;
    }
    //
    // Location (XYZ)
    //
    strcpy(keyname, "LOC_X");
    strcat(keyname, index);
    if (0 != fits_update_key(fits, TDOUBLE, keyname, &view->location[0],
                             "Location-X", &status)) {
      fits_report_error(stderr, status);
      fits_close_file(fits, &status);
      return FALSE;
    }
    strcpy(keyname, "LOC_Y");
    strcat(keyname, index);
    if (0 != fits_update_key(fits, TDOUBLE, keyname, &view->location[1],
                             "Location-Y", &status)) {
      fits_report_error(stderr, status);
      fits_close_file(fits, &status);
      return FALSE;
    }
    strcpy(keyname, "LOC_Z");
    strcat(keyname, index);
    if (0 != fits_update_key(fits, TDOUBLE, keyname, &view->location[2],
                             "Location-Z", &status)) {
      fits_report_error(stderr, status);
      fits_close_file(fits, &status);
      return FALSE;
    }
    //
    // Boresight (XYZ)
    //
    strcpy(keyname, "DIR_X");
    strcat(keyname, index);
    if (0 != fits_update_key(fits, TDOUBLE, keyname, &view->boresight[0],
                             "Boresight-X", &status)) {
      fits_report_error(stderr, status);
      fits_close_file(fits, &status);
      return FALSE;
    }
    strcpy(keyname, "DIR_Y");
    strcat(keyname, index);
    if (0 != fits_update_key(fits, TDOUBLE, keyname, &view->boresight[1],
                             "Boresight-Y", &status)) {
      fits_report_error(stderr, status);
      fits_close_file(fits, &status);
      return FALSE;
    }
    strcpy(keyname, "DIR_Z");
    strcat(keyname, index);
    if (0 != fits_update_key(fits, TDOUBLE, keyname, &view->boresight[2],
                             "Boresight-Z", &status)) {
      fits_report_error(stderr, status);
      fits_close_file(fits, &status);
      return FALSE;
    }
    strcpy(keyname, "ANGLE");
    strcat(keyname, index);
    if (0 != fits_update_key(fits, TDOUBLE, keyname, &view->fov,
                             "Apparent diameter", &status)) {
      fits_report_error(stderr, status);
      fits_close_file(fits, &status);
      return FALSE;
    }
  }

  return TRUE;
}

//////////////////////////////////////////////////////////////////////
// Set the header for each FITS image
//////////////////////////////////////////////////////////////////////
static BOOL set_fits_header_body(FLOWIGConfig* config, SV_DOC* doc, fitsfile* fits)
{
  int status = 0;
  double ratio, theta;
  double d;
  int n;

  if (0 != fits_update_key(fits, TSTRING, "OBJECT", doc->head->title, "Title", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }

  ratio = doc->frames[0]->view->angle_res / config->scale;
  theta = deg2rad(doc->frames[0]->view->pos_angle);

  //
  // WCS Common
  //
  n = 2;
  if (0 != fits_update_key(fits, TUINT, "WCSAXES", &n,
                           "Number of axes for WCS", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }

  //
  // AXIS 1
  //
  d = (double)doc->frames[0]->view->image_size[0] * config->scale / 2.0 + 0.5;
  if (0 != fits_update_key(fits, TDOUBLE, "CRPIX1", &d,
                           "Pixel coordinate of reference point", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }
  d = -ratio* cos(theta);
  if (0 != fits_update_key(fits, TDOUBLE, "CD1_1", &d, "", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }
  d = ratio * sin(theta);
  if (config->flip) d = -d;
  if (0 != fits_update_key(fits, TDOUBLE, "CD1_2", &d, "", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }
  if (0 != fits_update_key(fits, TSTRING, "CTYPE1", "RA---TAN", "Gnomonic projection", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }
  if (0 != fits_update_key(fits, TDOUBLE, "CRVAL1", &doc->frames[0]->view->center[0],
                           "RA at reference point", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }
  if (0 != fits_update_key(fits, TSTRING, "CUNIT1", "deg     ", "Angles are degrees always", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }

  //
  // AXIS 2
  //
  d = (double)doc->frames[0]->view->image_size[1] * config->scale / 2.0 + 0.5;
  if (0 != fits_update_key(fits, TDOUBLE, "CRPIX2", &d,
                           "Pixel coordinate of reference point", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }
  d = ratio * sin(theta);
  if (0 != fits_update_key(fits, TDOUBLE, "CD2_1", &d, "", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }
  d = ratio * cos(theta);
  if (config->flip) d = -d;
  if (0 != fits_update_key(fits, TDOUBLE, "CD2_2", &d, "", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }
  if (0 != fits_update_key(fits, TSTRING, "CTYPE2", "DEC--TAN", "Gnomonic projection", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }
  if (0 != fits_update_key(fits, TDOUBLE, "CRVAL2", &doc->frames[0]->view->center[1],
                           "Dec at reference point", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }
  if (0 != fits_update_key(fits, TSTRING, "CUNIT2", "deg     ", "Angles are degrees always", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }

  //
  // etc
  //
  if (0 != fits_update_key(fits, TSTRING, "RADESYS", "FK5     ",
                           "Mean IAU 1984 equatorial coordinates", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }
  d = 2000.0;
  if (0 != fits_update_key(fits, TDOUBLE, "EQUINOX", &d,
                           "Equator and equinox of J2000.0", &status)) {
    fits_report_error(stderr, status);
    fits_close_file(fits, &status);
    return FALSE;
  }

  return TRUE;
}

//////////////////////////////////////////////////////////////////////
// Callback function from output routine to write FITS header
//////////////////////////////////////////////////////////////////////
static int canvas_write_fits_callback(canvas* canvas, fitsfile* fits, void* userdata)
{
  canvasCallbackData* data = userdata;
  int n;
  fits_get_hdu_num(fits, &n);
  if (1 == n) {
    set_fits_header_primary(data->config, data->doc, fits);
    if (data->config->monochrome == 0 &&             // ignore imageextension in monochrome
        data->config->fitsformat == CVFormatFitsImageExtension) {
      return 1;
    }
  }
  set_fits_header_body(data->config, data->doc, fits);

  return 1;
}
#endif

//////////////////////////////////////////////////////////////////////
// Create canvas and fitsfile and write it in fits format.
// TODO:If necessary, change the header, drawing order, etc.
//////////////////////////////////////////////////////////////////////
BOOL make_image(FLOWIGConfig* config, SV_DOC* doc)
{
  canvas* cv = NULL;
  canvas* cv_frame = NULL;
  int i, n;
  int attr = 0;
  // for fitsio
  int flen = strlen(config->output_file);
  char* outfile = NULL;
#ifdef HAVE_CFITSIO
  fitsfile* fits = NULL;
#endif
  int status = 0;
  canvasCallbackData callbackData = {config, doc};
  enum FLOWImageFormat format = config->output_format;
  int image_width, image_height;
  int join = FRAME_JOIN_VERTICAL;
  render re;

  if (doc->frame_count < 1) {
    fprintf(stderr, "%s: no frame in document.\n", app_name);
    goto make_image_error;
  }

  if (FLOWImageFormatAuto == format) {
    format = predict_image_format_from_filename(config->output_file);
  }
  if (format == FLOWImageFormatError) {
    exit(1);
  }
  if (format == FLOWImageFormatAuto) {
#ifdef HAVE_CFITSIO
    format = FLOWImageFormatFITS;
#elif defined HAVE_JPEGLIB_H
    format = FLOWImageFormatJPEG;
#elif defined HAVE_LIBPNG
    format = FLOWImageFormatPNG;
#else
    format = FLOWImageFormatUnkown;
#endif
  }

  // Image size determination
  image_width  = get_image_width(doc, 0, config->scale);
  image_height = get_image_height(doc, 0, config->scale);
  if (image_width < image_height) {
    join = FRAME_JOIN_HORIZONTAL;
  }
  if (join == FRAME_JOIN_VERTICAL)  {
    image_height = 0;
    for (i = 0; i < doc->frame_count; i++) {
      image_height += get_image_height(doc, i, config->scale);
    }
  }
  else {
    image_width = 0;
    for (i = 0; i < doc->frame_count; i++) {
      image_width += get_image_width(doc, i, config->scale);
    }
  }
  if (image_width < MIN_WIDTH || image_height < MIN_HEIGHT) {
    fprintf(stderr, "%s: scale option (-s) value too small.\n", app_name);
    goto make_image_error;
  }
  if (MAX_WIDTH < image_width || MAX_HEIGHT < image_height) {
    fprintf(stderr, "%s: scale option (-s) value too large.\n", app_name);
    goto make_image_error;
  }

#ifdef HAVE_CFITSIO
  // Create image export instances
  if (format == FLOWImageFormatFITS) {
    // verwrite FITS file name.
    outfile = malloc(flen + 2);
    outfile[0] = '\0';
    if (0 != strcmp(config->output_file, "-")) {
      strcpy(outfile, "!");
    }
    strcat(outfile, config->output_file);

    if (0 != fits_create_file(&fits, outfile, &status)) {
      fits_report_error(stderr, status);
      return FALSE;
    }
    free(outfile);
    outfile = NULL;
  }
#endif

  // canvas creation
  if (config->monochrome) {
    cv = cv_create(image_width, image_height, CVDepthGray8);
  }
  else {
    cv = cv_create(image_width, image_height, CVDepthRGB8);
  }
  if (!cv) {
    fprintf(stderr, "%s: cannot create canvas.\n", app_name);
    goto make_image_error;
  }

  // image generation
  n = 0;
  for (i = 0; i < doc->frame_count; i++) {
    int frame_height = get_image_height(doc, i, config->scale);
    int frame_width  = get_image_width(doc, i, config->scale);
    canvas* cv_frame;
    if (join == FRAME_JOIN_VERTICAL) {
      cv_frame = cv_create(image_width, frame_height, CVDepthRGB8);
    }
    else {
      cv_frame = cv_create(frame_width, image_height, CVDepthRGB8);
    }
    if (!cv_frame) {
      fprintf(stderr, "%s: cannot create canvas for a frame.\n", app_name);
      goto make_image_error;
    }
    render_init(&re, app_name, config, doc->frames[i], cv_frame,
                FLOWProjectionModeScaleAspectFit);
    if (!render_image(&re)) {
      fprintf(stderr, "%s: render_image() failed.\n", app_name);
      fprintf(stderr, "%s\n", render_get_message(&re));
      goto make_image_error;
    }
    if (join == FRAME_JOIN_VERTICAL) {
      cv_paste_ex(cv, cv_frame, 0, n, image_width, frame_height, CVBlendNone);
      n += frame_height;
    }
    else {
      cv_paste_ex(cv, cv_frame, n, 0, frame_width, image_height, CVBlendNone);
      n += frame_width;
    }
    cv_free(cv_frame);
    cv_frame = NULL;
  }

  // write contents on canvas
  switch(format) {
#ifdef HAVE_CFITSIO
  case FLOWImageFormatFITS:
    if (!config->flip) attr |= CVFormatBottomUp;              /* flip */
    attr |= config->fitsformat;
    cv_write_fits(cv, fits, attr, canvas_write_fits_callback, &callbackData);             // write
    fits_close_file(fits, &status);             // save
    fits = NULL;
    break;
#endif

#ifdef HAVE_JPEGLIB_H
  case FLOWImageFormatJPEG:
    if (0 == strcmp("-", config->output_file)) {
      cv_save_jpeg_with_std(cv, stdout, config->jpeg_quality,
                            0, NULL, NULL);
    }
    else {
      cv_save_jpeg(cv, config->output_file, config->jpeg_quality,
                   0, NULL, NULL);
    }
    break;
#endif

#ifdef HAVE_LIBPNG
  case FLOWImageFormatPNG:
    if (0 == strcmp("-", config->output_file)) {
      cv_save_png_with_std(cv, stdout, config->png_compression_level);
    }
    else {
      cv_save_png(cv, config->output_file, config->png_compression_level);
    }
    break;
#endif

  default:
    fprintf(stderr, "%s: cannot create image. Output format is not supported.\n", app_name);
    goto make_image_error;
    break;
  }

  // post processing
  cv_free(cv);
  cv = NULL;

  return TRUE;

make_image_error:
#ifdef HAVE_CFITSIO
  if (fits) {
    fits_close_file(fits, &status);
    fits = NULL;
  }
#endif
  if (outfile) {
    free(outfile);
    outfile = NULL;
  }
  if (cv) {
    cv_free(cv);
    cv = NULL;
  }
  if (cv_frame) {
    cv_free(cv_frame);
    cv_frame = NULL;
  }
  return FALSE;
}

//////////////////////////////////////////////////////////////////////
// Show usage
//////////////////////////////////////////////////////////////////////
void usage(void)
{
  char* name = basename((char *)app_name);
  fprintf(stderr, "usage: %s [-h] [-f] [-m] [-v] [-s <scale>] [-t <format>]\n"
          "                [[-x <opts>] ...] [-o <outfile>] [<infile>]\n", name);
  fprintf(stderr, "options:\n"
          "  -h           Print this message and exit.\n"
          "  -f           Use vertical flipped file format for output/texture file.\n"
          "               This option affects FITS file only.\n"
          "  -m           Generate grayscale image.\n"
          "  -v           Output xml parser error message.\n"
          "  -s <scale>   Image magnification scale. Default is 1.0.\n"
          "  -t <format>  Output image format.\n"
          "               Available formats are fits,fits-imageextension,\n"
          "               fits-rgbcube,png,jpeg and auto.\n"
          "               Default is auto.\n"
          "  -x <opts>    Extra option. multiple -x allowed.\n"
          "               ge_wire: wire frame rendering.\n"
          "               ge_disable_light: disable lighting.\n"
#ifdef ENABLE_WMS
          "               wms_exit_on_error: exit process at get WMS error.\n"
          "               wms_no_cache: don't use WMS cache file.\n"
          "               wms_delete_cache: remove cache before run.\n"
#endif
          "  -o <outfile> Output filename. Default is STDOUT.\n"
          "  <infile>     Input XML filename. Default is STDIN.\n");
}

//////////////////////////////////////////////////////////////////////
// Show version information
//////////////////////////////////////////////////////////////////////
void version(void)
{
#define SIZE_YESNO 5
  char cfits_support[SIZE_YESNO];
  char jpeg_support[SIZE_YESNO];
  char png_support[SIZE_YESNO];
  char dsk_support[SIZE_YESNO];
  char wms_support[SIZE_YESNO];

#ifdef HAVE_CFITSIO
  strcpy(cfits_support, "yes");
#else
  strcpy(cfits_support, "no");
#endif

#ifdef HAVE_LIBPNG
  strcpy(png_support, "yes");
#else
  strcpy(png_support, "no");
#endif

#ifdef HAVE_JPEGLIB_H
  strcpy(jpeg_support, "yes");
#else
  strcpy(jpeg_support, "no");
#endif

#ifdef ENABLE_SPICE_DSK
  strcpy(dsk_support, "yes");
#else
  strcpy(dsk_support, "no");
#endif

#ifdef ENABLE_WMS
  strcpy(wms_support, "yes");
#else
  strcpy(wms_support, "no");
#endif

  fprintf(stdout, "%s version %s\n", app_name, VERSION);
  fprintf(stdout, "    JPEG support ... %s\n", jpeg_support);
  fprintf(stdout, "    PNG  support ... %s\n", png_support);
  fprintf(stdout, "    FITS support ... %s\n", cfits_support);
  fprintf(stdout, "    WMS  support ... %s\n", wms_support);
  fprintf(stdout, "    DSK  support ... %s\n", dsk_support);
}

////////////////////////////////////////////////////////////////////////
// main
////////////////////////////////////////////////////////////////////////
int main(int argc, char* argv[])
{
  SV_DOC* sv_doc;
  FLOWIGConfig* config;

  // get program path and name
  get_appname(argv[0], app_name, app_path);

  // parse option
  if ((config = flowigconf_create()) == NULL) {
    return 1;
  }
  if (!flowigconf_parse_option(config, argc, argv)) {
    flowigconf_print_message(config);
    flowigconf_free(config);
    return 1;
  }
  if (config->help) {
    /* Print HELP and normal exit */
    usage();
    return 0;
  }
  if (config->version) {
    /* Print VERSION and normal exit */
    version();
    return 0;
  }
#ifdef ENABLE_WMS
  if (config->wms_delete_cache) {
    size_t len;
    char* path;
    path = mpc_get_config_path(FLOW_WMS_CACHE_FILE, NULL, 0, &len);
    unlink(path);
    free(path);
  }
#endif
  // create sv_doc
  if ((sv_doc = new_sv_doc()) == NULL) {
    return 1;
  }

  if (!read_sv_doc(config->input_file, sv_doc, config->xmlerror)) {
    destroy_sv_doc(sv_doc);
    return 1;
  }

  //
  // main process start here
  //
#ifdef ENABLE_IMAGE_DEBUG
  make_lookdown_image(config, sv_doc);
#else
  make_image(config, sv_doc);
#endif

  // clean-up
  destroy_sv_doc(sv_doc);
  flowigconf_free(config);

  return 0;
}

/**
 * create an overhead view
 */
void make_lookdown_image(FLOWIGConfig* config, SV_DOC* doc)
{
  canvas* cv;
  SV_FRAME* frame = doc->frames[0];
  enum FLOWImageFormat format = config->output_format;
  render re;
  int attr = 0;

  // image format
  if (FLOWImageFormatAuto == format) {
    format = predict_image_format_from_filename(config->output_file);
  }
  if (format == FLOWImageFormatError) {
    exit(1);
  }
  if (format == FLOWImageFormatAuto) {
#ifdef HAVE_CFITSIO
    format = FLOWImageFormatFITS;
#elif defined HAVE_JPEGLIB_H
    format = FLOWImageFormatJPEG;
#elif defined HAVE_LIBPNG
    format = FLOWImageFormatPNG;
#else
    format = FLOWImageFormatUnkown;
    fprintf(stderr, "%s: appropriate image format is not found.\n", app_name);
    exit(1);
#endif
  }
  if (!config->flip) attr |= CVFormatBottomUp;        /* flip */

  cv = cv_create(1200, 1200, CVDepthRGB8);
  render_init(&re, app_name, config, frame, cv,
              FLOWProjectionModeScaleAspectFit);
  render_lookdown_image(&re);

  switch(format) {
#ifdef HAVE_CFITSIO
  case FLOWImageFormatFITS:
  {
    char outfile[256];
    outfile[0] = '\0';
    if (0 != strcmp(config->output_file, "-")) {
      strcpy(outfile, "!");
    }
    strcat(outfile, config->output_file);
    cv_save_fits(cv, outfile, attr, NULL, NULL);
  }
  break;
#endif

#ifdef HAVE_JPEGLIB_H
  case FLOWImageFormatJPEG:
    if (0 == strcmp("-", config->output_file)) {
      cv_save_jpeg_with_std(cv, stdout, 85, 0, NULL, NULL);
    }
    else {
      cv_save_jpeg(cv, config->output_file, 85, 0, NULL, NULL);
    }
    break;
#endif

#ifdef HAVE_LIBPNG
  case FLOWImageFormatPNG:
    if (0 == strcmp("-", config->output_file)) {
      cv_save_png_with_std(cv, stdout, 9);
    }
    else {
      cv_save_png(cv, config->output_file, 9);
    }
    break;
#endif
  }
}
