/**
 * @file    flow_se.c
 * @brief   simulator engine
 * @date    2011-03-11
 * @update  2017-08-12
 *
 * @copyright
 * Copyright 2010-2017 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif /* ifdef HAVE_CONFIG_H */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <assert.h>
#include <libgen.h>
#include <SpiceUsr.h>
#include "sv_doc.h"
#include "flow_common.h"
#include "flowse_config.h"
#include "flowse_lua_bind.h"
#include "canvas.h"
#include "tge.h"

#ifndef VERSION
static char *VERSION = "1.0.0 (2011-03-11)";
#endif /* ifndef VERSION */
char app_path[256], app_name[256];

#define FLOWSE_UNKNOWN_STRING "unknown"
#define FLOWSE_UNKNOWN_INT    (-9999)
#define FLOWSE_UNKNOWN_DOUBLE (-9999.0)

// Debug flag: Whether to output observation object information?
// #define PRINT_OBS_INFO

// type for bounds operation
typedef double vec3_t[3];

/**
 * Information of observation object
 */
typedef struct _OBS_INFO {
  char    *basis;            /* Name of observation object */
  double   et;               /* Ephemeris Time */
  double   jd;               /* Time (JD) */
  char    *ref_frame;        /* Input reference frame (spkezr) */
  int      inst_id;          /* Instrument ID of camera */
  char    *shape;            /* FOV Shape of camera (getofv) */
  char    *frame;            /* Camera frame (getfov) */
  char    *aberration;       /* Aberration flag (spkezr) */
  double   boresight[3];     /* Boresight vector */
  vec3_t  *bounds;           /* Boundary vectors */
  SpiceInt nbounds;          /* Number of boundary vectors */
  double   center[3];        /* FOV center vector */
  double   fovmax;           /* Angular radius enclosing FOV */
  double   ref2obsmtx[3][3]; /* Matrix from ref_frame to rame */
  double   obs2refmtx[3][3]; /* Matrix from frame to ref_frame */
} OBS_INFO;

// ////////////////////////////////////////////////////////////////////
// check LUA global variables
// TODO: Increase the number of check items as the number of variables
//     in the configuration file increases
// ////////////////////////////////////////////////////////////////////
static int
check_variables(FLOWSEConfig *config) {
  // kernel (table)
  if (LUA_TTABLE != flowse_lua_get_type("spice_kernels")) {
    fprintf(stderr, "%s: `spice_kernels' is not defined.\n", app_name);
    return 0;
  }

  // title (string)
  if (LUA_TSTRING != flowse_lua_get_type("title")) {
    fprintf(stderr, "%s: no `title' value specified.\n", app_name);
    return 0;
  }

  // time (string)
  // utc2et_c()
  if (!config->datetime &&
      (LUA_TSTRING != flowse_lua_get_type("time")) &&
      (LUA_TSTRING != flowse_lua_get_type("time1")) &&
      ((LUA_TSTRING != flowse_lua_get_type("time_start")) ||
       (LUA_TSTRING != flowse_lua_get_type("time_end")) ||
       (LUA_TNUMBER != flowse_lua_get_type("time_step")))) {
    fprintf(stderr, "%s: no time specifications found.\n", app_name);
    return 0;
  }

  // aberration (string)
  // spkezr_c()
  if (LUA_TSTRING != flowse_lua_get_type("aberration_correction")) {
    fprintf(stderr,
            "%s: cannot parse `aberration_correction' value.\n",
            app_name);
    return 0;
  }

  // Reference coordinate/Observer (string)
  // spkezr_c()
  if (LUA_TSTRING != flowse_lua_get_type("basis_object")) {
    fprintf(stderr, "%s: cannot parse `basis_object' value.\n", app_name);
    return 0;
  }

  // Instrument (string)
  // getfov_c()
  if (LUA_TNUMBER != flowse_lua_get_type("basis_instrument_id")) {
    fprintf(stderr, "%s: cannot parse `basis_instrument_id' value.\n", app_name);
    return 0;
  }

  // Canpus size (width, height)
  if ((LUA_TNUMBER != flowse_lua_get_type("image_width")) ||
      (LUA_TNUMBER != flowse_lua_get_type("image_height"))) {
    fprintf(stderr,
            "%s: cannot parse `image_width' nor 'image_height' value.\n",
            app_name);
    return 0;
  }

  // Drawing object (table)
  // spkezr_c()
  if (LUA_TTABLE != flowse_lua_get_type("output_objects")) {
    fprintf(stderr, "%s: cannot parse `output_objects' value.\n", app_name);
    return 0;
  }

  return 1;
} /* check_variables */

// ////////////////////////////////////////////////////////////////////
// Create time list
// ////////////////////////////////////////////////////////////////////
static int get_time_list(FLOWSEConfig *config,
                         double      **plist) {
  // command-line argument -t
  if (config->datetime) {
    double *p = (double *)malloc(sizeof(double));
    str2et_c(config->datetime, p);
    *plist = p;
    return 1;
  }

  // time
  if (LUA_TSTRING == flowse_lua_get_type("time")) {
    char   *str = flowse_lua_copy_string("time");
    double *p   = (double *)malloc(sizeof(double));
    str2et_c(str, p);
    *plist = p;
    free(str);
    return 1;
  }

  // time1, time2, ...
  if (LUA_TSTRING == flowse_lua_get_type("time1")) {
    int i, count;
    double *p;

    for (i = 2;; i++) {
      char slabel[32];
      sprintf(slabel, "time%d", i);

      if (LUA_TSTRING != flowse_lua_get_type(slabel)) {
        break;
      }
    }
    count = i - 1;
    p     = (double *)malloc(sizeof(double) * count);

    for (i = 0; i < count; i++) {
      char *str, slabel[32];
      sprintf(slabel, "time%d", i + 1);
      str = flowse_lua_copy_string(slabel);
      str2et_c(str, &(p[i]));
      free(str);
    }
    *plist = p;
    return count;
  }

  // time_start, time_end, time_step
  if ((LUA_TSTRING == flowse_lua_get_type("time_start")) &&
      (LUA_TSTRING == flowse_lua_get_type("time_end")) &&
      (LUA_TNUMBER == flowse_lua_get_type("time_step"))) {
    double et_start, et_end, step, *p;
    char  *str;
    int    count, i;

    // start
    str = flowse_lua_copy_string("time_start");
    str2et_c(str, &et_start);
    free(str);

    // end
    str = flowse_lua_copy_string("time_end");
    str2et_c(str, &et_end);
    free(str);

    // step (seconds)
    step = flowse_lua_get_double("time_step");

    // check and go!
    if (step < 1.0e-5) {
      fprintf(stderr, "%s: too small time_step defined in LUA.\n", app_name);
      return 0;
    }
    count  = (int)floor((et_end - et_start) / step + 0.5) + 1;
    p      = (double *)malloc(sizeof(double) * count);
    *plist = p;

    for (i = 0; i < count; i++) {
      *p++ = et_start + (double)i * step;
    }
    return count;
  }

      fprintf(stderr, "%s: no time specification found.\n", app_name);
  return 0;
} /* get_time_list */

// ////////////////////////////////////////////////////////////////////
// Temporarily disable error handling
// ////////////////////////////////////////////////////////////////////
static char error_action_device[128];
static char error_action_action[128];
static void
push_error_action() {
  errdev_c("GET", sizeof(error_action_device), error_action_device);
  erract_c("GET", sizeof(error_action_action), error_action_action);
  errdev_c("SET", sizeof(error_action_device), "NULL");
  erract_c("SET", sizeof(error_action_action), "REPORT");
}

static SpiceBoolean pop_error_action(SpiceInt lenout, SpiceChar *msg) {
  SpiceBoolean r = SPICEFALSE;

  errdev_c("SET", sizeof(error_action_device), error_action_device);
  erract_c("SET", sizeof(error_action_action), error_action_action);

  if (failed_c()) {
    getmsg_c("SHORT", lenout, msg);
    r = SPICETRUE;
  }
  reset_c();
  return r;
}

// ////////////////////////////////////////////////////////////////////
// Get celestial body (main body) texture filename
// ////////////////////////////////////////////////////////////////////
static SpiceBoolean get_object_table_texture_body(SpiceChar *target,
                                                  SV_OBJECT *sv_object,
                                                  int        index) {
  SV_TEXTURE *sv_texture;
  SpiceChar  *src, *region;
  int i;
  double d[4];

  sv_texture            = new_sv_texture();
  sv_texture->component = SV_TEXTURE_COMPONENT_BODY;
  sv_texture->type      = SV_TEXTURE_TYPE_UNKNOWN;

  //
  // file or wms_url + wms_title
  //
  src = flowse_lua_object_table_copy_value("object_table",
                                           target,
                                           index,
                                           "file");

  if (src != NULL) {
    // is a file
    sv_texture->type  = SV_TEXTURE_TYPE_FILE;
    sv_texture->src   = src;
    sv_texture->title = NULL;
  } else {
    #ifdef ENABLE_WMS

    // otherwise, it must be WMS
    SpiceChar *title, *resolution;
    double     resolution_value;
    src = flowse_lua_object_table_copy_value("object_table",
                                             target,
                                             index,
                                             "wms_url");

    if (src == NULL) {
      destroy_sv_texture(sv_texture);
      fprintf(stderr, "%s: no file nor wms_url tag found for object `%s'.\n",
              app_name, target);
      return SPICEFALSE;
    }
    title = flowse_lua_object_table_copy_value("object_table",
                                               target,
                                               index,
                                               "wms_title");

    if (title == NULL) {
      free(src);
      destroy_sv_texture(sv_texture);
      fprintf(stderr,
              "%s: no wms_title tag found for object `%s'.",
              app_name,
              target);
      return SPICEFALSE;
    }
    resolution = flowse_lua_object_table_copy_value("object_table",
                                                    target,
                                                    index,
                                                    "resolution");

    if ((resolution == NULL) ||
        (sscanf(resolution, "%lf", &resolution_value) != 1)) {
        free(src);
        free(title);

      if (resolution != NULL) {
        free(resolution);
      }
      destroy_sv_texture(sv_texture);
      fprintf(
        stderr,
        "%s: no resolution tag found or invalid resolution for object `%s'.",
        app_name,
        target);
      return SPICEFALSE;
    }
    sv_texture->type       = SV_TEXTURE_TYPE_WMS;
    sv_texture->src        = src;
    sv_texture->title      = title;
    sv_texture->resolution = resolution_value;
    free(resolution);
    #endif /* ifdef ENABLE_WMS */
  }

  //
  // region
  //
  region = flowse_lua_object_table_copy_value("object_table",
                                              target,
                                              index,
                                              "region");

  if (region != NULL) {
    if (sscanf(region, "%lf,%lf,%lf,%lf", &d[0], &d[1], &d[2], &d[3]) == 4) {
      for (i = 0; i < 4; i++) {
        sv_texture->region[i] = d[i];
      }
      sv_texture->region_string = region;
    } else {
      fprintf(stderr, "%s: invalid region found for `%s'.\n", app_name, target);
      free(region);
      destroy_sv_texture(sv_texture);
      return SPICEFALSE;
    }
  } else {
    for (i = 0; i < 4; i++) {
      sv_texture->region[i] = 0.0;
    }
    sv_texture->region_string = NULL;

    if (region != NULL) {
      free(region);
    }
  }

  // no ring_radius
  sv_texture->ring_radius = -1.0;

  // add
  sv_object_add_texture(sv_object, sv_texture);

  return SPICETRUE;
} /* get_object_table_texture_body */

// ////////////////////////////////////////////////////////////////////
// Get celestial ring texture filename
// ////////////////////////////////////////////////////////////////////
static SpiceBoolean get_object_table_texture_ring(SpiceChar *target,
                                                  SV_OBJECT *sv_object,
                                                  int        index) {
  SV_TEXTURE *sv_texture;
  SpiceChar  *src, *ring_radius;
  int i;

  sv_texture            = new_sv_texture();
  sv_texture->component = SV_TEXTURE_COMPONENT_RING;

  //
  // file
  //
  src = flowse_lua_object_table_copy_value("object_table",
                                           target,
                                           index,
                                           "file");

  if (src == NULL) {
    destroy_sv_texture(sv_texture);
    fprintf(stderr, "%s: no file tag found for `%s'.\n", app_name, target);
    return SPICEFALSE;
  }
  sv_texture->src   = src;
  sv_texture->type  = SV_TEXTURE_TYPE_FILE;
  sv_texture->title = NULL;

  // ring_radius
  ring_radius = flowse_lua_object_table_copy_value("object_table",
                                                   target,
                                                   index,
                                                   "radius");

  if (ring_radius == NULL) {
    destroy_sv_texture(sv_texture);
    fprintf(stderr, "%s: no radius tag found for `%s'.\n", app_name, target);
    return SPICEFALSE;
  }
  sv_texture->ring_radius = strtod(ring_radius, NULL);
  free(ring_radius);

  // no region
  for (i = 0; i < 4; i++) {
    sv_texture->region[i] = 0.0;
  }
  sv_texture->region_string = NULL;

  // add
  sv_object_add_texture(sv_object, sv_texture);

  return SPICETRUE;
} /* get_object_table_texture_ring */

// ////////////////////////////////////////////////////////////////////
// Obtain model STL filename for target celestial bodies
// ////////////////////////////////////////////////////////////////////
static SpiceBoolean get_object_table_model(SpiceChar *target,
                                           SV_OBJECT *sv_object,
                                           int        index) {
  SV_MODEL  *sv_model;
  SpiceChar *src, *str, *format;
  int rgb[3];

  sv_model = new_sv_model();

  //
  // file
  //
  src = flowse_lua_object_table_copy_value("object_table",
                                           target,
                                           index,
                                           "file");

  if (src == NULL) {
    destroy_sv_model(sv_model);
    fprintf(stderr, "%s: no file tag found for `%s'.\n", app_name, target);
    return SPICEFALSE;
  }
  sv_model->src = src;

  //
  // scale
  //
  str = flowse_lua_object_table_copy_value("object_table",
                                           target,
                                           index,
                                           "scale");

  if (str != NULL) {
    sv_model->scale = strtod(str, NULL);
      free(str);
  } else {
    sv_model->scale = 1.0;
  }

  //
  // color
  //
  str = flowse_lua_object_table_copy_value("object_table",
                                           target,
                                           index,
                                           "color");

  if (str != NULL) {
    if (sscanf(str, "%d,%d,%d", &rgb[0], &rgb[1], &rgb[2]) != 3) {
      free(str);
      destroy_sv_model(sv_model);
      fprintf(stderr, "%s: invalid color found for `%s'.\n", app_name, target);
      return SPICEFALSE;
    }
    sv_model->color = (rgb[0] << 16) | (rgb[1] << 8) | (rgb[2]);
    free(str);
  } else {
    sv_model->color = 0xffffffff;
  }

  //
  // format
  //
  format = flowse_lua_object_table_copy_value("object_table",
                                              target,
                                              index,
                                              "format");

  if (format == NULL) {
    destroy_sv_model(sv_model);
    fprintf(stderr, "%s: invalid model format for `%s'.\n", app_name, target);
    return SPICEFALSE;
  }
  sv_model->format = format;

  //
  // set
  //
  sv_object->model = sv_model;

  return SPICETRUE;
} /* get_object_table_model */

// ////////////////////////////////////////////////////////////////////
// Get the filename specifing texture for object
// ////////////////////////////////////////////////////////////////////
static SpiceBoolean get_object_table(SpiceChar *target,
                                     SV_OBJECT *sv_object) {
  int max, i;

  assert(sv_object->texture_count == 0);
  assert(sv_object->textures == NULL);
  assert(sv_object->model == NULL);

  if (LUA_TTABLE != flowse_lua_get_type("object_table")) {
    return SPICETRUE;
  }

  if ((max =
         flowse_lua_object_table_get_length("object_table",
                                            target)) == (size_t)-1) {
    return SPICETRUE;
  }

  for (i = 0; i < max; i++) {
    char *type = flowse_lua_object_table_copy_value("object_table",
                                                    target,
                                                    i,
                                                    "type");

    if (type == NULL) {
      fprintf(stderr, "%s: type not defined in object_table[%s], ignored.\n",
              app_name, target);
      return SPICETRUE;
    }

    if (strcmp(type, "texture-body") == 0) {
      if (!get_object_table_texture_body(target, sv_object, i)) {
        free(type);
        return SPICEFALSE;
      }
    } else if (strcmp(type, "texture-ring") == 0) {
      if (!get_object_table_texture_ring(target, sv_object, i)) {
        free(type);
        return SPICEFALSE;
      }
    } else if (strcmp(type, "model") == 0) {
      if (sv_object->model != NULL) {
        fprintf(stderr, "%s: duplicate model tag for `%s', ignored.\n",
                app_name, target);
      } else {
        if (!get_object_table_model(target, sv_object, i)) {
          free(type);
          return SPICEFALSE;
        }
      }
    } else {
          free(type);
      fprintf(stderr, "%s: unrecognizable type in object_table[%s], ignored.\n",
              app_name, target);
      return SPICETRUE;
    }
    free(type);
  }

  return SPICETRUE;
} /* get_object_table */

// ////////////////////////////////////////////////////////////////////
// Truncate the trailing space at the end of the string
// ////////////////////////////////////////////////////////////////////
static char* strnolsp(char *str) {
  int   len = strlen(str);
  char *p;

  for (p = str + len - 1; p != str; p--) {
    if (*p != ' ') {
      p[1] = '\0';
      break;
    }
  }
  return str;
}

// ////////////////////////////////////////////////////////////////////
// Obtain RGB values from the spectral type of a star
// ////////////////////////////////////////////////////////////////////
static unsigned long get_star_color(const char *spectral) {
  static struct RGB {
    int r;
    int g;
    int b;
  } rgb[] = {
    { 128, 128, 255 }, // O
    { 160, 160, 255 }, // B
    { 192, 192, 255 }, // A
    { 224, 224, 255 }, // F
    { 255, 192, 192 }, // G
    { 255, 160, 160 }, // K
    { 255, 128, 128 }, // M
    #if 0
    { 170, 187, 255 }, // O
    { 208, 218, 255 }, // B
    { 255, 255, 255 }, // A
    { 255, 255, 203 }, // F
    { 255, 253, 151 }, // G
    { 255, 205, 128 }, // K
    { 255, 158, 151 }, // M
    #endif /* if 0 */
  };
  char type;
  int  index;

  type = *spectral;

  if ((type == '(') || (type == 'D')) {
    type = *(spectral + 1);
  }

  index = 4; // default is 'G'

  switch (type) {
    case 'O':
    case 'W': /* Wolf-Rayet */
      index = 0;
      break;
    case 'B':
      index = 1;
      break;
    case 'A':
      index = 2;
      break;
    case 'F':
      index = 3;
      break;
    case 'G':
      index = 4;
      break;
    case 'K':
    case 'k':
    case 'R':
      index = 5;
      break;
    case 'M':
    case 'N':
    case 'S':
    case 's':
    case 'C': /* carbon star */
      index = 6;
      break;
  }
  return (rgb[index].r << 16)
         | (rgb[index].g << 8)
         | (rgb[index].b);
} /* get_star_color */

// ////////////////////////////////////////////////////////////////////
// Planet and Moon magnitudes
// ////////////////////////////////////////////////////////////////////
SpiceBoolean get_planet_magnitude(SpiceInt     object_id,
                                  SpiceDouble  xyz_planet[3],
                                  SpiceDouble  xyz_basis[3],
                                  SpiceDouble *p_mag) {
  // Magnitude viewed from the distance of 1AU
  static SpiceDouble planet_abs_mag[] = {
    -0.42, // OBJECT_ID_MERCURY
    -4.40, // OBJECT_ID_VENUS
    -2.96, // OBJECT_ID_EARTH
    -1.52, // OBJECT_ID_MARS
    -9.40, // OBJECT_ID_JUPITER
    -8.68, // OBJECT_ID_SATURN
    -7.19, // OBJECT_ID_URANUS
    -6.87, // OBJECT_ID_NEPTUNE
    -1.00, // OBJECT_ID_PLUTO
  };

  SpiceInt index;
  SpiceDouble d_planet_basis, d_planet_sun, d_sun_basis;
  SpiceDouble mag;

  // distance between each objects in AU
  d_planet_basis = vdist_c(xyz_planet, xyz_basis);
  convrt_c(d_planet_basis, "km", "AU", &d_planet_basis);
  d_planet_sun = vnorm_c(xyz_planet);
  convrt_c(d_planet_sun,   "km", "AU", &d_planet_sun);
  d_sun_basis = vnorm_c(xyz_basis);
  convrt_c(d_sun_basis,    "km", "AU", &d_sun_basis);

  switch (object_id) {
    case OBJECT_ID_SUN:
      *p_mag = -27.3 + 5.0 * log10(d_sun_basis);
      return SPICETRUE;

    case OBJECT_ID_MOON:
      *p_mag = 0.38;
      return SPICETRUE;

    case OBJECT_ID_MERCURY:
    case OBJECT_ID_VENUS:
    case OBJECT_ID_EARTH:
    case OBJECT_ID_MARS:
    case OBJECT_ID_JUPITER:
    case OBJECT_ID_SATURN:
    case OBJECT_ID_URANUS:
    case OBJECT_ID_NEPTUNE:
    case OBJECT_ID_PLUTO:
      break;
    default:
      return SPICEFALSE;
  }

  index = (object_id - OBJECT_ID_MERCURY) / 100;
  mag   = planet_abs_mag[index];
  mag  += 5.0 * log10(d_planet_basis * d_planet_sun);

  if (object_id <= OBJECT_ID_JUPITER) {
    SpiceDouble pa  = 0.0;
    SpiceDouble tpa = (d_planet_sun * d_planet_sun
                       + d_planet_basis * d_planet_basis - d_sun_basis)
                      / (2.0 * d_planet_sun * d_planet_basis);
    tpa = min(1.0, max(-1.0, tpa));
    pa  = rad2deg(acos(tpa));

    switch (object_id) {
      case OBJECT_ID_MERCURY:
        mag += (0.0380 - 0.000273 * pa + 0.000002 * pa * pa) * pa;
        break;
      case OBJECT_ID_VENUS:
        mag += (0.0009 + 0.000239 * pa - 0.00000065 * pa * pa) * pa;
        break;
      case OBJECT_ID_MARS:
        mag += 0.016 * pa;
        break;
      case OBJECT_ID_JUPITER:
        mag += 0.005 * pa;
        break;
    }
  } else if (object_id == OBJECT_ID_SATURN) {
    mag -= 1.1 * 0.3;
  }
  *p_mag = mag;

  return SPICETRUE;
} /* get_planet_magnitude */

// ////////////////////////////////////////////////////////////////////
// / Specify the object name and create rotation matrix to camera coordinates
// / @param target   target object
// / @param mtx      matrix
// ////////////////////////////////////////////////////////////////////
SpiceBoolean get_orientation_matrix(OBS_INFO   *obsinfo,
                                    const char *target,
                                    int         object_id,
                                    double      mtx[3][3]) {
  char iau_frame[128];
  char frame_name[128];
  SpiceInt frcode, n;
  SpiceBoolean found;
  char msg[128];
  SpiceDouble bore_rot[3][3];

  strcpy(iau_frame, "IAU_");
  strcat(iau_frame, target);
  namfrm_c(iau_frame, &frcode);

  if (frcode == 0) {
    sprintf(frame_name, "FRAME_%d_NAME", (int)object_id);
    gcpool_c(frame_name, 0, 1, sizeof(iau_frame), &n, iau_frame, &found);

    if (found) {
      namfrm_c(iau_frame, &frcode);
    }
  }

  if (frcode == 0) {
    fprintf(stderr, "%s: no frame found for `%s', ignored.\n", app_name, target);
    return SPICEFALSE;
  }

  push_error_action();
  pxform_c(iau_frame, obsinfo->frame, obsinfo->et, mtx);

  if (pop_error_action(sizeof(msg), msg)) {
    // Issues errors other than FRAMEDATANOTFOUND signal normally
    if (strcmp(msg, "SPICE(FRAMEDATANOTFOUND)") != 0) {
      sigerr_c(msg);
      exit(1);
    }
    fprintf(stderr,
            "%s: no frame data found `%s', ignored.\n",
            app_name,
            iau_frame);
    return SPICEFALSE;
  }

  // Rotate the camera in the direction of the field vector
  mpc_get_boresight_rotation_matrix(obsinfo->boresight, bore_rot);
  mxm_c(bore_rot, mtx, mtx);

  return SPICETRUE;
} /* get_orientation_matrix */

// ////////////////////////////////////////////////////////////////////
// / Retrieve the planet data and set it to SV_DOC.
// / NOTE:
// / * Celestial bodies far from MARS are calculating with BARYCENTER
// / @param sv_frame   SV_FRAME object
// / @param target     target object
// / @param et         epoch (Spice ET)
// / @param ref_frame  reference frame
// / @param aberration aberration
// / @param basis      observing object
// ////////////////////////////////////////////////////////////////////
static void add_solar_object(SV_FRAME *sv_frame,
                             OBS_INFO *obsinfo,
                             char     *target) {
  SV_OBJECT *sd;
  SpiceInt   object_id;
  SpiceInt   i, j;
  char msg[128];
  char object_tag[128];
  char targetname[128];
  char radii[128];
  SpiceDouble  vec[6];
  SpiceDouble  lt;
  SpiceDouble  dist;
  SpiceDouble  radius[3];
  SpiceInt     dim;
  SpiceDouble  mtx[3][3];
  SpiceDouble  tpa, tdist;
  SpiceDouble  radmax, semidm, radmin, ratio;
  SpiceDouble  pl_vec[3];
  SpiceDouble  mag;
  SpiceBoolean found;
  TGEdouble    pos[3], pos_xy[3];

  bodn2c_c(target, &object_id, &found);

  if (!found) {
    // Trust the incoming value as it is,
    // but there is also the possibility of an error.
    fprintf(stderr, "%s: object %s not found, ignored.\n", app_name, target);
    return;
  } else {
    switch (object_id) {
      case OBJECT_ID_SUN:
      case OBJECT_ID_MERCURY:
      case OBJECT_ID_VENUS:
      case OBJECT_ID_EARTH:
      case OBJECT_ID_MOON:
        strcpy(targetname, target);
        break;
      case OBJECT_ID_MARS:

      // On Mars, DE 430 provides only BARYCENTER information.
      case OBJECT_ID_JUPITER:
      case OBJECT_ID_SATURN:
      case OBJECT_ID_URANUS:
      case OBJECT_ID_NEPTUNE:
      case OBJECT_ID_PLUTO:

        // For these celestial bodies above,
        // DE 421 provides only BARYCENTER information.
        sprintf(targetname, "%s BARYCENTER", target);
        break;
      default:

        // Incoming value may occur some errors
        strcpy(targetname, target);
        break;
    }
  }

  // <object> Name of celestial object for node
  switch (get_object_type(object_id)) {
    case OBJECT_TYPE_SUN:
        strcpy(object_tag, target);
      break;
    case OBJECT_TYPE_PLANET:
      sprintf(object_tag, "PLANET.%s",    target);
      break;
    case OBJECT_TYPE_SATELLITE:
      sprintf(object_tag, "SATELLITE.%s", target);
      break;
    case OBJECT_TYPE_ASAT:
    case OBJECT_TYPE_SPACECRAFT:
    case OBJECT_TYPE_BARYCENTER:
    case OBJECT_TYPE_SMALL_BODY:
    case OBJECT_TYPE_INVALID:
    default:
      strcpy(object_tag, target);
      break;
  }

  // Position calculation
  push_error_action();
  spkezr_c(targetname, obsinfo->et, obsinfo->ref_frame,
           obsinfo->aberration, obsinfo->basis, vec, &lt);

  if (pop_error_action(sizeof(msg), msg)) {
    // Issues errors other than SPKINSUFFDATA signal normally
    if (strcmp(msg, "SPICE(SPKINSUFFDATA)") != 0) {
      sigerr_c(msg);
      exit(1);
    }
    fprintf(stderr,
            "%s: spkezr_c() raised error `%s' SPICE(SPKINSUFFDATA), ignored.\n",
            app_name,
            targetname);
    return;
  }

  // convert from reference frame to camera frame
  mxv_c(obsinfo->ref2obsmtx, vec, vec);
  dist = vnorm_c(vec);

  // radius of target (require pck00008.tpc)
  // bodvrd_c(target, "RADII", 3, &dim, radius);
  sprintf(radii, "BODY%d_RADII", (int)object_id);
  gdpool_c(radii, 0, 3, &dim, radius, &found);

  if (!found) {
    for (i = 0; i < 3; i++) {
      radius[i] = 0.0;
    }
    radmax = radmin = 0.0;
    semidm = 0.0;
  } else {
    for (i = dim; i < 3; i++) {
      radius[i] = 0.0;
    }
    radmax = radius[0];
    radmin = radius[0];

    for (i = 1; i < dim; i++) {
      if (radius[i] > radmax) {
        radmax = radius[i];
      }

      if (radius[i] < radmin) {
        radmin = radius[i];
      }
    }

    if (radmin / 2.0 <= dist) {
      ratio = radmax / dist;

      if (fabs(ratio) < 1.0) {
        semidm = asin(radmax / dist); // Maximum angular radius
      } else {
        semidm = 0.0;
      }
    }
  }

  // orientation matrix
  if (!get_orientation_matrix(obsinfo, target, (int)object_id, mtx)) {
    // Use an identity matrix if it can not be obtained
    for (i = 0; i < 3; i++) {
      for (j = 0; j < 3; j++) {
        mtx[i][j] = (i == j) ? 1.0 : 0.0;
      }
    }
  }

  vec_padist(obsinfo->center, vec, &tpa, &tdist);

  if ((dist < radmin / 2.0) || (tdist < obsinfo->fovmax + semidm)) {
    //
    // set values into SV_DOC
    //
    sd              = new_sv_object(SV_OBJECT_TYPE_SOLAR);
    sd->id          = strdup(object_tag);
    sd->name        = strdup(target);
    sd->position[0] = vec[0];
    sd->position[1] = vec[1];
    sd->position[2] = vec[2];

    // magnitude calculation
    vadd_c(sv_frame->view->location, vec, pl_vec);

    if (get_planet_magnitude(object_id, pl_vec,
                             sv_frame->view->location, &mag))
    {
      sd->magnitude = mag;
    }
    sd->distance = dist;

    // texture, model
    if (!get_object_table(object_tag, sd)) {
      destroy_sv_object(sd);
      return;
    }

    sd->solar.radius[0] = radius[0];
    sd->solar.radius[1] = radius[1];
    sd->solar.radius[2] = radius[2];

    // rotation matrix
    for (i = 0; i < 3; i++) {
      for (j = 0; j < 3; j++) {
        sd->solar.rotation[i][j] = mtx[i][j];
      }
    }

    // Coordinates in the canvas
    pos[0] = sd->position[0];
    pos[1] = sd->position[1];
    pos[2] = sd->position[2];
    tgeiVertexToViewport(3, TGE_DOUBLE, pos, pos_xy);
    sd->image_pos[0] = pos_xy[0];
    sd->image_pos[1] = pos_xy[1];

    // Add into SV_DOC
    sv_frame_add_object(sv_frame, sd);
  }
} /* add_solar_object */

// ////////////////////////////////////////////////////////////////////
// / Extract stellar data and set it to SV_DOC
// / TODOf172:
// / * The fixed star ID is hipparcos fixed. Is there other catalogue?
// / * proper motion is not used.
// / * Stellar data without columns with the same name as the columns
// /  used in Hipparcos can not be used.
// / * There is no star name but Hipparcos number istead.
// / @param sv_frame SV_FRAME object
// / @param obsinfo  OBS_INFO object
// / @param eh     E-Kernel handle
// / @param limit   Limit of magnitude
// ////////////////////////////////////////////////////////////////////
static void add_star_data(SV_FRAME *sv_frame,
                          OBS_INFO *obsinfo,
                          SpiceInt  eh,
                          double    limit) {
  int segmax;
  int segno;
  int rowmax;
  int rowno;
  SpiceInt nvals;
  SpiceDouble ra;
  SpiceDouble dec;

  // ID
  SpiceInt id;

  // epoch
  SpiceDouble raep;
  SpiceDouble decep;

  // proper motion
  SpiceDouble rapm;
  SpiceDouble decpm;

  // mag
  SpiceDouble mag;

  // spectral type
  SpiceChar sptype[64];

  // paralax
  SpiceDouble paralax;

  char starid[128];
  char starname[64];
  SpiceDouble   vec[3];
  SpiceEKSegSum segsum;
  SpiceBoolean  idnull, ranull, decnull, magnull, paralaxnull;
  SpiceBoolean  raepnull, decepnull, rapmnull, decpmnull, spnull;
  SV_OBJECT    *sd;
  SpiceDouble   tvec[3];
  SpiceDouble   tpa, tdist;

  TGEdouble pos[3], pos_xy[3];

  if (eh == 0) {
    return;
  }

  // Get number of segments
  segmax = eknseg_c(eh);

  for (segno = 0; segno < segmax; segno++) {
    ekssum_c(eh, segno, &segsum);

    // Get number of rows
    rowmax = segsum.nrows;

    for (rowno = 0; rowno < rowmax; rowno++) {
      // Get stellar data
      ekrcei_c(eh, segno, rowno, "CATALOG_NUMBER", &nvals, &id, &idnull);
      ekrced_c(eh, segno, rowno, "RA",  &nvals, &ra,  &ranull);
      ekrced_c(eh, segno, rowno, "DEC", &nvals, &dec, &decnull);
      ra  *= rpd_c();
      dec *= rpd_c();

      // epoch
      ekrced_c(eh, segno, rowno, "RA_EPOCH",         &nvals, &raep,  &raepnull);
      ekrced_c(eh, segno, rowno, "DEC_EPOCH",        &nvals, &decep, &decepnull);

      // proper motion
      ekrced_c(eh, segno, rowno, "RA_PM",            &nvals, &rapm,  &rapmnull);
      ekrced_c(eh, segno, rowno, "DEC_PM",           &nvals, &decpm, &decpmnull);

      // mag
      ekrced_c(eh, segno, rowno, "VISUAL_MAGNITUDE", &nvals, &mag,   &magnull);

      // spectral type
      ekrcec_c(eh,
               segno,
               rowno,
               "SPECTRAL_TYPE",
               sizeof(sptype),
               &nvals,
               sptype,
               &spnull);

      // annual paralax
      ekrced_c(eh, segno, rowno, "PARLAX", &nvals, &paralax, &paralaxnull);

      // If each value is not NULL and the light intensity is below the limit
      if (!idnull && !ranull && !decnull && !magnull && (mag < limit)) {
        radrec_c(1.0, ra, dec, vec); // RA, Dec -> XYZ
        // Convert to camera coordinates
        mxv_c(obsinfo->ref2obsmtx, vec, tvec);
        vec_padist(obsinfo->center, tvec, &tpa, &tdist);

        // Add data if it falls within angle of view
        if (tdist < obsinfo->fovmax) {
          sprintf(starid,   "STAR.HIPPARCOS.%d", (int)id);
          sprintf(starname, "%d",                (int)id);

          //
          // Set value in SV_DOC
          //
          sd                = new_sv_object(SV_OBJECT_TYPE_STAR);
          sd->id            = strdup(starid);
          sd->name          = strdup(starname);
          sd->position[0]   = tvec[0];
          sd->position[1]   = tvec[1];
          sd->position[2]   = tvec[2];
          sd->magnitude     = mag;
          sd->textures      = NULL;
          sd->texture_count = 0;

          if (!spnull) {
            sd->star.spectral = strdup(strnolsp(sptype));
            sd->star.color    = get_star_color(sd->star.spectral);
          } else {
            sd->star.spectral = NULL;
          }

          if (paralaxnull) {
            sd->distance = -1.0;
          } else {
            double d = 1.0 / tan(paralax * rpd_c());
            convrt_c(d, "AU", "km", &sd->distance);
          }

          // Coordinates in the canvas
          pos[0] = sd->position[0];
          pos[1] = sd->position[1];
          pos[2] = sd->position[2];
          tgeVectorMuld(3, pos, sd->distance, pos);
          tgeiVertexToViewport(3, TGE_DOUBLE, pos, pos_xy);
          sd->image_pos[0] = pos_xy[0];
          sd->image_pos[1] = pos_xy[1];

          // Add data
          sv_frame_add_object(sv_frame, sd);
        }
      }
    }
  }
} /* add_star_data */

// //////////////////////////////////////////////////////////////////////
// Create OBS_INFO structure
// //////////////////////////////////////////////////////////////////////
static void create_obs_info(OBS_INFO *obsinfo,
                            double    et) {
  char *str, jdstr[64];
  char *ref_frame = "J2000";
  char  shape[64];
  char  frame[64];
  SpiceDouble r[3][3];
  int i, j;

  obsinfo->basis      = NULL;
  obsinfo->aberration = NULL;
  obsinfo->ref_frame  = NULL;
  obsinfo->frame      = NULL;
  obsinfo->shape      = NULL;
  obsinfo->bounds     = NULL;
  obsinfo->nbounds    = 0;

  // Ephemeris Time
  obsinfo->et = et;

  // JD conversion
  et2utc_c(obsinfo->et, "J", 7, sizeof(jdstr), jdstr);
  str         = strchr(jdstr, ' ');
  obsinfo->jd = strtod(str, NULL);

  // Reference coordinates / observer
  obsinfo->basis   = flowse_lua_copy_string("basis_object");
  obsinfo->inst_id = flowse_lua_get_integer("basis_instrument_id");

  // Aberration correction etc.
  obsinfo->aberration = flowse_lua_copy_string("aberration_correction");
  obsinfo->ref_frame  = strdup(ref_frame);

  // Get camera's FOV
  i = 0;

  do {
    vec3_t *bounds;
    i     += 10;
    bounds = realloc(obsinfo->bounds, sizeof(vec3_t) * i);

    if (bounds == NULL) {
      break;
    }
    obsinfo->bounds = bounds;
    getfov_c(obsinfo->inst_id, i, sizeof(shape), sizeof(frame), shape, frame,
             obsinfo->boresight, &obsinfo->nbounds, obsinfo->bounds);
  } while (obsinfo->nbounds > 0 && obsinfo->nbounds == i);
  obsinfo->shape = strdup(shape);
  obsinfo->frame = strdup(frame);

  // Coordinate transformation matrix - requires SCLK kernel
  pxform_c(obsinfo->ref_frame, obsinfo->frame, obsinfo->et, obsinfo->ref2obsmtx);

  // Rotate using boresight vector
  mpc_get_boresight_rotation_matrix(obsinfo->boresight, r);
  mxm_c(r, obsinfo->ref2obsmtx, obsinfo->ref2obsmtx);
  invert_c(obsinfo->ref2obsmtx, obsinfo->obs2refmtx);

  // Angular radius of the circle that encloses FOV
  if ((0 == strcmp(shape, "RECTANGLE")) || (0 == strcmp(shape, "POLYGON"))) {
    obsinfo->fovmax = -1.0;

    for (i = 0; i < obsinfo->nbounds - 1; i++) {
      for (j = i + 1; j < obsinfo->nbounds; j++) {
        SpiceDouble dist;
        dist = vsep_c(obsinfo->bounds[i], obsinfo->bounds[j]) * 0.5;

        if (dist > obsinfo->fovmax) {
          obsinfo->fovmax = dist;
        }
      }
    }
  } else if (0 == strcmp(shape, "CIRCLE")) {
    obsinfo->fovmax = vsep_c(obsinfo->boresight, obsinfo->bounds[0]);
  } else if (0 == strcmp(shape, "ELLIPSE")) {
    SpiceDouble dist1, dist2;
    dist1           = vsep_c(obsinfo->boresight, obsinfo->bounds[0]);
    dist2           = vsep_c(obsinfo->boresight, obsinfo->bounds[1]);
    obsinfo->fovmax = ((dist1 > dist2) ? dist1 : dist2);
  }
} /* create_obs_info */

// //////////////////////////////////////////////////////////////////////
// Release OBS_INFO structure
// //////////////////////////////////////////////////////////////////////
static void destroy_obsinfo(OBS_INFO *obsinfo) {
  free(obsinfo->basis);
  free(obsinfo->aberration);
  free(obsinfo->ref_frame);
  free(obsinfo->frame);
  free(obsinfo->shape);
  free(obsinfo->bounds);
}

// //////////////////////////////////////////////////////////////////////
// Is it a valid planet name?
// //////////////////////////////////////////////////////////////////////
static SpiceBoolean is_planet_name(char *name) {
  static char *planets[] = {
    "MERCURY", "VENUS",   "EARTH",    "MARS",    "JUPITER",
    "SATURN",  "URANUS",  "NEPTUNE",  "PLUTO",   NULL,
  };
  int i;

  for (i = 0; planets[i] != NULL; i++) {
    if (strcmp(planets[i], name) == 0) {
      return SPICETRUE;
    }
  }
  return SPICEFALSE;
}

// //////////////////////////////////////////////////////////////////////
// Is it a valid satellite name?
// //////////////////////////////////////////////////////////////////////
static SpiceBoolean is_satellite_name(char *name) {
  int   i, j;
  char *str;

  SPICEINT_CELL(ids, /*MAXOBJ=*/ 1000);
  SpiceInt sat_kernel_max;

  if (strcmp(name, "MOON") == 0) {
    return SPICETRUE;
  }

  sat_kernel_max = flowse_lua_get_array_length("satellite_kernels");

  if (sat_kernel_max == (size_t)-1) {
    sat_kernel_max = 0;
  }

  for (i = 0; i < sat_kernel_max; i++) {
    str = flowse_lua_copy_array_item_as_string("satellite_kernels", i);
    spkobj_c(str, &ids);

    for (j = 0; j < card_c(&ids); j++) {
      SpiceInt object_id = SPICE_CELL_ELEM_I(&ids, j);

      if (get_object_type(object_id) == OBJECT_TYPE_SATELLITE) {
        SpiceChar object_name[128];
        SpiceBoolean found;
        bodc2n_c(object_id, sizeof(object_name) - 1, object_name, &found);

        if (strcmp(object_name, name) == 0) {
          return SPICETRUE;
        }
      }
    }
  }
  return SPICEFALSE;
}

// //////////////////////////////////////////////////////////////////////
// Create SV_FRAME structure
// //////////////////////////////////////////////////////////////////////
static int make_sv_frame(FLOWSEConfig *config,
                         SV_FRAME     *sv_frame,
                         double        et,
                         SpiceInt      eh) {
  int rv = 0; // error return

  OBS_INFO obsinfo;
  int max;
  int i, j, k;

  // from config file
  char *target;

  // getting observer's position (relative coordinate from SUN)
  double obsvec[6];
  double obslt;
  char   date[64];

  // getting camera's FoV
  SpiceDouble cvec[3], cvec_ref[3];
  SpiceDouble mvec[3], mvec_ref[3];
  SpiceDouble range, ra, dec;

  // canvas
  canvas *cv;
  int     w, h;
  double  tge_rect[6];
  double  proj_rect[4];

  struct {
    double top;
    double left;
    double bottom;
    double right;
    double z;
  } projection;
  TGEContext *ge;
  double pa, dist;
  TGEdouble pos_xy[3];

  // Create an OBS_INFO structure
  create_obs_info(&obsinfo, et);

  // location: Since it will cause an error if it is SUN center,
  //        make it an inverse vector on the basis of the basis.
  spkezr_c("SUN", obsinfo.et, obsinfo.ref_frame, obsinfo.aberration,
           obsinfo.basis, obsvec, &obslt);

  // Velocity is unnecessary so ignore the second half of obsvec
  vminus_c(obsvec, obsvec);

  // Convert to camera coordinates
  mxv_c(obsinfo.ref2obsmtx, obsvec, obsvec);

  //
  // view
  //
  sv_frame->view = new_sv_view();
  et2utc_c(obsinfo.et, "ISOC", 3, sizeof(date), date);
  sv_frame->view->date         = strdup(date);
  sv_frame->view->location[0]  = obsvec[0];
  sv_frame->view->location[1]  = obsvec[1];
  sv_frame->view->location[2]  = obsvec[2];
  sv_frame->view->boresight[0] = obsinfo.boresight[0];
  sv_frame->view->boresight[1] = obsinfo.boresight[1];
  sv_frame->view->boresight[2] = obsinfo.boresight[2];

  for (i = 0; i < obsinfo.nbounds; i++) {
    if (!sv_view_add_bound(sv_frame->view, &obsinfo.bounds[i][0])) {
      goto make_sv_frame_error;
    }
  }

  // Angular diameter in degrees
  sv_frame->view->fov   = obsinfo.fovmax * 2.0 * dpr_c();
  sv_frame->view->shape = strdup(obsinfo.shape);

  // Canvas size
  sv_frame->view->image_size[0] = flowse_lua_get_integer("image_width");
  sv_frame->view->image_size[1] = flowse_lua_get_integer("image_height");

  if ((sv_frame->view->image_size[0] <= 0) ||
      (sv_frame->view->image_size[1] <= 0))
  {
        fprintf(stderr, "%s: image size not defined or invalid.\n", app_name);
    goto make_sv_frame_error;
  }

  // SKY color
  if (flowse_lua_get_type("sky_color") == LUA_TSTRING) {
    char *str;
    str = flowse_lua_copy_string("sky_color");

    if (str != NULL) {
      double rgba[4];

      if (sscanf(str, "%lf,%lf,%lf,%lf", &rgba[0], &rgba[1], &rgba[2],
                 &rgba[3]) != 4) {
        fprintf(stderr, "%s: invalid sky_color", app_name);
        free(str);
        goto make_sv_frame_error;
      }
      sv_frame->view->sky_color[0] = rgba[0];
      sv_frame->view->sky_color[1] = rgba[1];
      sv_frame->view->sky_color[2] = rgba[2];
      sv_frame->view->sky_color[3] = rgba[3];
        free(str);
    }
  } else {
    sv_frame->view->sky_color[0] = 0.0;
    sv_frame->view->sky_color[1] = 0.0;
    sv_frame->view->sky_color[2] = 0.0;
    sv_frame->view->sky_color[3] = 1.0;
  }

  //
  // Preparation to determine drawing position canvas and GE
  //
  w  = sv_frame->view->image_size[0];
  h  = sv_frame->view->image_size[1];
  cv = cv_create_null(w, h, CVDepthRGB8);

  mpc_rotate_frame(sv_frame);

  //
  // Determine the projection range of GE
  //
  mpc_get_bounds_rect(sv_frame->view->shape,
                      sv_frame->view->boresight,
                      sv_frame->view->nbounds,
                      sv_frame->view->bounds,
                      tge_rect);

  // Adjust bounds for tgeFrustum
  mpc_get_projection_rect(FLOWProjectionModeScaleAspectFit, w, h,
                          tge_rect[0], tge_rect[1], tge_rect[3], tge_rect[4],
                          proj_rect);

  // To be human readable
  projection.left   = proj_rect[0];
  projection.top    = proj_rect[1];
  projection.right  = proj_rect[2];
  projection.bottom = proj_rect[3];
  projection.z      = sv_frame->view->bounds[2];

  //
  // GE initialization
  //
  ge = tgeCreateContext();
  tgeMakeCurrent(cv, ge);
  tgeMatrixMode(TGE_PROJECTION);
  tgeLoadIdentity();

  // RENDER_FAR should be large enough
  tgeFrustum(projection.left, projection.right,
             projection.bottom, projection.top,
             projection.z, /*RENDER_FAR=*/ 1.e20);
  tgeMatrixMode(TGE_MODELVIEW);
  tgeLoadIdentity();

  // Equator coordinates of the center of the field of view
  vec_middle(&tge_rect[0], &tge_rect[3], obsinfo.center);
  vhat_c(obsinfo.center, cvec);
  mxv_c(obsinfo.obs2refmtx, &cvec, cvec_ref);
  recrad_c(cvec_ref, &range, &ra, &dec);
  sv_frame->view->center[0] = ra * dpr_c();
  sv_frame->view->center[1] = dec * dpr_c();

  // Azimuth in the upward direction of the screen
  mvec[0] = (tge_rect[0] + tge_rect[3]) / 2.0;
  mvec[1] = tge_rect[1];
  mvec[2] = tge_rect[2];
  vhat_c(mvec, mvec);
  mxv_c(obsinfo.obs2refmtx, &mvec, mvec_ref);
  vec_padist(cvec_ref, mvec_ref, &pa, &dist);
  sv_frame->view->pos_angle = pa * dpr_c();

  // Pixel resolution up to the center of the screen top edge
  tgeiVertexToViewport(3, TGE_DOUBLE, mvec, pos_xy);
  sv_frame->view->angle_res = dist / (h / 2.0 - pos_xy[1]) * dpr_c();

  // Output
  if ((max = flowse_lua_get_array_length("output_objects")) == (size_t)-1) {
    max = 0;
  }

  for (i = 0; i < max; i++) {
    target = flowse_lua_copy_array_item_as_string("output_objects", i);

    //
    // Confirm the name of the body
    //
    if (0 == strncmp(target, "PLANET.", 7)) {
      if (!is_planet_name(target + 7)) {
        fprintf(stderr, "%s: unknown planet target `%s'.\n", app_name, target);
        free(target);
        goto make_sv_frame_error;
      }
      strcpy(target, target + 7);
    } else if (0 == strncmp(target, "SATELLITE.", 10)) {
      if (!is_satellite_name(target + 10)) {
        fprintf(stderr,
                "%s: unknown satellite target `%s'.\n",
                app_name,
                target);
        free(target);
        goto make_sv_frame_error;
      }
      strcpy(target, target + 10);
    }

    //
    // add object
    //
    if (0 == strcmp(target, "PLANETS")) {
      // Planet - DE 421 kernel required
      static char *planets[] = {
        "MERCURY", "VENUS",   "EARTH",    "MARS",    "JUPITER",
        "SATURN",  "URANUS",  "NEPTUNE",  "PLUTO",   NULL,
      };

      for (j = 0; planets[j] != NULL; j++) {
        add_solar_object(sv_frame, &obsinfo, planets[j]);
      }
    } else if (0 == strcmp(target, "SATELLITES")) {
      // Moon
        add_solar_object(sv_frame, &obsinfo, "MOON"); // moon

      // A satellite other than the moon
      if (LUA_TTABLE == flowse_lua_get_type("satellite_kernels")) {
        SPICEINT_CELL(satellites, 1000); // Tables to avoid duplication
        SpiceInt sat_kernel_max;

        if ((sat_kernel_max =
               flowse_lua_get_array_length("satellite_kernels")) ==
            (size_t)-1) {
          sat_kernel_max = 0;
        }

        for (j = 0; j < sat_kernel_max; j++) {
          char *str;
          SPICEINT_CELL(ids, /*MAXOBJ=*/ 1000);
          str = flowse_lua_copy_array_item_as_string("satellite_kernels", j);
          spkobj_c(str, &ids);

          for (k = 0; k < card_c(&ids); k++) {
            SpiceInt object_id = SPICE_CELL_ELEM_I(&ids, k);
            SpiceBoolean found = SPICEFALSE;
            int l;

            for (l = 0; l < card_c(&satellites); l++) {
              if (SPICE_CELL_ELEM_I(&satellites, l) == object_id) {
                found = SPICETRUE;
                break;
              }
            }

            if (found) {
              continue;
            }

            if (get_object_type(object_id) == OBJECT_TYPE_SATELLITE) {
              SpiceChar object_name[128];
              SpiceBoolean found;
              bodc2n_c(object_id, sizeof(object_name) - 1, object_name, &found);

              if (!found) {
                sprintf(object_name, "%d", (int)object_id);
              }
              add_solar_object(sv_frame, &obsinfo, object_name);
              appndi_c(object_id, &satellites);
            }
          }
          free(str);
        }
      }
    } else if (0 == strcmp(target, "STARS")) {
      // Star data - hipparcos or similar kernel required
      if (eh == 0) {
          free(target);
        fprintf(stderr, "%s: `star_kernel' not found.\n", app_name);
        goto  make_sv_frame_error;
      }

      if (LUA_TNUMBER != flowse_lua_get_type("star_magnitude_limit")) {
        free(target);
        fprintf(stderr,
                "%s: `star_magnitude_limit' not found or invalid.\n",
                app_name);
        goto  make_sv_frame_error;
      }
      double maglimit = flowse_lua_get_double("star_magnitude_limit");
      add_star_data(sv_frame, &obsinfo, eh, maglimit);
    } else {
      // SUN, PLANET.X, SATELLITE.X, other names
      add_solar_object(sv_frame, &obsinfo, target);
    }
    free(target);
  }

  #ifdef PRINT_OBS_INFO
  printf("JD\n");
  printf("  %f\n", jd);

  printf("OBS location (SUN center, J2000)\n");
  spkezr_c("SUN", obsinfo.et, "J2000", obsinfo.aberration,
           obsinfo.basis, obsvec, &obslt);
  vminus_c(obsvec, obsvec);
  printf("  %f  %f  %f\n", obsvec[0], obsvec[1], obsvec[2]);

  printf("OBS location (EARTH center, J2000)\n");
  spkezr_c("EARTH", obsinfo.et, "J2000", obsinfo.aberration,
           obsinfo.basis, obsvec, &obslt);
  vminus_c(obsvec, obsvec);
  printf("  %f  %f  %f\n", obsvec[0], obsvec[1], obsvec[2]);
  recrad_c(obsvec, &range, &ra, &dec);
  printf("  %f  %f\n", rad2deg(ra), rad2deg(dec));
  #endif /* ifdef PRINT_OBS_INFO */

  rv = 1; // successful return

make_sv_frame_error:

  tgeDestroyContext(ge);
  cv_free(cv);
  destroy_obsinfo(&obsinfo);

  return rv;
} /* make_sv_frame */

// //////////////////////////////////////////////////////////////////////
// Create SV_DOC structure.
// //////////////////////////////////////////////////////////////////////
static int make_sv_doc(FLOWSEConfig *config,
                       SV_DOC       *sv_doc) {
  int   i;
  char *str;
  size_t   max;
  SpiceInt eh = 0; // E-Kernel handle (for star)
  double  *et_list;
  int      et_count;

  // Load the kernel
  if ((max = flowse_lua_get_array_length("spice_kernels")) == (size_t)-1) {
    max = 0;
  }

  for (i = 0; i < max; i++) {
    str = flowse_lua_copy_array_item_as_string("spice_kernels", i);
    furnsh_c(str);
    free(str);
  }

  // Load satellite kernel
  if ((max = flowse_lua_get_array_length("satellite_kernels")) == (size_t)-1) {
    max = 0;
  }

  for (i = 0; i < max; i++) {
    str = flowse_lua_copy_array_item_as_string("satellite_kernels", i);
    furnsh_c(str);
    free(str);
  }

  // Read stellar data
  if ((flowse_lua_get_type("star_kernel") == LUA_TSTRING) &&
      (flowse_lua_get_type("star_magnitude_limit") == LUA_TNUMBER))
  {
    str = flowse_lua_copy_string("star_kernel");
    eklef_c(str, &eh);
    free(str);
  }

  // Time list creation
  if ((et_count = get_time_list(config, &et_list)) < 1) {
    if (eh != 0) {
      ekcls_c(eh);
    }
    return 0;
  }

  //
  // head
  //
  sv_doc->head        = new_sv_head();
  sv_doc->head->title = strdup(flowse_lua_copy_string("title"));

  //
  // frames
  //
  for (i = 0; i < et_count; i++) {
    SV_FRAME *sv_frame = new_sv_frame();

    if (make_sv_frame(config, sv_frame, et_list[i], eh)) {
      sv_doc_add_frame(sv_doc, sv_frame);
    }
  }

  free(et_list);

  if (eh != 0) {
    ekcls_c(eh);
  }

  return 1;
} /* make_sv_doc */

// ////////////////////////////////////////////////////////////////////
// Read the configuration file
// ////////////////////////////////////////////////////////////////////
int read_flowse_lua(FLOWSEConfig *config) {
  int success = 0;

  if (!flowse_lua_init(app_name)) {
    return 0;
  }

  //
  // read
  //
  if (0 == strcmp(config->config_file, "-")) {
    if (flowse_lua_parse(NULL)) {
      success = 1;
    }
  } else {
    if (flowse_lua_parse(config->config_file)) {
      success = 1;
    }
  }

  //
  // check contents
  //
  if (success) {
    if (!check_variables(config)) {
      flowseconf_free(config);
      success = 0;
    }
  }

  //
  // free all if error found
  //
  if (!success) {
    flowse_lua_end();
  }

  return success;
} /* read_flowse_lua */

// ////////////////////////////////////////////////////////////////////
// Display usage
// ////////////////////////////////////////////////////////////////////
void usage() {
  fprintf(stderr,
          "usage: %s [-h] [-t <time>] [-o <outfile>] [<infile>]\n",
          app_name);
  fprintf(
    stderr,
    "options:\n"
    "  -h        Print this message and exit.\n"
    "  -t <time>   Date/time which overrides 'time' value of <infile>.\n"
    "  -o <outfile> Output XML filename. Default is STDOUT.\n"
    "  <infile>    Input configuration filename.\n");
}

// ////////////////////////////////////////////////////////////////////
// Display version information
// ////////////////////////////////////////////////////////////////////
void version(void) {
  fprintf(stdout, "%s version %s\n", app_name, VERSION);
}

// //////////////////////////////////////////////////////////////////////
// main
// //////////////////////////////////////////////////////////////////////
int main(int argc, char *argv[]) {
  int rv;
  SV_DOC *sv_doc;
  FLOWSEConfig *config;

  // get program path and name
  get_appname(argv[0], app_name, app_path);

  if ((config = flowseconf_create()) == NULL) {
    return 1;
  }

  if (!flowseconf_parse_option(config, argc, argv)) {
    flowseconf_print_message(config);
    flowseconf_free(config);
    return 1;
  }

  if (config->help) {
    /* Print HELP and normal exit */
    usage();
    return 0;
  }

  if (config->version) {
    /* Print version and normal exit */
    version();
    return 0;
  }

  // create LUA state
  // now you can use flowse_lua_*** functions
  if (!read_flowse_lua(config)) {
    return 1;
  }

  rv = 1; // error exit

  if ((sv_doc = new_sv_doc()) != NULL) {
    if (make_sv_doc(config, sv_doc)) {
      write_sv_doc(config->output_file, sv_doc);
      rv = 0;
    }
    destroy_sv_doc(sv_doc);
  }

  flowseconf_free(config);

  // free LUA state
  flowse_lua_end();

  return rv;
} /* main */
