/**
 * @file      flowig_config.c
 * @brief     flow_ig option analysis
 * @date      2010-03-12
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <unistd.h>
#include <regex.h>
#include "regscan.h"
#include "sv_doc.h"
#include "flowig_config.h"
#include "flow_common.h"

static int flowigconf_regmatch(FLOWIGConfig* self, const char* str, const char* pattern, int cflags);

//////////////////////////////////////////////////////////////////////
// Constructor
//////////////////////////////////////////////////////////////////////
FLOWIGConfig* flowigconf_create(void)
{
  FLOWIGConfig* self = malloc(sizeof(FLOWIGConfig));
  if (!self) {
    return NULL;
  }
  memset(self, 0, sizeof(FLOWIGConfig));
  self->scale = 1.0;
#ifdef HAVE_CFITSIO
  self->fitsformat = CVFormatFitsImageExtension;
#endif

#ifdef HAVE_JPEGLIB_H
  self->jpeg_quality = 85;
#endif

#ifdef HAVE_LIBPNG
  self->png_compression_level = 9;
#endif
  return self;
}

//////////////////////////////////////////////////////////////////////
// Destructor
//////////////////////////////////////////////////////////////////////
void flowigconf_free(FLOWIGConfig* self)
{
  if (self) {
    if (self->input_file) free(self->input_file);
    if (self->output_file) free(self->output_file);
    if (self->error_message) free(self->error_message);
    free(self);
  }
}

//////////////////////////////////////////////////////////////////////
// analyze command-line options
//////////////////////////////////////////////////////////////////////
int flowigconf_parse_option(FLOWIGConfig* self, int argc, char** argv)
{
  char* input  = NULL;
  char* output = NULL;
  char* format = NULL;
  int ch;
  int errlen = 256;
  char* errmsg = malloc(errlen);
  regscan* rs;
  int len;
  const char* s;
  int n;

  /* command-line option analysis */
  while ((ch = getopt(argc, argv, "?hmVfvs:o:t:x:")) != -1) {
    switch (ch) {
    case 'f':             /* flip */
      self->flip = 1;
      break;
    case 's':             /* scale */
      self->scale = strtod(optarg, NULL);
      break;
    case 'o':             /* output */
      output = optarg;
      break;
    case 't':             /* output format */
      format = optarg;
      break;
    case 'm':             /* monochrome */
      self->monochrome = 1;
      break;
    case 'v':             /* xml error */
      self->xmlerror = 1;
      break;
    case 'h': case '?':             /* help */
      self->help = 1;
      break;
    case 'x':
      if (0 == strcmp("ge_wire", optarg)) {
        self->ge_wire = 1;
      }
      else if (0 == strcmp("ge_disable_light", optarg)) {
        self->ge_disable_light = 1;
      }
#ifdef ENABLE_WMS
      else if (0 == strcmp("wms_exit_on_error", optarg)) {
        self->wms_exit_on_error = 1;
      }
      else if (0 == strcmp("wms_no_cache", optarg)) {
        self->wms_no_cache = 1;
      }
      else if (0 == strcmp("wms_delete_cache", optarg)) {
        self->wms_delete_cache = 1;
      }
#endif
      else {
        snprintf(errmsg, errlen, "%s: unknown arguments for -x.", app_name);
        self->error_message = errmsg;
        return 0;
      }
      break;
    case 'V':             /* version */
      self->version = 1;
      break;
    }
  }

  /* check option errors */
  if (self->help || self->version) {
    free(errmsg);
    return 1;
  }
  if (self->scale <= 0.0) {
    snprintf(errmsg, errlen, "%s: -s <scale> must more than 0.0.", app_name);
    self->error_message = errmsg;
    return 0;
  }
  if (optind < argc) {
    if (optind + 1 < argc) {
      snprintf(errmsg, errlen, "%s: too many arguments.", app_name);
      self->error_message = errmsg;
      return 0;
    }
    input = argv[optind];
  } else {
    input = "-";
  }
  if (!output) {
    output = "-";
  }
  if (!format) {
    format = "auto";
  }
  if (0 == strlen(format)) {
    snprintf(errmsg, errlen, "%s: -t <format> require fits, png or jpeg.", app_name);
    self->error_message = errmsg;
    return 0;
  }
  if (format[0] != 'a' && format[0] != 'A' &&
      format[0] != 'f' && format[0] != 'F' &&
      format[0] != 'p' && format[0] != 'P' &&
      format[0] != 'j' && format[0] != 'J') {
    snprintf(errmsg, errlen, "%s: -t <format> require fits, png or jpeg.", app_name);
    self->error_message = errmsg;
    return 0;
  }

  switch (format[0]) {
  case 'a': case 'A': self->output_format = FLOWImageFormatAuto; break;

#ifdef HAVE_LIBPNG
  case 'p': case 'P':
    self->output_format = FLOWImageFormatPNG;
    rs = regscan_create();
    if (REGSCAN_STATUS_NO_ERROR == regscan_compile(rs, "^png:([1-9])$", REG_ICASE|REG_EXTENDED)) {
      if (REGSCAN_STATUS_MATCH == regscan_exec(rs, format)) {
        s = regscan_match(rs, 1, &len);
        n = strtol(s, NULL, 10);
        self->png_compression_level = n;
      }
    }
    regscan_free(rs);
    break;
#endif

#ifdef HAVE_JPEGLIB_H
  case 'j': case 'J':
    self->output_format = FLOWImageFormatJPEG;
    rs = regscan_create();
    if (REGSCAN_STATUS_NO_ERROR == regscan_compile(rs, "^jpe?g:([1-9][0-9]+)$", REG_ICASE|REG_EXTENDED)) {
      if (REGSCAN_STATUS_MATCH == regscan_exec(rs, format)) {
        s = regscan_match(rs, 1, &len);
        n = strtol(s, NULL, 10);
        if (n <= 100) {
          self->jpeg_quality = n;
        }
        else {
          fprintf(stderr, "%s: max JPEG quality is 100 (cut down).\n", app_name);
          self->jpeg_quality = 100;
        }
      }
    }
    regscan_free(rs);
    break;
#endif

#ifdef HAVE_CFITSIO
  case 'f': case 'F':
    self->output_format = FLOWImageFormatFITS;
    if (flowigconf_regmatch(self, format, "^fits-rgbcube$", REG_ICASE)) {
      self->fitsformat = CVFormatFitsCube;
    }
    else if (flowigconf_regmatch(self, format, "^fits-extension$", REG_ICASE) ||
             flowigconf_regmatch(self, format, "^fits-imageextension$", REG_ICASE)) {
      self->fitsformat = CVFormatFitsImageExtension;
    }
    break;
#endif
  }

  if (self->input_file) {
    free(self->input_file);
  }
  self->input_file = strdup(input);
  if (self->output_file) {
    free(self->output_file);
  }
  self->output_file = strdup(output);

  free(errmsg);
  return 1;
}

//////////////////////////////////////////////////////////////////////
// get error message
//////////////////////////////////////////////////////////////////////
const char* flowigconf_get_message(FLOWIGConfig* self)
{
  return self->error_message;
}

//////////////////////////////////////////////////////////////////////
// show error message
//////////////////////////////////////////////////////////////////////
void flowigconf_print_message(FLOWIGConfig* self)
{
  const char* message = flowigconf_get_message(self);
  if (message) fprintf(stderr, "%s\n", message);
}

static int flowigconf_regmatch(FLOWIGConfig* self, const char* str, const char* pattern, int cflags)
{
  regex_t re;
  regmatch_t m;
  int ret;

  if (0 != regcomp(&re, pattern, cflags)) {
    return 0;
  }
  ret = regexec(&re, str, 0, &m, 0);
  regfree(&re);
  return (ret == 0) ? 1 : 0;
}
