/**
 * @file      flowse_config.c
 * @brief     flow_se option analysis
 * @date      2010-03-12
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include "flowse_config.h"
#include "sv_doc.h"
#include "flow_common.h"

//////////////////////////////////////////////////////////////////////
// Constructor
//////////////////////////////////////////////////////////////////////
FLOWSEConfig* flowseconf_create(void)
{
  FLOWSEConfig* self = malloc(sizeof(FLOWSEConfig));
  if (!self) {
    return NULL;
  }
  memset(self, 0, sizeof(FLOWSEConfig));
  return self;
}

//////////////////////////////////////////////////////////////////////
// Destructor
//////////////////////////////////////////////////////////////////////
void flowseconf_free(FLOWSEConfig* self)
{
  if (self) {
    if (self->output_file) free(self->output_file);
    if (self->error_message) free(self->error_message);
    if (self->datetime) free(self->datetime);
    free(self);
  }
}

//////////////////////////////////////////////////////////////////////
// Analyze command-line options
//////////////////////////////////////////////////////////////////////
int flowseconf_parse_option(FLOWSEConfig* self, int argc, char** argv)
{
  char* config     = NULL;
  char* output     = NULL;
  char* datetime   = NULL;
  int ch;
  int errlen = 512;
  char* errmsg = malloc(errlen);
  struct stat st;

  /* parse option */
  while ((ch = getopt(argc, argv, "?hvVo:t:")) != -1) {
    switch (ch) {
    case 'o':             /* output */
      output = optarg;
      break;
    case 'h': case '?':             /* help */
      self->help = 1;
      break;
    case 't':             /* time (override config value) */
      datetime = optarg;
      break;
    case 'V':             /* version */
      self->version = 1;
      break;
    }
  }

  /* error check */
  if (self->help || self->version) {
    free(errmsg);
    return 1;
  }
  if (optind < argc) {
    if (optind + 1 < argc) {
      snprintf(errmsg, errlen, "%s: too many arguments.", app_name);
      self->error_message = errmsg;
      return 0;
    }
    config = argv[optind];
  } else {
    config = "-";
  }
  if (strcmp(config, "-") && stat(config, &st) != 0) {
    snprintf(errmsg, errlen, "%s: %s: %s",
             app_name, config, strerror(errno));
    self->error_message = errmsg;
    return 0;
  }
  if (!output) {
    output = "-";
  }

  // config file
  if (self->config_file) {
    free(self->config_file);
  }
  self->config_file = strdup(config);
  // output file
  if (self->output_file) {
    free(self->output_file);
  }
  self->output_file = strdup(output);
  // datetime
  if (datetime) {
    self->datetime = strdup(datetime);
  }

  free(errmsg);
  return 1;
}

//////////////////////////////////////////////////////////////////////
// Get error message
//////////////////////////////////////////////////////////////////////
const char* flowseconf_get_message(FLOWSEConfig* self)
{
  return self->error_message;
}

//////////////////////////////////////////////////////////////////////
// Show error message
//////////////////////////////////////////////////////////////////////
void flowseconf_print_message(FLOWSEConfig* self)
{
  const char* message = flowseconf_get_message(self);
  if (message) fprintf(stderr, "%s\n", message);
}
