/**
 * @file      flowse_lua_bind.c
 * @brief     SPICE/FLOW lua binding
 * @date      2010-03-12
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <assert.h>
#include "SpiceUsr.h"
#include "flowse_lua_bind.h"

static lua_State* Lua = NULL;
static const char* lua_app_name = NULL;

#if LUA_VERSION_NUM>=502
#define lua_objlen lua_rawlen
#endif

//////////////////////////////////////////////////////////////////////
// init / end
//////////////////////////////////////////////////////////////////////

/**
 * Create the LUA state and bind the flow function.
 * @result LUA state if successful, otherwide NULL
 */
int flowse_lua_init(const char* app_name)
{
  lua_State* lua;

  // Check the initialization
  if (Lua) {
    return (int)Lua;
  }
  lua_app_name = app_name;

  // Initialize LUA
  lua = luaL_newstate();
  luaL_openlibs(lua);           // Include standard library

  // Set LUA functions
  spice_lua_init(lua);

  Lua = lua;

  return (int)lua;
}

/*
 * Create a temporary file and read the file as LUA
 */
int flowse_lua_parse_fd(int infd)
{
  char luafile[] = "flow_se_conf_XXXXXX";
  int fd;
  char buf[512];
  char msg[256];
  int rc;
  int wc;

  if ((fd = mkstemp(luafile)) == -1) {
    sprintf(msg, "%s: cannot create temporary file", lua_app_name);
    perror(msg);
    return 0;
  }
  while ((rc = read(infd, buf, sizeof(buf))) > 0) {
    if ((wc = write(fd, buf, rc)) < 0) {
      close(fd);
      unlink(luafile);
      sprintf(msg, "%s: cannot write to temporary file", lua_app_name);
      perror(msg);
      return 0;
    }
  }
  close(fd);
  if (rc < 0) {
    unlink(luafile);
    sprintf(msg, "%s: cannot read temporary file", lua_app_name);
    perror(msg);
    return 0;
  }
  if (!flowse_lua_parse(luafile)) {
    unlink(luafile);
    return 0;
  }
  unlink(luafile);
  return 1;
}

/**
 * Read the LUA format setting file and execute it.
 * @param path  path to setting file
 * @result return 1 if successful, otherwise 0
 */
int flowse_lua_parse(const char* path)
{
  lua_State* lua = flowse_lua_get_state();
  int err;
  const char* errmsg;

  luaL_loadfile(lua, path);
  if ((err = lua_pcall(lua, 0, 0, 0)) != 0) {
    errmsg = lua_tostring(lua, 1);
    fprintf(stderr, "%s: (config file error): %s\n", lua_app_name, errmsg);
    return 0;
  }
  return 1;
}

/**
 * Release LUA state
 */
void flowse_lua_end(void)
{
  if (Lua) {
    lua_close(Lua);
  }
  Lua = NULL;
}

/**
 * Get LUA state
 * @result lua_State variable
 */
lua_State* flowse_lua_get_state(void)
{
  return Lua;
}

/**
 * Check whether global variable: name is defined
 * @param name  name variable
 */
int flowse_lua_is_defined(const char* name)
{
  int is_defined = 1;
  lua_State* lua = flowse_lua_get_state();
  if (!lua_checkstack(lua, 1)) {
    luaL_error(lua, "cannot take stack space %s", name);
    return 0;               /* Stack can not be extended */
  }
  lua_getglobal(lua, name); /* Even if there is no value, the stack will increase */
  if (lua_isnil(lua, -1)) {
    is_defined = 0;         /* Stack top is nil = there is no value */
  }
  lua_pop(lua, 1);
  return is_defined;
}

/**
 * Examine whether the global variable name is defined and returns its type.
 * @param name  variable
 */
int flowse_lua_get_type(const char* name)
{
  lua_State* lua = flowse_lua_get_state();
  int type;

  if (!lua_checkstack(lua, 1)) {
    luaL_error(lua, "cannot take stack space %s", name);
    return 0;              /* Stack can not be extended */
  }
  lua_getglobal(lua, name);
  type = lua_type(lua, -1);
  lua_pop(lua, 1);

  return type;
}

/**
 * Returns the int value of the global variable name.
 */
int flowse_lua_get_integer(const char* name)
{
  lua_State* lua = flowse_lua_get_state();
  int value;
  int level;

  if (LUA_TNUMBER != flowse_lua_get_type(name)) {
    luaL_error(lua, "value %s is not number", name);
    return 0.0;
  }
  if (!lua_checkstack(lua, 1)) {
    luaL_error(lua, "cannot take stack space %s", name);
    return 0.0;             /* Stack can not be extended */
  }
  level = lua_gettop(lua);
  lua_getglobal(lua, name);
  if (level >= lua_gettop(lua)) {
    luaL_error(lua, "cannot get value %s", name);
    return 0.0;             /* Stack is not stacked */
  }
  value = lua_tointeger(lua, -1);
  lua_pop(lua, 1);

  return value;
}

/**
 * Return the double value of the global variable name.
 */
double flowse_lua_get_double(const char* name)
{
  lua_State* lua = flowse_lua_get_state();
  double value;
  int level;

  if (LUA_TNUMBER != flowse_lua_get_type(name)) {
    luaL_error(lua, "value %s is not number", name);
    return 0.0;
  }
  if (!lua_checkstack(lua, 1)) {
    luaL_error(lua, "cannot take stack space %s", name);
    return 0.0;             /* Stack can not be extended */
  }
  level = lua_gettop(lua);
  lua_getglobal(lua, name);
  if (level >= lua_gettop(lua)) {
    luaL_error(lua, "cannot get value %s", name);
    return 0.0;             /* Stack is not stacked */
  }
  value = lua_tonumber(lua, -1);
  lua_pop(lua, 1);

  return value;
}

/**
  * Get string from global variable of LUA, alloc and return it.
  * Please free memory with returned value.
  * @param name  Global variable name.
  * @result returns the values of variable. NULL will be returned if
  *         the value is nil, not convertible, or none.
  */
char* flowse_lua_copy_string(const char* name)
{
  lua_State* lua = flowse_lua_get_state();
  const char* str;
  char* ptr;
  int level;

  if (!lua_checkstack(lua, 1)) {
    luaL_error(lua, "cannot take stack space %s", name);
    return NULL;                 /* Stack can not be extended */
  }
  level = lua_gettop(lua);
  lua_getglobal(lua, name);
  if (level >= lua_gettop(lua)) {
    luaL_error(lua, "cannot get value %s", name);
    return NULL;                 /* Stack is not stacked */
  }
  str = lua_tostring(lua, -1);
  if (!str) {
    luaL_error(lua, "cannot convert %s to string", name);
    lua_pop(lua, 1);             /* String conversion failure */
    return NULL;
  }
  ptr = strdup(str);
  lua_pop(lua, 1);

  return ptr;
}

/**
 * Returns the array length of the global variable of LUA.
 * @param name  variable name of LUA
 * @result returns the length. -1 will be returned if the name is not array, or
 *         the value is not defined.
 */
size_t flowse_lua_get_array_length(const char* name)
{
  lua_State* lua = flowse_lua_get_state();
  size_t length;

  if (!lua_checkstack(lua, 1)) {
    luaL_error(lua, "cannot take stack space %s", name);
    return (size_t)-1;             /* Stack is not stacked */
  }
  if (LUA_TTABLE != flowse_lua_get_type(name)) {
    return (size_t)-1;             /* not table */
  }
  lua_getglobal(lua, name);
  length = lua_objlen(lua, -1);
  lua_pop(lua, 1);

  return length;
}

/**
 * Obtains an array element (character string) from the global variable of LUA,
 * alloc, and returns it.
 * Please free memory with return value.
 * @param name  variable of LUA
 * @param index index of the element.
 *              Unlike LUA rules, the leading element index is 0.
 * @result table The element string of table. NULL if the stack can not be
 *         extended, the variable does not exist, the index is out of range,
 *         or the element can not be converted to a string.
 */
char* flowse_lua_copy_array_item_as_string(const char* name, int index)
{
  lua_State* lua = flowse_lua_get_state();
  size_t length;
  const char* str;
  char* ptr;

  if (!lua_checkstack(lua, 2)) {
    luaL_error(lua, "cannot take stack space %s", name);
    return NULL;             /* Stack can not be extended */
  }
  length = flowse_lua_get_array_length(name);
  if (length == (size_t)-1) {
    return NULL;
  }
  if (index < 0 || length <= index) {
    return NULL;                 /* index is out of range*/
  }
  lua_getglobal(lua, name);
  lua_pushnumber(lua, (lua_Number)index+1);       /* index -> top */
  lua_gettable(lua, -2);         /* name[index] -> top (index is removed) */
  str = lua_tostring(lua, -1);
  if (!str) {
    /* String conversion failure */
    lua_pop(lua, 2);
    return NULL;
  }
  ptr = strdup(str);
  lua_pop(lua, 2);

  return ptr;
}

/**
 * Get the number of array elements of the object specified in object_table.
 * @param name   variable name of LUA object_table
 * @param object Celestial body ID (Subscript of object_table array)
 * @result returns the length. -1 will be returned if the name is not array, or
 *         the value is not defined.
 */
size_t flowse_lua_object_table_get_length(const char* name, const char* object)
{
  lua_State* lua = flowse_lua_get_state();
  size_t length;

  if (!lua_checkstack(lua, 2)) {
    luaL_error(lua, "cannot take stack space %s", name);
    return (size_t)-1;             /* Stack can not be extended */
  }
  if (LUA_TTABLE != flowse_lua_get_type(name)) {
    return (size_t)-1;             /* not table*/
  }
  lua_getglobal(lua, name);
  lua_pushstring(lua, object);
  lua_gettable(lua, -2);           /* name[object] -> top (index is removed) */
  if (lua_isnil(lua, -1)) {
    lua_pop(lua, 2);
    return (size_t)-1;             /* no definition of name[object] */
  }
  length = lua_objlen(lua, -1);
  lua_pop(lua, 2);

  return length;
}

/**
 * Get the value of the specified object / index of object_table.
 * @param name   variable name of LUA object_table
 * @param object Celestial body ID (Subscript of object_table array)
 * @param index  Object definition array subscript
 * @param tag    name of property
 * @result Object definition value. NULL if the value is not defined.
 */
char* flowse_lua_object_table_copy_value(const char* name, const char* object,
                                         int index, const char* tag)
{
  lua_State* lua = flowse_lua_get_state();
  size_t length;
  const char* str;
  char* ptr;

  if (!lua_checkstack(lua, 4)) {
    luaL_error(lua, "cannot take stack space %s", name);
    return NULL;             /* Stack can not be extended */
  }
  if (LUA_TTABLE != flowse_lua_get_type(name)) {
    return NULL;             /* not table */
  }
  lua_getglobal(lua, name);
  lua_pushstring(lua, object);
  lua_gettable(lua, -2);     /* name[object] -> top (object is removed) */
  if (lua_isnil(lua, -1)) {
    lua_pop(lua, 2);
    return NULL;             /* no definition of name[object] */
  }
  length = lua_objlen(lua, -1);
  if (index < 0 || length <= index) {
    lua_pop(lua, 2);
    return NULL;             /* index is out of range */
  }
  lua_pushnumber(lua, (lua_Number)index+1);       /* index -> top */
  lua_gettable(lua, -2);     /* name[object][index] -> top (index is removed) */
  if (lua_isnil(lua, -1)) {
    lua_pop(lua, 3);
    return NULL;             /* no definition of name[object][index] */
  }
  lua_pushstring(lua, tag);
  lua_gettable(lua, -2);     /* name[object][index][tag] -> top (tag is removed) */
  if (lua_isnil(lua, -1)) {
    lua_pop(lua, 4);
    return NULL;             /* no definition of ame[object][index][tag] */
  }
  str = lua_tostring(lua, -1);
  if (!str) {
    /* String conversion failure */
    lua_pop(lua, 4);
    return NULL;
  }
  ptr = strdup(str);
  lua_pop(lua, 4);

  return ptr;
}
