/**
 * @file      flowse_lua_bind.h
 * @brief     SPICE/FLOW lua binding
 * @date      2010-03-12
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

#ifndef _FLOWSE_LUA_BIND_H
#define _FLOWSE_LUA_BIND_H

#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>
#include "spice_lua.h"

__BEGIN_DECLS
//////////////////////////////////////////////////////////////////////
// 初期化、ファイル実行
//////////////////////////////////////////////////////////////////////

/** 初期化。*/
int flowse_lua_init(const char* app_name);
/** ファイル読み込み＆実行。*/
int flowse_lua_parse(const char* path);
/** ファイルディスクリプタ読み込み＆実行。*/
int flowse_lua_parse_fd(int infd);
/** 開放。*/
void flowse_lua_end(void);
/** LUA ステート取得。*/
lua_State* flowse_lua_get_state(void);


//////////////////////////////////////////////////////////////////////
// グローバル変数取得用関数
//////////////////////////////////////////////////////////////////////

/**
 * グローバル変数 name が定義されているかどうかを返します。
 */
int flowse_lua_is_defined(const char* name);

/**
 * グローバル変数 name が定義されているか調べ、その型を返します。
 */
int  flowse_lua_get_type(const char* name);

/**
 * Integer 型のグローバル変数 name の値を返します。
 */
int flowse_lua_get_integer(const char* name);

/**
 * Double 型のグローバル変数 name の値を返します。
 */
double flowse_lua_get_double(const char* name);

/**
 * String 型のグローバル変数 name の値を返します。
 */
char* flowse_lua_copy_string(const char* name);

/**
 * Table (array) 型のグローバル変数 name の要素数を返します。
 */
size_t flowse_lua_get_array_length(const char* name);

/**
 * Table (array) 型のグローバル変数 name の指定位置の要素を String 型で
 * 返します。
 */
char* flowse_lua_copy_array_item_as_string(const char* name, int index);

/**
 * object_table の指定天体の texture/model 定義数を返します。
 */
size_t flowse_lua_object_table_get_length(const char* name, const char* object);

/**
 * object_table の指定プロパティ文字列を String 型で返します。
 */
char* flowse_lua_object_table_copy_value(const char* name, const char* object,
                                         int index, const char* tag);

__END_DECLS

#endif // _FLOWSE_LUA_BIND_H
