/**
 * @file      poly_mesh.c
 * @brief     3-Dimensional polygon mesh
 * @date      2017-05-20
 *
 * @copyright
 * Copyright 2010 AstroArts Inc.
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <SpiceUsr.h>
#include <assert.h>
#include <math.h>
#include "poly_mesh.h"
#include "sv_doc.h"
#include "flow_common.h"

static double degmal(double x)
{
  if (0 <= x && x >= 360.0) {
    return x;
  }
  if (0 < x) {
    return fmod(x, 360.0);
  }
  return 360.0 - fmod(-x, 360.0);
}

poly_mesh* mesh_alloc(PolyMeshType type, PolyMeshAccuracy accuracy)
{
  switch (type) {
  case PolyMeshTypeSphere:
    switch (accuracy) {
    case PolyMeshAccuracyMedium:
      return mesh_alloc_sphere(72, 36);
      break;
    case PolyMeshAccuracyHigh:
      return mesh_alloc_sphere(240, 120);
      break;
    case PolyMeshAccuracyNormal:
    default:
      return mesh_alloc_sphere(60, 30);
      break;
    }
    break;
  }
  return NULL;
}

poly_mesh* mesh_alloc_sphere(int xsplit, int ysplit)
{
  poly_mesh* self;
  int x, y;
  double alpha, delta;
  double xyz[3];
  int pos;
  int off;
  int idx;

  self = malloc(sizeof(poly_mesh));
  if (!self) {
    return NULL;
  }
  memset(self, 0, sizeof(poly_mesh));

  self->polygon_size = xsplit*ysplit*2;
  self->vertex_size = (xsplit+1)*(ysplit+1);
  self->vertex = malloc(sizeof(double)*3*self->vertex_size);
  self->texcoord = malloc(sizeof(double)*2*self->vertex_size);
  self->index = malloc(sizeof(int)*self->polygon_size*3);

  /* create vertices */
  for (y = 0; y <= ysplit; y++) {
    delta = 90.0 - 180.0 * (double)y / (double)ysplit;
    delta = deg2rad(delta);
    for (x = 0; x <= xsplit; x++) {
      alpha = deg2rad(360.0 * (double)x / (double)xsplit);
      if (x == xsplit) {
        alpha = 0;
      }
      radrec_c(1.0, alpha, delta, xyz);
      pos = y * (xsplit+1) + x;
      off = pos*3;
      self->vertex[off++] = xyz[0];
      self->vertex[off++] = xyz[1];
      self->vertex[off++] = xyz[2];
      off = pos*2;
      self->texcoord[off++] = (double)x / (double)xsplit;
      self->texcoord[off++] = (double)y / (double)ysplit;
    }
  }
  /* create indices */
  for (y = 0; y < ysplit; y++) {
    for (x = 0; x < xsplit; x++) {
      off = (y*xsplit+x) * 6;
      idx = (y*(xsplit+1)+x);
      self->index[off]   = (idx);
      self->index[off+1] = (idx+xsplit+1);
      self->index[off+2] = (idx+xsplit+2);
      self->index[off+3] = (idx);
      self->index[off+4] = (idx+xsplit+2);
      self->index[off+5] = (idx+1);
    }
  }
  return self;
}

poly_mesh* mesh_alloc_sphere_part(int xsplit, int ysplit, double minlat, double minlng, double maxlat, double maxlng)
{
  poly_mesh* self;
  int x, y;
  double alpha, delta;
  double xyz[3];
  int pos;
  int off;
  int idx;

  assert(minlat < maxlat);
  assert(-90.0 <= minlat);
  assert(90.0 >= maxlat);
  assert(minlng < maxlng);
  assert(maxlng - minlng < 360.0);
  assert(minlng != maxlng);

  if (minlng > maxlng) {
    minlng -= 360.0;
  }
  self = malloc(sizeof(poly_mesh));
  if (!self) {
    return NULL;
  }
  memset(self, 0, sizeof(poly_mesh));

  self->polygon_size = xsplit*ysplit*2;
  self->vertex_size = (xsplit+1)*(ysplit+1);
  self->vertex = malloc(sizeof(double)*3*self->vertex_size);
  self->texcoord = malloc(sizeof(double)*2*self->vertex_size);
  self->index = malloc(sizeof(int)*self->polygon_size*3);

  /* create vertices */
  for (y = 0; y <= ysplit; y++) {
    delta = maxlat - (maxlat - minlat) * (double)y / (double)ysplit;
    delta = deg2rad(delta);
    for (x = 0; x <= xsplit; x++) {
      alpha = deg2rad(degmal(minlng + (maxlng - minlng) * (double)x / (double)xsplit));
      radrec_c(1.0, alpha, delta, xyz);
      pos = y * (xsplit+1) + x;
      off = pos*3;
      self->vertex[off++] = xyz[0];
      self->vertex[off++] = xyz[1];
      self->vertex[off++] = xyz[2];
      off = pos*2;
      self->texcoord[off++] = (double)x / (double)xsplit;
      self->texcoord[off++] = (double)y / (double)ysplit;
    }
  }
  /* create indices */
  for (y = 0; y < ysplit; y++) {
    for (x = 0; x < xsplit; x++) {
      off = (y*xsplit+x) * 6;
      idx = (y*(xsplit+1)+x);
      self->index[off]   = (idx);
      self->index[off+1] = (idx+xsplit+1);
      self->index[off+2] = (idx+xsplit+2);
      self->index[off+3] = (idx);
      self->index[off+4] = (idx+xsplit+2);
      self->index[off+5] = (idx+1);
    }
  }
  return self;
}

void mesh_move(poly_mesh* self, double x, double y, double z)
{
  int i;
  int off;
  for (i = 0; i < self->vertex_size; i++) {
    off = i*3;
    self->vertex[off++] += x;
    self->vertex[off++] += y;
    self->vertex[off]   += z;
  }
}

void mesh_scale(poly_mesh* self, double x, double y, double z)
{
  int i;
  int off;
  for (i = 0; i < self->vertex_size; i++) {
    off = i*3;
    self->vertex[off++] *= x;
    self->vertex[off++] *= y;
    self->vertex[off]   *= z;
  }
}

void mesh_rotate(poly_mesh* self, double mtx[3][3])
{
  int i;
  int off;
  for (i = 0; i < self->vertex_size; i++) {
    off = i*3;
    mxv_c(mtx, &self->vertex[off], &self->vertex[off]);
  }
}

void mesh_free(poly_mesh* self)
{
  free(self->vertex);
  free(self->texcoord);
  free(self->index);
  free(self);
}
