/**
 * @file      poly_mesh.h
 * @brief     3-Dimensional polygon mesh
 * @date      2017-05-20
 *
 * @copyright
 * Copyright 2010 AstroArts Inc.
 *
 */

#ifndef _POLY_MESH_H
#define _POLY_MESH_H

typedef struct _poly_mesh {
  size_t polygon_size;
  size_t vertex_size;
  double* vertex;
  double* texcoord;
  int* index;
} poly_mesh;

typedef enum _PolyMeshType {
  PolyMeshTypeSphere = 0
} PolyMeshType;

typedef enum _PolyMeshAccuracy {
  PolyMeshAccuracyNormal = 0,
  PolyMeshAccuracyMedium,
  PolyMeshAccuracyHigh
} PolyMeshAccuracy;

__BEGIN_DECLS

/**
 * @brief Generate the model
 * @param[in] type      The polygon type (currently, sphere is only available)
 * @param[in] accuracy  The resolution of the model (normal, medium, high)
 * @return              Upon successful completion, mesh_alloc() returns
 *                      the pointer of the struct poly_mesh.
 */
poly_mesh* mesh_alloc(PolyMeshType type, PolyMeshAccuracy accuracy);

/**
 * @brief Generate the spherical model
 * @param[in] xsplit  The number of divisions in the x (longitude) direction
 * @param[in] ysplit  The number of divisions in the y (latitude) direction
 * @return            Upon successful completion, mesh_alloc_sphere returns
 *                    the pointer of the struct poly_mesh for a spherical model.
 */
poly_mesh* mesh_alloc_sphere(int xsplit, int ysplit);

/**
 * @brief Generate a partial model of a sphere
 * @param[in] xsplit  The number of divisions in the x (longitude) direction
 * @param[in] ysplit  The number of divisions in the y (latitude) direction
 * @param[in] minlat  Minimum latitude (-90 <= deg < 90).
 * @param[in] minlng  Minimum longitude (0 <= deg < 360).
 * @param[in] maxlat  Maximum latitude (-90 < deg <= 90).
 * @param[in] maxlng  Maximum longigutde (0 < deg <= 360).
 * @return            An error occurs if minlat < maxlat and minlng < maxlng
 *                    is not satisfied.
 */
poly_mesh* mesh_alloc_sphere_part(int xsplit, int ysplit, double minlat, double minlng, double maxlat, double maxlng);

/**
 * @brief Move the vertex by the given x y z offsets
 * @param[in] x  X-offset
 * @param[in] y  Y-offset
 * @param[in] z  Z-offset
 */
void mesh_move(poly_mesh* self, double x, double y, double z);

/**
 * @brief Scale the vertex for each axis
 * @param[in] x  the scale ratio for x-axis
 * @param[in] y  the scale ratio for y-axis
 * @param[in] z  the scale ratio for z-axis
 */
void mesh_scale(poly_mesh* self, double x, double y, double z);

/**
 * @brief Rotate the vertex
 * @param[in] mtx  3x3 double precision matrix to rotate
 */
void mesh_rotate(poly_mesh *self, double mtx[3][3]);

/**
 * @brief Free the allocated memory for the model
 */
void mesh_free(poly_mesh* self);

__END_DECLS

#endif /* _POLY_MESH_H */
