/**
 * @file      regscan.c
 * @brief     Regular expression wrapper object
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2011 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "regscan.h"

regscan* regscan_create()
{
  regscan* self;
  self = malloc(sizeof(*self));
  if (!self) {
    return NULL;
  }
  memset(self, 0, sizeof(*self));
  return self;
}

void regscan_free(regscan* self)
{
  if (self) {
    regscan_clear(self);
    free(self);
  }
}

void regscan_clear(regscan* self)
{
  if (self->compiled) {
    regfree(&self->re);
  }
  memset(self, 0, sizeof(*self));
}

regscanStatus regscan_compile(regscan* self, const char* pattern, int cflags)
{
  if (self->compiled) {
    regscan_clear(self);
  }
  if (0 != regcomp(&self->re, pattern, cflags)) {
    return REGSCAN_STATUS_COMPILE_ERROR;
  }
  return REGSCAN_STATUS_NO_ERROR;
}

regscanStatus regscan_test(regscan* self, const char* pattern, const char* str, int cflags)
{
  regscanStatus err;
  err = regscan_compile(self, pattern, cflags);
  if (REGSCAN_STATUS_NO_ERROR != err) {
    return err;
  }
  if (0 != regexec(&self->re, str, REGSCAN_NMATCH, self->match, 0)) {
    return REGSCAN_STATUS_NOT_MATCH;
  }
  return REGSCAN_STATUS_MATCH;
}

regscanStatus regscan_exec(regscan* self, const char* str)
{
  self->str = str;
  if (0 == regexec(&self->re, str, REGSCAN_NMATCH, self->match, 0)) {
    self->rest = str + self->match[0].rm_eo;
    return REGSCAN_STATUS_MATCH;
  }
  return REGSCAN_STATUS_NOT_MATCH;
}

regscanStatus regscan_gexec(regscan* self, const char* str)
{
  int eflags = 0;
  if (str == self->str && self->rest) {
    str = self->rest;
    eflags = (REG_NOTBOL|REG_NOTEOL);
  }
  else {
    self->str = str;
  }
  if (0 != regexec(&self->re, str, REGSCAN_NMATCH, self->match, eflags)) {
    return REGSCAN_STATUS_NOT_MATCH;
  }
  self->rest = str + self->match[0].rm_eo;
  return REGSCAN_STATUS_MATCH;
}

const char* regscan_match(regscan* self, int index, int* length)
{
  if (index < 0) {
    return NULL;
  }
  if (REGSCAN_NMATCH <= index) {
    return NULL;
  }
  if (self->match[index].rm_so == -1) {
    return NULL;
  }
  if (length) {
    *length = self->match[index].rm_eo - self->match[index].rm_so + 1;
  }
  return self->str + self->match[index].rm_so;
}

regscanStatus regscan_cpmatch(regscan* self, int index, int nmemb, char* str, int* length)
{
  int len;
  int minlen;
  int err = REGSCAN_STATUS_NO_ERROR;
  if (index < 0) {
    return REGSCAN_STATUS_INDEX_UNDERFLOW;
  }
  if (REGSCAN_NMATCH <= index) {
    return REGSCAN_STATUS_INDEX_OVERFLOW;
  }
  if (self->match[index].rm_so == -1) {
    return REGSCAN_STATUS_NOT_MATCH;
  }
  len = self->match[index].rm_eo - self->match[index].rm_so + 1;
  *length = len;
  if (str && 0 < nmemb) {
    if (nmemb < len) {
      err = REGSCAN_STATUS_BUFFER_TOO_SHORT;
      minlen = nmemb-1;
    }
    else {
      minlen = len-1;
    }
    strncpy(str, self->str + self->match[index].rm_so, minlen);
    str[minlen] = '\0';
  }
  return err;
}
