/**
 * @file      render.c
 * @brief     flow_ig rendering functions
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2011 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <SpiceUsr.h>

#include "sv_doc.h"
#include "flow_common.h"
#include "canvas.h"
#include "tge.h"
#include "render.h"
#include "stl.h"
#include "poly_mesh.h"
#include "starimage.h"

#ifdef HAVE_CFITSIO
#include "canvas_fitsio.h"
#endif

#ifdef HAVE_LIBPNG
#include "canvas_pngio.h"
#endif

#ifdef HAVE_JPEGLIB_H
#include "canvas_jpegio.h"
#endif

#ifdef ENABLE_SPICE_DSK
#include "spice_dsk.h"
#endif

#include "wms.h"
#include "wms_parser.h"
#include "wms_req.h"
#include "wms_res.h"
#include "wms_error.h"
#include "epdb.h"

#define RENDER_OPTION_WIRE           0x01
#define RENDER_OPTION_DISABLE_LIGHT  0x02

#define RENDER_SHAPE_CIRCLE             1
#define RENDER_SHAPE_ELLIPSE            2
#define RENDER_SHAPE_RECTANGLE          3
#define RENDER_SHAPE_POLYGON            4

#define RENDER_FAR                   1.e20
#define RENDER_NEAR                  1.e-6 // unit: millimeter

static int _sort_objects(const void* a, const void* b);
static void set_rotation_to_tgemtx(const SV_OBJECT* obj, TGEdouble* mtx);
static int draw_solar(render* self, SV_OBJECT* obj);
static BOOL render_shape(render* self, shape_model* mdl);
static BOOL draw_sphere_with_texture(render* self, SV_OBJECT* obj, int texidx);
static void render_mesh(render* self, poly_mesh* mesh, int opts);
static BOOL render_fov_mask(render* self);
static BOOL render_fov_mask2(render* self, int shape);
static void render_star(render* self, SV_OBJECT* obj, double mag, uint32_t color);


/**
 * Initialize render
 */
void render_init(render* self,
                 const char* app_name,
                 FLOWIGConfig* config,
                 SV_FRAME* frame,
                 canvas* cv,
                 enum FLOWProjectionMode mode) {
  int i;
  self->app_name = app_name;
  self->config = config;
  self->frame = frame;
  self->canvas = cv;
  self->projection_mode = mode;
  self->error_code = RENDER_NO_ERROR;
  for (i = 0; i < SIZE_STAR_TEX; i++) {
    self->star_tex[i] = starimage_create(i);
  }
}


/**
 * Release the render's internal memory
 */
void render_free(render* self) {
  int i;
  if (self->star_tex) {
    for (i = 0; i < SIZE_STAR_TEX; i++) {
      if (self->star_tex[i]) {
        cv_free(self->star_tex[i]);
      }
      self->star_tex[i] = NULL;
    }
  }
}

FLOWIGConfig* render_config(render* self) {
  return self->config;
}

const char* render_app_name(render* self) {
  return self->app_name;
}

SV_FRAME* render_frame(render* self) {
  return self->frame;
}

canvas* render_canvas(render* self) {
  return self->canvas;
}

canvas* render_star_tex(render* self,
                        int magidx) {
  return self->star_tex[magidx];
}

const char* render_get_message(render* self) {
  static char msg[128];
  switch (self->error_code) {
  case RENDER_NO_ERROR:
    return "no error";
  case RENDER_ILLEGAL_PROJECTION_Z:
    return "instrument's bound-Z must be greater than 0.";
  case RENDER_SHAPE_MODEL_LOAD_ERROR:
    return "can't open Shape model.";
  case RENDER_MEMORY_FULL:
    return "memory full.";
  case RENDER_WMS_CACHE_OPEN_ERROR:
    return "can't open WMS cache.";
  case RENDER_WMS_GET_CAPABILITY_ERROR:
    return "can't get WMS capability.";
  case RENDER_WMS_PARSE_CAPABILITY_ERROR:
    return "error occured while parse WMS capability.";
  case RENDER_WMS_PNG_UNSUPPORT_ERROR:
    return "WMS server does not support PNG image.";
  case RENDER_WMS_LAYER_NOT_FOUND_ERROR:
    return "specified WMS layer does not found.";
  case RENDER_WMS_KNOWN_CRS_NOT_FOUND_ERROR:
    return "known CRS does not found.";
  case RENDER_WMS_GET_IMAGE_ERROR:
    return "can't get image from WMS server.";
  case RENDER_IMAGE_FORMAT_ERROR:
    return "WMS image format error.";
  case RENDER_LOAD_IMAGE_ERROR:
    return "can't load texture.";
  }
  snprintf(msg, sizeof(msg), "unknown error %d\n", self->error_code);
  return msg;
}

/**
 * Render the frame
 */
BOOL render_image(render* self) {
  SV_FRAME* frame = render_frame(self);
  canvas* cv = render_canvas(self);
  int i, max;
  SV_OBJECT* obj;
  int w = cv->width;
  int h = cv->height;
  double* xyz;
  double rect[6];
  double proj_rect[4];
  double scale;
  struct {
    double top;
    double left;
    double bottom;
    double right;
    double z;
  } projection;
  SV_OBJECT** objs = NULL;
  TGEContext* ge = NULL;

  mpc_rotate_frame(frame);

  // Determine the GE projection area
  mpc_get_bounds_rect(frame->view->shape,
                      frame->view->boresight,
                      frame->view->nbounds,
                      frame->view->bounds,
                      rect);
  mpc_get_projection_rect(self->projection_mode,
                          w, h, rect[0], rect[1], rect[3], rect[4], proj_rect);

  if (frame->view->bounds[2] <= 0.0) {
    self->error_code = RENDER_ILLEGAL_PROJECTION_Z;
    goto render_image_error;
  }

  scale = RENDER_NEAR / frame->view->bounds[2];
  projection.left   = proj_rect[0] * scale;
  projection.top    = proj_rect[1] * scale;
  projection.right  = proj_rect[2] * scale;
  projection.bottom = proj_rect[3] * scale;
  projection.z      = RENDER_NEAR;

  objs = malloc(sizeof(SV_OBJECT *)*frame->object_count);
  for (i = 0; i < frame->object_count; i++) {
    objs[i] = frame->objects[i];
  }
  qsort(objs, frame->object_count, sizeof(SV_OBJECT *), _sort_objects);

  // GE Initialization
  ge = tgeCreateContext();
  tgeMakeCurrent(cv, ge);
  tgeMatrixMode(TGE_PROJECTION);
  tgeLoadIdentity();
  tgeFrustum(projection.left, projection.right,
             projection.bottom, projection.top,
             projection.z, RENDER_FAR);
  if (tgeGetError()) {
    abort();
  }
  tgeMatrixMode(TGE_MODELVIEW);
  tgeLoadIdentity();

  tgeEnable(TGE_CULL_FACE);
  tgeEnable(TGE_DEPTH_TEST);
  tgeCullFace(TGE_BACK);
  tgeEnable(TGE_LIGHT0);
  tgeColor4f(1.0, 1.0, 1.0, 1.0);
  tgeClearColor(frame->view->sky_color[0],
                frame->view->sky_color[1],
                frame->view->sky_color[2],
                frame->view->sky_color[3]);
  tgeClear(TGE_COLOR_BUFFER_BIT | TGE_DEPTH_BUFFER_BIT);

  // Drawing
  max = frame->object_count;
  for (i = 0; i < max; i++) {
    obj = objs[i];
    xyz = obj->position;

    if (obj->type == SV_OBJECT_TYPE_SOLAR) {
      // planets
      if (!draw_solar(self, obj)) {
        goto render_image_error;
      }
    }
    else if (obj->type == SV_OBJECT_TYPE_STAR) {
      // planets
      // calculate the coordinate
      render_star(self, obj, obj->magnitude, obj->star.color);
    }
    else {
      // other bodies
      render_star(self, obj, obj->magnitude, 0xffffff);
    }
  }
  free(objs);
  objs = NULL;

  tgeDisable(TGE_LIGHT0);
  tgeDisable(TGE_CULL_FACE);
  tgeDisable(TGE_DEPTH_TEST);
  tgeDestroyContext(ge);
  ge = NULL;

  // apply fov masking
  render_fov_mask(self);

  return TRUE;

render_image_error:
  if (objs) {
    free(objs);
    objs = NULL;
  }
  if (ge) {
    tgeDestroyContext(ge);
    ge = NULL;
  }
  return FALSE;
}

/**
 * Mask the image according to the shape of the camera
 * Judge the shape of the camera and call render_fov_mask2()
 */
static BOOL render_fov_mask(render* self) {
  SV_FRAME* frame = render_frame(self);
  int shape;

  if (0 == strcmp(frame->view->shape, "POLYGON")) {
    shape = RENDER_SHAPE_POLYGON;
  }
  else if (0 == strcmp(frame->view->shape, "RECTANGLE")) {
    shape = RENDER_SHAPE_RECTANGLE;
  }
  else if (0 == strcmp(frame->view->shape, "CIRCLE")) {
    shape = RENDER_SHAPE_CIRCLE;
  }
  else if (0 == strcmp(frame->view->shape, "ELLIPSE")) {
    shape = RENDER_SHAPE_ELLIPSE;
  }
  else {
    fprintf(stderr, "%s: unknown camera's shape %s.\n", render_app_name(self), frame->view->shape);
    return FALSE;
  }

  return render_fov_mask2(self, shape);
}

/**
 * Mask the image according to the shape of the camera
 */
static BOOL render_fov_mask2(render* self,
                             int     shape) {
  SV_FRAME* frame = render_frame(self);
  canvas* cv = render_canvas(self);
  canvas* mask;
  TGEContext* ge;
  TGEdouble vertex[362*2];
  int i;
  TGEint w = cv->width;
  TGEint h = cv->height;
  const TGEint* index;
  TGEsizei nindex;
  double rect[6];
  double proj_rect[4];
  double scale;
  struct {
    double top;
    double left;
    double bottom;
    double right;
    double z;
  } projection;
  double a,b,cx,cy,dx,dy,t;
  double* va;
  double* vb;
  TGEdouble* v;
  TGEItessellator* tess = NULL;

  if (shape == RENDER_SHAPE_POLYGON) {
    tess = tgeiNewTess();
    if (!tess) {
      return FALSE;
    }
  }
  mask = cv_create(w, h, CVDepthGray8);
  if (!mask) {
    fprintf(stderr, "%s: cannot create temporary canvas.\n", render_app_name(self));
    if (tess) tgeiDeleteTess(tess);
    return FALSE;
  }

  // create vertex
  if (shape == RENDER_SHAPE_CIRCLE || shape == RENDER_SHAPE_ELLIPSE) {
    vertex[0] = frame->view->boresight[0];
    vertex[1] = frame->view->boresight[1];
    cx = frame->view->boresight[0];
    cy = frame->view->boresight[1];
    va = frame->view->bounds;
    dx = va[0] - cx;
    dy = va[1] - cy;
    t = atan2(dy, dx);
    a = sqrt(dx*dx+dy*dy);
    if (shape == RENDER_SHAPE_CIRCLE) {
      b = a;
    }
    else {
      vb = frame->view->bounds + 3;
      dx = vb[0] - cx;
      dy = vb[1] - cy;
      b = sqrt(dx*dx+dy*dy);
    }
    for (i = 0; i <= 360; i++) {
      v = &vertex[2+i*2];
      v[0] = cx + a * cos(deg2rad((double)i));
      v[1] = cy + b * sin(deg2rad((double)i));
    }
  }
  else if (shape == RENDER_SHAPE_POLYGON) {
    // Polygon splitting
    tgeiTessVertexPointer(tess, TGE_DOUBLE, sizeof(TGEdouble),
                          frame->view->nbounds, frame->view->bounds);
    index = tgeiTessIndex(tess, &nindex);
  }

  // Determine the GE projection area
  mpc_get_bounds_rect(frame->view->shape,
                      frame->view->boresight,
                      frame->view->nbounds,
                      frame->view->bounds,
                      rect);
  mpc_get_projection_rect(self->projection_mode,
                          w, h, rect[0], rect[1], rect[3], rect[4], proj_rect);

  scale = RENDER_NEAR / frame->view->bounds[2];
  projection.left   = proj_rect[0] * scale;
  projection.top    = proj_rect[1] * scale;
  projection.right  = proj_rect[2] * scale;
  projection.bottom = proj_rect[3] * scale;
  projection.z      = RENDER_NEAR;

  // Polygon splitting

  // Initialize GE
  ge = tgeCreateContext();
  tgeMakeCurrent(mask, ge);
  tgeMatrixMode(TGE_PROJECTION);
  tgeLoadIdentity();
  tgeFrustum(projection.left, projection.right,
             projection.bottom, projection.top,
             projection.z, RENDER_FAR);
  tgeMatrixMode(TGE_MODELVIEW);
  tgeLoadIdentity();
  if (shape == RENDER_SHAPE_ELLIPSE) {
    tgeRotatef(rad2deg(-t), 0.0, 0.0, 1.0);
  }
  tgeDisable(TGE_CULL_FACE);
  tgeDisable(TGE_DEPTH_TEST);
  tgeDisable(TGE_TEXTURE_2D);
  tgeDisable(TGE_LIGHT0);
  tgeClearColor(0.0, 0.0, 0.0, 1.0);
  tgeClear(TGE_COLOR_BUFFER_BIT);
  tgeColor4f(1.0, 1.0, 1.0, 1.0);

  // Draw
  tgeEnableClientState(TGE_VERTEX_ARRAY);
  if (shape == RENDER_SHAPE_CIRCLE || shape == RENDER_SHAPE_ELLIPSE) {
    tgeTranslatef(0.0, 0.0, frame->view->bounds[2]);
    tgeVertexPointer(2, TGE_DOUBLE, 0, vertex);
    tgeDrawArrays(TGE_TRIANGLE_FAN, 0, 362);
  }
  else if (shape == RENDER_SHAPE_POLYGON) {
    tgeVertexPointer(3, TGE_DOUBLE, 0, frame->view->bounds);
    tgeDrawElements(TGE_TRIANGLES, nindex, TGE_INT, index);
  }
  else if (shape == RENDER_SHAPE_RECTANGLE) {
    TGEint rect_index[] = {0, 1, 3, 2};
    tgeVertexPointer(3, TGE_DOUBLE, 0, frame->view->bounds);
    tgeDrawElements(TGE_TRIANGLE_STRIP, 4, TGE_INT, rect_index);
  }
  tgeDestroyContext(ge);

  cv_paste_ex(cv, mask, 0, 0, w, h, CVBlendAnd);
  cv_free(mask);
  if (tess) tgeiDeleteTess(tess);
  return TRUE;
}

/**
 * Create an overhead view
 */
BOOL render_lookdown_image(render* self)
{
  SV_FRAME* frame = render_frame(self);
  canvas* cv = render_canvas(self);
  TGEContext* ge;
  TGEdouble vo[3];
  TGEdouble vsun[3];
  TGEdouble vc[3] = {0, 0, 0};
  TGEdouble va[3];
  TGEdouble dist;
  TGEdouble distnear = 0;
  int nearidx = 0;
  TGEdouble scale;
  SV_OBJECT* obj;
  int i;
  TGEdouble vertex_craft[] = {
    0.0, -1.0, 0.0, 1.0,
    -1.0,  0.0, 1.0, 0.0
  };
  TGEdouble vertex[6];
  TGEdouble vertex_circle[36*2];
  TGEdouble r1;
  TGEdouble r2;
  TGEdouble* b;
  TGEfloat flv[3];
  TGEdouble mtx[16];

  for (i = 0; i < 36; i++) {
    vertex_circle[i*2]   = cos(deg2rad((double)i));
    vertex_circle[i*2+1] = sin(deg2rad((double)i));
  }

  for (i = 0; i < 3; i++) vsun[i] = -frame->view->location[i];
  for (i = 0; i < frame->object_count; i++) {
    obj = frame->objects[i];
    if (obj->type != SV_OBJECT_TYPE_SOLAR) {
      continue;
    }
    tgeVectorAddd(3, vc, obj->position, vo);
    dist = tgeVectorNormd(3, vo, va);
    if (dist < obj->solar.radius[0]) {
      dist = obj->solar.radius[0];
    }
    if (i == 0 || dist < distnear) {
      distnear = dist;
      nearidx = i;
    }
  }
  scale = distnear / 350 * 2;

  b = frame->view->boresight;
  obj = frame->objects[nearidx];

  recsph_c(obj->position, &dist, &r1, &r2);
  r2 = deg2rad(r2);

  ge = tgeCreateContext();
  tgeMakeCurrent(cv, ge);
  tgeEnable(TGE_LIGHT0);
  tgeEnable(TGE_DEPTH_TEST);
  tgeEnable(TGE_CULL_FACE);
  tgeEnable(TGE_TEXTURE_2D);
  tgeCullFace(TGE_BACK);
  tgeMatrixMode(TGE_PROJECTION);
  tgeLoadIdentity();
  tgeOrtho(-400, 400, 400, -400, -10000, 10000);
  tgeMatrixMode(TGE_MODELVIEW);
  tgeLoadIdentity();
  tgeTranslatef(0, 0, 0);

  tgeClearColor(0.2, 0.2, 0.2, 1.0);
  tgeClear(TGE_COLOR_BUFFER_BIT|TGE_DEPTH_BUFFER_BIT);

  tgeEnable(TGE_LIGHT0);
  tgeVectorNormd(3, vsun, vsun);
  for (i = 0; i < 3; i++) flv[i] = vsun[i];
  tgeLightfv(TGE_LIGHT0, TGE_POSITION, flv);

  // target OBJECT
  tgeColor4f(1.0, 1.0, 1.0, 1.0);
  tgeVectorAddd(3, vc, obj->position, vo);
  tgeVectorDivd(3, vo, scale, vo);
  set_rotation_to_tgemtx(obj, mtx);
  tgePushMatrix();
  tgeMultMatrixd(mtx);
  tgeLightfv(TGE_LIGHT0, TGE_POSITION, flv);
  tgeScalef(obj->solar.radius[0]/scale,
            obj->solar.radius[1]/scale,
            obj->solar.radius[2]/scale);
  tgeTranslatef(vo[0], vo[1], vo[2]);
  //tgeRotatef(r1, 1.0, 0.0, 0.0);
  //tgeRotatef(r2, 0.0, 1.0, 0.0);
  tgeRotatef(90, 0.0, 1.0, 0.0);
  for (i = 0; i < obj->texture_count; i++) {
    draw_sphere_with_texture(self, obj, i);
  }
  tgePopMatrix();

  tgeDisable(TGE_LIGHT0);
  tgeDisable(TGE_DEPTH_TEST);

  tgeLightfv(TGE_LIGHT0, TGE_POSITION, flv);

  // SUN
  tgeColor4f(1.0, 1.0, 0.0, 1.0);
  tgeVectorNormd(3, vsun, vsun);
  for (i = 0; i < 6; i++) vertex[i] = 0.0;
  for (i = 0; i < 3; i++) vertex[i+3] = -vsun[i];
  tgePushMatrix();
  tgeScalef(100.0, 100.0, 100.0);
  //tgeRotatef(r1, 1.0, 0.0, 0.0);
  //tgeRotatef(r2, 0.0, 1.0, 0.0);
  tgeRotatef(90, 0.0, 1.0, 0.0);
  tgeEnableClientState(TGE_VERTEX_ARRAY);
  tgeVertexPointer(3, TGE_DOUBLE, 0, vertex);
  tgeDrawArrays(TGE_LINES, 0, 2);
  tgeDisableClientState(TGE_VERTEX_ARRAY);
  tgePopMatrix();


  // Spacecraft
  tgeColor4f(1.0, 0.0, 0.0, 1.0);
  tgePushMatrix();
  tgeScalef(5.0, 5.0, 1.0);
  tgeEnableClientState(TGE_VERTEX_ARRAY);
  tgeVertexPointer(2, TGE_DOUBLE, 0, vertex_craft);
  tgeDrawArrays(TGE_LINES, 0, 4);
  tgeDisableClientState(TGE_VERTEX_ARRAY);
  tgePopMatrix();

  // camera
  tgeColor4f(1.0, 1.0, 1.0, 1.0);
  tgePushMatrix();
  for (i = 0; i < 6; i++) vertex[i] = 0.0;
  tgeVectorNormd(3, frame->view->boresight, va);
  dist = tgeVectorDistd(3, obj->position);
  dist -= (obj->solar.radius[0] + obj->solar.radius[1] + obj->solar.radius[2]) / 3;
  dist = dist / scale;
  for (i = 0; i < 3; i++) vertex[i+3] = va[i];
  tgeScalef(dist, dist, dist);
  //tgeRotatef(r1, 1.0, 0.0, 0.0);
  //tgeRotatef(r2, 0.0, 1.0, 0.0);
  tgeRotatef(90, 0.0, 1.0, 0.0);
  tgeEnableClientState(TGE_VERTEX_ARRAY);
  tgeVertexPointer(3, TGE_DOUBLE, 0, vertex);
  tgeDrawArrays(TGE_LINES, 0, 2);
  tgeDisableClientState(TGE_VERTEX_ARRAY);
  tgePopMatrix();

  tgeDestroyContext(ge);
  return TRUE;
}

/**
 * Load image
 */
canvas* load_image(FLOWIGConfig* config, const char* path)
{
  enum FLOWImageFormat format;
  canvas* cv;
  int texattr = 0;

  format = predict_image_format_from_filename(path);
  if (FLOWImageFormatError == format) {
    return NULL;
  }
  if (format == FLOWImageFormatAuto) {
#ifdef HAVE_CFITSIO
    format = FLOWImageFormatFITS;
#elif defined HAVE_JPEGLIB_H
    format = FLOWImageFormatJPEG;
#elif defined HAVE_LIBPNG
    format = FLOWImageFormatPNG;
#else
    format = FLOWImageFormatUnkown;
#endif
  }
  switch (format) {
#ifdef HAVE_CFITSIO
  case FLOWImageFormatFITS:
    if (!config->flip) {
      texattr |= CVFormatBottomUp;
    }
    cv = cv_load_fits(path, texattr);
    break;
#endif
#ifdef HAVE_JPEGLIB_H
  case FLOWImageFormatJPEG:
    cv = cv_load_jpeg(path, 0);
    break;
#endif
#ifdef HAVE_LIBPNG
  case FLOWImageFormatPNG:
    cv = cv_load_png(path);
    break;
#endif
  default:
    fprintf(stderr, "%s: unknown image format for file %s.\n", app_name, path);
    return NULL;
  }
  if (!cv) {
    fprintf(stderr, "%s: cannot load %s to canvas.\n", app_name, path);
    return NULL;
  }
  return cv;
}


/**
 * Draw solar object
 */
static int draw_solar(render* self, SV_OBJECT* obj)
{
  FLOWIGConfig* config = render_config(self);
  SV_FRAME* frame = render_frame(self);
  TGEdouble* pos = obj->position;
  TGEdouble npixels, maxradi;
  TGEdouble va[3] = {0, 0, pos[2]};
  TGEdouble vb[3] = {0, 0, pos[2]};
  TGEdouble vva[3];
  TGEdouble vvb[3];
  TGEdouble light_posd[3];
  TGEfloat light_posf[3];
  BOOL is_part_map = FALSE;

  vminus_c(frame->view->location, light_posd);
  light_posf[0] = light_posd[0];
  light_posf[1] = light_posd[1];
  light_posf[2] = light_posd[2];

  maxradi = max(obj->solar.radius[0],
                max(obj->solar.radius[1],
                    obj->solar.radius[2]));
  va[0] = -maxradi;
  vb[0] = maxradi;
  tgePushMatrix();
  tgeScalef(maxradi, maxradi, maxradi);
  tgeTranslatef(pos[0], pos[1], pos[2]);
  tgeiVertexToViewport(3, TGE_DOUBLE, va, vva);
  tgeiVertexToViewport(3, TGE_DOUBLE, vb, vvb);
  tgePopMatrix();
  npixels = vvb[0] - vva[0];

  if (1.0 <= npixels) {
    TGEdouble mtx[16];
    if (obj->model) {
      // Draw Shape
      shape_model* pm = shape_create();
      TGEubyte color[3];
      if (obj->model->format == NULL) {
        fprintf(stderr,
                "%s: XML must have the following entry for `%s':\n"
                "  <object><model><format>STK</format></model></object>.\n",
                app_name, obj->name);
        self->error_code = RENDER_SHAPE_MODEL_LOAD_ERROR;
        return FALSE;
      }
      if (strncmp(obj->model->format, "STL", 3) == 0) {
        if (!stl_parse_file(pm, obj->model->src)) {
          self->error_code = RENDER_SHAPE_MODEL_LOAD_ERROR;
          return FALSE;
        }
      }
#ifdef ENABLE_SPICE_DSK
      else if (strncmp(obj->model->format, "DSK", 3) == 0) {
        int code;
        if (!dsk_type02_load(pm, obj->model->src, &code)) {
          self->error_code = RENDER_SHAPE_MODEL_LOAD_ERROR;
          return FALSE;
        }
      }
#endif
      else {
        fprintf(stderr, "%s: unknown shape mode format %s.\n",
                app_name, obj->model->format);
        if (strncmp(obj->model->format, "DSK", 3) == 0) {
          fprintf(stderr, "Compile with --enable-spice-dsk if you use DSK\n");
        }
        fprintf(stderr, "Only 'STK' or 'DSK' is supported.\n");
      }
      color[0] = (obj->model->color>>16)&0xFF;
      color[1] = (obj->model->color>>8)&0xFF;
      color[2] = obj->model->color&0xFF;
      tgeColor4f(color[0]/255.0, color[1]/255.0, color[2]/255.0, 1.0);
      tgeDisable(TGE_TEXTURE_2D);
      tgeEnable(TGE_DEPTH_TEST);
      tgePushMatrix();
      tgeScalef(obj->model->scale,
                obj->model->scale,
                obj->model->scale);
      set_rotation_to_tgemtx(obj, mtx);
      tgeInvertMatrixd(mtx, mtx);
      tgeMultMatrixd(mtx);
      tgeTranslatef(pos[0], pos[1], pos[2]);
      tgeLightfv(TGE_LIGHT0, TGE_POSITION, light_posf);
      render_shape(self, pm);
      tgePopMatrix();
      shape_free(pm);
    }
    else {
      // Draw a sphere
      // Calculate the division number of mesh
      double vappx, vappy, vapp;
      int sep;
      is_part_map = FALSE;
      vappx  = atan2(obj->solar.radius[0], obj->distance - pos[2]);
      vappy  = atan2(obj->solar.radius[0], obj->distance - pos[2]);
      vapp   = max(vappx, vappy) * 180.0 / pi_c();
      if (vapp < 45) {
        sep = 120;
      } else if (vapp < 60) {
        sep = 150;
      } else if (vapp < 75) {
        sep = 180;
      } else {
        sep = 240;
      }
      if (obj->texture_count < 1) {
        fprintf(stderr, "%s: no texture node found for %s.\n", app_name, obj->name);
      } else {
        int i;
        tgeEnable(TGE_TEXTURE_2D);
        tgePushMatrix();
        tgeScalef(obj->solar.radius[0], obj->solar.radius[1], obj->solar.radius[2]);
        set_rotation_to_tgemtx(obj, mtx);
        tgeInvertMatrixd(mtx, mtx);
        tgeMultMatrixd(mtx);
        tgeTranslatef(pos[0], pos[1], pos[2]);
        tgeLightfv(TGE_LIGHT0, TGE_POSITION, light_posf);

        for (i = 0; i < obj->texture_count; i++) {
          draw_sphere_with_texture(self, obj, i);
        }
        tgePopMatrix();
      }
    }
  }
  else {
    // Draw a point
    render_star(self, obj, obj->magnitude, 0xFFFFFF);
  }

  return TRUE;
}

/**
 * Callback function to sort in order of disances of
 * bodies (far to near)
 */
static int _sort_objects(const void* a, const void* b)
{
  const SV_OBJECT* obja = *(SV_OBJECT * *)a;
  const SV_OBJECT* objb = *(SV_OBJECT * *)b;
  if (obja->distance < objb->distance) {
    return -1;
  }
  else if (obja->distance > objb->distance) {
    return 1;
  }
  return 0;
}

/**
 * Render a polygonal object
 */
static BOOL render_shape(render* self, shape_model* mdl)
{
  FLOWIGConfig* config = render_config(self);
  size_t i;
  size_t len;
  const shapeTriangle* tri;
  float* normal;
  float* vertex;
  float* fp;
  TGEdouble vec[3];

  len = shape_size(mdl);
  if (!len) {
    fprintf(stderr, "%s: STL triangle length is 0.\n", app_name);
    return FALSE;
  }

  normal = malloc(sizeof(float) * 3 * 3 * len);
  vertex = malloc(sizeof(float) * 3 * 3 * len);

  // extract Shape to OpenGL array
  for (i = 0; i < len; i++) {
    tri = shape_triangle_at(mdl, i);
    fp = normal + i*9;
    vec[0] = tri->normal.x;
    vec[1] = tri->normal.y;
    vec[2] = tri->normal.z;
    tgeVectorNormd(3, vec, vec);
    fp[0] = vec[0]; fp[1] = vec[1]; fp[2] = vec[2];
    fp[3] = vec[0]; fp[4] = vec[1]; fp[5] = vec[2];
    fp[6] = vec[0]; fp[7] = vec[1]; fp[8] = vec[2];
    fp = vertex + i*9;
    fp[0] = tri->va.x; fp[1] = tri->va.y; fp[2] = tri->va.z;
    fp[3] = tri->vb.x; fp[4] = tri->vb.y; fp[5] = tri->vb.z;
    fp[6] = tri->vc.x; fp[7] = tri->vc.y; fp[8] = tri->vc.z;
  }

  if (config->ge_disable_light) {
    tgeDisable(TGE_LIGHT0);
  }
  tgeEnableClientState(TGE_VERTEX_ARRAY);
  tgeEnableClientState(TGE_NORMAL_ARRAY);
  tgeVertexPointer(3, TGE_FLOAT, 0, vertex);
  tgeNormalPointer(TGE_FLOAT, 0, normal);
  if (config->ge_wire) {
    for (i = 0; i < len; i++) {
      tgeDrawArrays(TGE_LINE_LOOP, i*3, 3);
    }
  }
  else {
    tgeDrawArrays(TGE_TRIANGLES, 0, len*3);
  }
  tgeDisableClientState(TGE_VERTEX_ARRAY);
  tgeDisableClientState(TGE_NORMAL_ARRAY);

  free(vertex);
  free(normal);
  return TRUE;
}

#ifdef ENABLE_WMS

/**
 * Draw a part of the object with texture
 * @param self   render object.
 * @param obj    solar object.
 * @param tex    texture.
 * @param region drawing area {minlng, minlat, maxlng, maxlat}
 * @return TRUE: success, FALSE: error.
 */
static BOOL render_draw_wms(render* self, SV_TEXTURE* tinfo)
{
  FLOWIGConfig* config = render_config(self);
  int drawopt = 0;
  wmsRegion* regions = NULL;
  int nregion;
  int n, x, y, nx, ny, w, h;
  double d, d2, ud;
  wmsRegion all_region;
  double maxlng;
  wmsRegion lregion;
  const char* name;
  const char* format;
  enum WMSCrs crs;
  wms* wm = NULL;
  wmsReq* req = NULL;
  wmsLayer* layer = NULL;
  wmsRes* res = NULL;
  canvas* cv = NULL;
  poly_mesh* mesh;
  epdb* db = NULL;
  int use_cache = 0;
  char* cache_key = NULL;
  const char* url;
  wmsReq* req_c = NULL;
  wmsRes* res_c = NULL;

  // open cache
  if (!config->wms_no_cache) {
    if (0 == mpc_create_config_dir()) {
      char* cache_path = NULL;
      size_t nlen;
      epdb_config conf;
      cache_path = mpc_get_config_path(FLOW_WMS_CACHE_FILE, NULL, 0, &nlen);
      if (!cache_path) {
        self->error_code = RENDER_MEMORY_FULL;
        goto render_draw_wms_error;
      }
      epdb_config_set_default(&conf);
      db = epdb_open(cache_path, &conf);
      free(cache_path);
      if (!db) {
        self->error_code = RENDER_WMS_CACHE_OPEN_ERROR;
        goto render_draw_wms_error;
      }
    }
    else {
      fprintf(stderr, "%s: WMS cache ignored: can't create config directory\n",
              render_app_name(self));
    }
  }

  if (config->ge_wire) {
    drawopt |= RENDER_OPTION_WIRE;
  }
  if (config->ge_disable_light) {
    drawopt |= RENDER_OPTION_DISABLE_LIGHT;
  }

  all_region.min.lng = mpc_degnorm(tinfo->region[0], 180);
  if (all_region.min.lng == 180) all_region.min.lng = -180.0;
  all_region.min.lat = tinfo->region[1];
  all_region.max.lng = mpc_degnorm(tinfo->region[2], 180);
  all_region.max.lat = tinfo->region[3];
  maxlng = (all_region.max.lng < all_region.min.lng)
           ? all_region.max.lng + 360.0 : all_region.max.lng;

  // Decide the number of divisions
  ud = 500.0 / tinfo->resolution;
  nx = 0;
  for (d = all_region.min.lng; d < maxlng; ) {
    d = (d < 180.0 && 180.0 < d + ud) ? 180.0 : d + ud;
    nx++;
  }
  ny = 0;
  for (d = all_region.min.lat; d < all_region.max.lat; ) {
    d += ud;
    ny++;
  }
  nregion = nx * ny;

  // Create region
  regions = malloc(sizeof(*regions)*nregion);
  x = 0;
  for (d = all_region.min.lng; d < maxlng; ) {
    d2 = (d < 180.0 && 180.0 < d + ud) ? 180.0 : d + ud;
    if (180.0 <= d && 180.0 <= d2) {
      for (y = 0; y < ny; y++) {
        regions[y*nx+x].min.lng = d-360.0;
        regions[y*nx+x].max.lng = d2-360.0;
      }
    }
    else {
      for (y = 0; y < ny; y++) {
        regions[y*nx+x].min.lng = d;
        regions[y*nx+x].max.lng = d2;
      }
    }
    d = d2;
    x++;
  }
  y = 0;
  for (d = all_region.min.lat; d < all_region.max.lat; ) {
    d2 = (all_region.max.lat < d + ud) ? all_region.max.lat : d + ud;
    for (x = 0; x < nx; x++) {
      regions[y*nx+x].min.lat = d;
      regions[y*nx+x].max.lat = d2;
    }
    d = d2;
    y++;
  }

  // Fetch WMS capability
  wm = wms_create();
  use_cache = 0;
  req_c = wmsReq_create(wm);
  wmsReq_set_capability_url(req_c, tinfo->src);
  if (db) {
    epdb_data* value = NULL;
    epdb_status err;
    url = wmsReq_url(req_c);
    cache_key = malloc(strlen(url) + 5);
    strcpy(cache_key, "URL:");
    strcat(cache_key, url);
    err = epdb_getsv(db, cache_key, &value);
    if (EPDB_NO_ERROR == err) {
      if (wms_read_from_buffer(wm, url, epdb_data_get(value),
                               epdb_data_size(value))) {
        //fprintf(stderr, "cached: %s\n", url);
        use_cache = 1;
      }
      epdb_data_release(value);
    }
  }
  if (!use_cache) {
    res_c = wmsReq_fetch(req_c);
    if (!res_c) {
      self->error_code = RENDER_WMS_GET_CAPABILITY_ERROR;
      goto render_draw_wms_error;
    }
    if (!wms_read_from_buffer(wm, tinfo->src, wmsRes_data(res_c),
                              wmsRes_data_size(res_c))) {
      self->error_code = RENDER_WMS_PARSE_CAPABILITY_ERROR;
      fprintf(stderr, "%s: %s(): can't get capability (illegal format).\n",
              __func__, render_app_name(self));
      fprintf(stderr, "%s\n", wmsError_message(wm->error));
      goto render_draw_wms_error;
    }
    if (db) {
      epdb_data* value = epdb_data_create(wmsRes_data_size(res_c), wmsRes_data(res_c),
                                          EPDB_NO_COPY|EPDB_READ_ONLY);
      epdb_setsv(db, cache_key, value);
      epdb_data_release(value);
    }
    wmsRes_free(res_c);
    res_c = NULL;
  }
  wmsReq_free(req_c);
  req_c = NULL;
  if (cache_key) {
    free(cache_key);
    cache_key = NULL;
  }

  // check format
  if (wms_has_format(wm, "image/png")) {
    format = "image/png";
  }
  else {
    self->error_code = RENDER_WMS_PNG_UNSUPPORT_ERROR;
    fprintf(stderr, "%s: %s(): server does not support PNG format.\n",
            render_app_name(self), __func__);
    goto render_draw_wms_error;
  }
  // check layer
  layer = wms_find_layer(wm, WMSFindTypeTitle, tinfo->title);
  if (!layer) {
    self->error_code = RENDER_WMS_LAYER_NOT_FOUND_ERROR;
    fprintf(stderr, "%s: %s(): titled layer '%s' does not found.\n",
            render_app_name(self), __func__, tinfo->title);
    goto render_draw_wms_error;
  }
  lregion = wmsLayer_region(layer);
  name = wmsLayer_name(layer);

  // check CRS
  if (wmsLayer_has_crs(layer, "CRS:83")) {
    crs = WMSCrsCRS83;
  }
  else if (wmsLayer_has_crs(layer, "EPSG:4326")) {
    crs = WMSCrsEPSG4326;
  }
  else {
    self->error_code = RENDER_WMS_KNOWN_CRS_NOT_FOUND_ERROR;
    fprintf(stderr, "%s: %s(): server does not support known CRS.\n",
            render_app_name(self), __func__);
    goto render_draw_wms_error;
  }

  // Obtain partial images from WMS
  req = wmsReq_create(wm);
  for (n = 0; n < nregion; n++) {
    w = (int)floor(500 * (regions[n].max.lng - regions[n].min.lng) / ud);
    h = (int)floor(500 * (regions[n].max.lat - regions[n].min.lat) / ud);
    wmsReq_set_map_url(req, name, format, crs, regions[n], w, h);
    //fprintf(stderr, "URL: %s\n", wmsReq_url(req));

    // get cache
    use_cache = 0;
    if (db) {
      epdb_data* value = NULL;
      epdb_status err;
      url = wmsReq_url(req);
      if (cache_key) free(cache_key);
      cache_key = malloc(strlen(url) + 5);
      if (!cache_key) {
        self->error_code = RENDER_MEMORY_FULL;
        goto render_draw_wms_error;
      }
      strcpy(cache_key, "URL:");
      strcat(cache_key, url);
      err = epdb_getsv(db, cache_key, &value);
      if (err == EPDB_NO_ERROR) {
        cv = cv_load_png_with_bytes(epdb_data_get(value), epdb_data_size(value));
        epdb_data_release(value);
        if (cv) {
          use_cache = 1;
          //fprintf(stderr, "cached: %s\n", url);
        }
        else {
          epdb_deletes(db, cache_key);
        }
      }
    }
    if (!use_cache) {
      res = wmsReq_fetch(req);
      if (!res) {
        self->error_code = RENDER_WMS_GET_IMAGE_ERROR;
        fprintf(stderr, "%s: %s(): can't get image from WMS.\n",
                render_app_name(self), __func__);
        goto render_draw_wms_error;
      }
      cv = cv_load_png_with_bytes(wmsRes_data(res), wmsRes_data_size(res));
      if (!cv) {
        self->error_code = RENDER_IMAGE_FORMAT_ERROR;
        goto render_draw_wms_error;
      }
      // set cache
      if (db) {
        epdb_data* value = epdb_data_create(wmsRes_data_size(res),
                                            wmsRes_data(res),
                                            (EPDB_NO_COPY|EPDB_READ_ONLY));
        epdb_setsv(db, cache_key, value);
        epdb_data_release(value);
      }
    }

    tgeTexImage2D(cv);
    mesh = mesh_alloc_sphere_part(20, 20,
                                  regions[n].min.lat, regions[n].min.lng,
                                  regions[n].max.lat, regions[n].max.lng);
    // #193 depth test
    tgeDepthMask(TGE_FALSE);
    tgeEnable(TGE_POLYGON_OFFSET_FILL);
    tgeEnable(TGE_POLYGON_OFFSET_LINE);
    tgePolygonOffset(0.98, 0);
    {
      render_mesh(self, mesh, drawopt);
    }
    tgeDepthMask(TGE_TRUE);
    tgeDisable(TGE_POLYGON_OFFSET_FILL);
    tgeDisable(TGE_POLYGON_OFFSET_LINE);
    tgePolygonOffset(0, 0);
    mesh_free(mesh);
    tgeTexImage2D(NULL);

    if (res) wmsRes_free(res);
    res = NULL;
    cv_free(cv);
    cv = NULL;
  }
  if (db) {
    epdb_close(db);
    db = NULL;
  }
  wmsReq_free(req);
  req = NULL;
  wms_free(wm);
  wm = NULL;
  free(regions);
  regions = NULL;

  return TRUE;

render_draw_wms_error:
  if (cache_key) free(cache_key);
  if (db) epdb_close(db);
  if (regions) free(regions);
  if (cv) cv_free(cv);
  if (req_c) wmsReq_free(req_c);
  if (res_c) wmsRes_free(res_c);
  if (res) wmsRes_free(res);
  if (req) wmsReq_free(req);
  if (wm) wms_free(wm);
  return FALSE;
}
#endif // ENABLE_WMS

/**
 * Draw an body with texture
 */
static BOOL draw_sphere_with_texture(render* self, SV_OBJECT* obj, int texidx)
{
  FLOWIGConfig* config = render_config(self);
  canvas* tex = NULL;
  BOOL is_part_map = FALSE;
  poly_mesh* mesh = NULL;
  int sep = 240;
  int drawopt = 0;
  int i;
  BOOL err;
  static TGEdouble ring_vertex[] = {
    -1.0, -1.0, 0.0,
    -1.0, 1.0, 0.0,
    1.0,  -1.0, 0.0,
    1.0,  1.0, 0.0,
  };
  static TGEdouble ring_texcoord[] = {
    0.0, 0.0,
    0.0, 1.0,
    1.0, 0.0,
    1.0, 1.0,
  };

  if (config->ge_wire) {
    drawopt |= RENDER_OPTION_WIRE;
  }
  if (config->ge_disable_light) {
    drawopt |= RENDER_OPTION_DISABLE_LIGHT;
  }
#if 1
  if (0 == strcmp("SUN",obj->name)) {
    drawopt |= RENDER_OPTION_DISABLE_LIGHT;
  }
#endif
  if (obj->textures[texidx]->type == SV_TEXTURE_TYPE_FILE) {
    // Select output image format
    tex = load_image(config, obj->textures[texidx]->src);
    if (!tex) {
      self->error_code = RENDER_LOAD_IMAGE_ERROR;
      return FALSE;
    }
    if (obj->textures[texidx]->region_string) {
      is_part_map = TRUE;
    }
    else {
      is_part_map = FALSE;
    }
  }

#ifdef ENABLE_WMS
  if (obj->textures[texidx]->type == SV_TEXTURE_TYPE_WMS) {
    err = render_draw_wms(self, obj->textures[texidx]);
    if (!err && config->wms_exit_on_error) {
      fprintf(stderr, "%s: error occured while get image from WMS server.\n",
              render_app_name(self));
      exit(1);
    }
    return err;
  }
#endif // ENABLE_WMS

  tgeTexImage2D(tex);
  if (SV_TEXTURE_COMPONENT_BODY == obj->textures[texidx]->component) {
    if (!is_part_map) {
      mesh = mesh_alloc_sphere(sep, sep/2);
      render_mesh(self, mesh, drawopt);
      mesh_free(mesh);
    }
    else {
      mesh = mesh_alloc_sphere_part(20, 20,
                                    obj->textures[texidx]->region[1],
                                    obj->textures[texidx]->region[0],
                                    obj->textures[texidx]->region[3],
                                    obj->textures[texidx]->region[2]);
      // #193 depth test
      tgeDepthMask(TGE_FALSE);
      tgeEnable(TGE_POLYGON_OFFSET_FILL);
      tgeEnable(TGE_POLYGON_OFFSET_LINE);
      tgePolygonOffset(0.98, 0);
      {
        render_mesh(self, mesh, drawopt);
      }
      tgeDepthMask(TGE_TRUE);
      tgeDisable(TGE_POLYGON_OFFSET_FILL);
      tgeDisable(TGE_POLYGON_OFFSET_LINE);
      tgePolygonOffset(0, 0);
      mesh_free(mesh);
    }
  }
  else if (SV_TEXTURE_COMPONENT_RING == obj->textures[texidx]->component) {
    // #206 Draw ring
    TGEdouble vertex[12];
    for (i = 0; i < 12; i++) {
      vertex[i] = ring_vertex[i] * obj->textures[texidx]->ring_radius;
    }
    tgeDisable(TGE_LIGHT0);
    tgeDisable(TGE_CULL_FACE);
    tgeEnable(TGE_BLEND);
    tgeBlendFunc(TGE_SRC_ALPHA, TGE_ONE_MINUS_SRC_ALPHA);
    tgeColor4f(1.0, 1.0, 1.0, 1.0);
    tgeEnableClientState(TGE_VERTEX_ARRAY);
    tgeEnableClientState(TGE_TEXTURE_COORD_ARRAY);
    tgeVertexPointer(3, TGE_DOUBLE, 0, vertex);
    tgeTexCoordPointer(2, TGE_DOUBLE, 0, ring_texcoord);
    if (drawopt & RENDER_OPTION_WIRE) {
      tgeDrawArrays(TGE_LINE_LOOP, 0, 3);
      tgeDrawArrays(TGE_LINE_LOOP, 1, 3);
    }
    else {
      tgeDrawArrays(TGE_TRIANGLE_STRIP, 0, 4);
    }
    tgeDisableClientState(TGE_VERTEX_ARRAY);
    tgeDisableClientState(TGE_TEXTURE_COORD_ARRAY);
    tgeDisable(TGE_BLEND);
    tgeEnable(TGE_CULL_FACE);
    tgeEnable(TGE_LIGHT0);
  }
  if (tex) {
    cv_free(tex);
  }
  return TRUE;
}

/**
 * Convert object rotation to matrix for TGE
 * @param obj  (in) object
 * @param mtx  (out) 4x4 matrix
 */
static void set_rotation_to_tgemtx(const SV_OBJECT* obj, TGEdouble* mtx)
{
  mtx[0]  = obj->solar.rotation[0][0];
  mtx[1]  = obj->solar.rotation[0][1];
  mtx[2]  = obj->solar.rotation[0][2];
  mtx[3]  = 0.0;
  mtx[4]  = obj->solar.rotation[1][0];
  mtx[5]  = obj->solar.rotation[1][1];
  mtx[6]  = obj->solar.rotation[1][2];
  mtx[7]  = 0.0;
  mtx[8]  = obj->solar.rotation[2][0];
  mtx[9]  = obj->solar.rotation[2][1];
  mtx[10] = obj->solar.rotation[2][2];
  mtx[11] = 0.0;
  mtx[12] = 0.0;
  mtx[13] = 0.0;
  mtx[14] = 0.0;
  mtx[15] = 1.0;
}

/**
 * Draw a mesh object
 */
static void render_mesh(render* self, poly_mesh* mesh, int opts)
{
  int i;
  int off;
  tgeEnableClientState(TGE_VERTEX_ARRAY);
  tgeEnableClientState(TGE_TEXTURE_COORD_ARRAY);
  tgeEnable(TGE_AUTO_NORMAL);
  tgeEnable(TGE_LIGHT0);

  if (opts & RENDER_OPTION_DISABLE_LIGHT) {
    tgeDisable(TGE_LIGHT0);
  }

  tgeVertexPointer(3, TGE_DOUBLE, 0, mesh->vertex);
  tgeTexCoordPointer(2, TGE_DOUBLE, 0, mesh->texcoord);
  if (opts & RENDER_OPTION_WIRE) {
    for (i = 0; i < mesh->polygon_size; i++) {
      off = i * 3;
      tgeDrawElements(TGE_LINE_LOOP, 3, TGE_UNSIGNED_INT, mesh->index + off);
    }
  }
  else {
    tgeDrawElements(TGE_TRIANGLES, mesh->polygon_size*3, TGE_UNSIGNED_INT, mesh->index);
  }

  tgeDisable(TGE_AUTO_NORMAL);
  tgeDisable(TGE_LIGHT0);
  tgeDisableClientState(TGE_VERTEX_ARRAY);
  tgeDisableClientState(TGE_TEXTURE_COORD_ARRAY);
}

/**
 * Draw a star image
 */
static void render_star(render* self, SV_OBJECT* obj, double mag, uint32_t color)
{
  TGEdouble* pos = obj->position;
  TGEdouble spos[3];
  canvas* tex = NULL;
  canvas* cv = render_canvas(self);
  int w = cv->width;
  int h = cv->height;
  int idx;
  int sw, sh;
  double vertex[8];
  static double texcoord[] = {
    0.0, 0.0, 0.0, 1.0, 1.0, 0.0, 1.0, 1.0
  };
  TGEfloat c[4];

  tgeVectorMuld(3, pos, obj->distance, spos);
  tgeiVertexToViewport(3, TGE_DOUBLE, spos, spos);

  c[3] = 1.0;
  c[0] = ((color & 0x00FF0000) >> 16) / 255.0;
  c[1] = ((color & 0x0000FF00) >> 8) / 255.0;
  c[2] = (color & 0x000000FF) / 255.0;

  if (mag < 0.0) mag = 0.0;
  if (mag > 7.0) mag = 7.0;
  idx = (SIZE_STAR_TEX - 1) - (int)floor(mag+0.5);
  tex = render_star_tex(self, idx);
  sw = tex->width;
  sh = tex->height;
  vertex[0] = 0.0; vertex[1] = 0.0;
  vertex[2] = 0.0; vertex[3] = sh;
  vertex[4] = sw; vertex[5] = 0.0;
  vertex[6] = sw; vertex[7] = sh;

  // Draw
  //(change matrix since stars are directly drawn on designated coordinates)
  tgeMatrixMode(TGE_MODELVIEW);
  tgePushMatrix();
  tgeMatrixMode(TGE_PROJECTION);
  tgePushMatrix();
  tgeLoadIdentity();
  tgeOrtho(0, w, h, 0, 0, 1.0e20);
  tgeMatrixMode(TGE_MODELVIEW);
  tgeLoadIdentity();
  {
    tgeTexImage2D(tex);
    tgeEnable(TGE_TEXTURE_2D);
    tgeEnable(TGE_DEPTH_TEST);
    tgeEnable(TGE_BLEND);
    tgeDisable(TGE_LIGHT0);
    tgeDisable(TGE_CULL_FACE);
    tgeBlendFunc(TGE_SRC_ALPHA, TGE_ONE_MINUS_SRC_ALPHA);
    {
      tgeColor4f(c[0], c[1], c[2], c[3]);
      tgeTranslatef(-sw/2, -sh/2, 0.0);
      tgeTranslatef(floor(spos[0]), floor(spos[1]), spos[2]);
      tgeEnableClientState(TGE_VERTEX_ARRAY);
      tgeEnableClientState(TGE_TEXTURE_COORD_ARRAY);
      tgeVertexPointer(2, TGE_DOUBLE, 0, vertex);
      tgeTexCoordPointer(2, TGE_DOUBLE, 0, texcoord);
      tgeDrawArrays(TGE_TRIANGLE_STRIP, 0, 4);
      tgeDisableClientState(TGE_VERTEX_ARRAY);
      tgeDisableClientState(TGE_TEXTURE_COORD_ARRAY);
    }
    tgeDisable(TGE_TEXTURE_2D);
    tgeDisable(TGE_BLEND);
  }
  tgePopMatrix();
  tgeMatrixMode(TGE_PROJECTION);
  tgePopMatrix();
  tgeMatrixMode(TGE_MODELVIEW);
}
