/**
 * @file      render.h
 * @brief     flow_ig rendering functions
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2011 Japan Aerospace Exploration Agency
 *
 */

#ifndef _RENDER_H_
#define _RENDER_H_

#include <stdio.h>
#include "flowig_config.h"
#include "sv_doc.h"
#include "canvas.h"

#define RENDER_NO_ERROR                       0
#define RENDER_ILLEGAL_PROJECTION_Z           1
#define RENDER_SHAPE_MODEL_LOAD_ERROR         2
#define RENDER_MEMORY_FULL                    3
#define RENDER_WMS_CACHE_OPEN_ERROR           4
#define RENDER_WMS_GET_CAPABILITY_ERROR       5
#define RENDER_WMS_PARSE_CAPABILITY_ERROR     6
#define RENDER_WMS_PNG_UNSUPPORT_ERROR        7
#define RENDER_WMS_LAYER_NOT_FOUND_ERROR      8
#define RENDER_WMS_KNOWN_CRS_NOT_FOUND_ERROR  9
#define RENDER_WMS_GET_IMAGE_ERROR           10
#define RENDER_IMAGE_FORMAT_ERROR            11
#define RENDER_LOAD_IMAGE_ERROR              12

#define SIZE_STAR_TEX 8

/**
 * @struct _render
 * @brief Main structure for render
 */
typedef struct _render {
  /** Application name */
  const char* app_name;

  /** Application configuration */
  FLOWIGConfig* config;

  /** Rendering frame */
  SV_FRAME* frame;

  /** Canvas to render */
  canvas* canvas;

  /** Taxonomy of stars */
  canvas* star_tex[SIZE_STAR_TEX];

  /** Projection mode */
  enum FLOWProjectionMode projection_mode;

  /** Error code */
  int error_code;
} render;

__BEGIN_DECLS
/**
 * @brief Initialize the render structure
 * @param[in] self      pointer of the render structure
 * @param[in] app_name  application name
 * @param[in] config    application configuration
 * @param[in] frame     frame data
 * @param[in] cv        canvas
 * @param[in] mode      projection mode
 */
void render_init(render* self, const char* app_name, FLOWIGConfig* config, SV_FRAME* frame, canvas* cv, enum FLOWProjectionMode mode);

/**
 * @brief Free the memory used by the render structure
 * @param[in] self  pointer of the render structure
 */
void render_free(render* self);

/**
 * @brief Return the application configuration
 * @param[in] self  pointer of the render structure
 * @return render_config() returns the pointer of the application configuration.
 */
FLOWIGConfig* render_config(render* self);

/**
 * @brief Return the application name
 * @param[in] self  pointer of the render structure
 * @return render_app_name() returns the application name.
 */
const char* render_app_name(render* self);

/**
 * @brief Return the rendering frame
 * @param[in] self  pointer of the render structure
 * @return render_frame() returns the rendering frame.
 */
SV_FRAME* render_frame(render* self);

/**
 * @brief Return the canvas to render
 * @param[in] self  pointer of the render structure
 * @return render_canvas() returns the canvas to render.
 */
canvas* render_canvas(render* self);

/**
 * @brief Return the canvas of the star taxonomy
 * @param[in] self     pointer of the render structure
 * @param[in] magidex  index of star magnitude
 @ return render_star_tex() returns the canvas of the star texture.
 */
canvas* render_star_tex(render* self, int magidx);

/**
 * @brief Main function for render
 * @param[in] self  pointer of the render structure
 */
BOOL render_image(render* self);

const char* render_get_message(render* self);

/**
 * @brief Render image for debug
 * @param[in] self  pointer of the render structure
 */
BOOL render_lookdown_image(render* self);

/**
 * @brief load an image from path and return canvas
 * canvas must be freed after finish
 * @param config  application configuration
 * @param path    file path
 */
canvas* load_image(FLOWIGConfig* config, const char* path);

__END_DECLS

#endif /* _RENDER_H_ */
