/**
 * @file      shape_model.h
 * @brief     Shape model definitions
 * @date      2012/08/15
 *
 * @copyright
 * Copyright 2012 Japan Aerospace Exploration Agency
 *
 * @par       History:
 * - 2012/08/15 First release
 *
 */

#ifndef __SHAPE_MODEL_H__
#define __SHAPE_MODEL_H__

#include <stdlib.h>

__BEGIN_DECLS

typedef int ShapeBool;
#define SHAPE_TRUE  1
#define SHAPE_FALSE 0

/*! @brief Shape capacity is expanded by this magnification factor
 *         when the registered triangles reaches the current maximum.
 */
#define SHAPE_CAPACITY_MAGNIFICATION_FACTOR (1.3)

typedef struct tag_shapeTriangle {
  struct {
    float x, y, z;
  } normal;
  struct {
    float x, y, z;
  } va;
  struct {
    float x, y, z;
  } vb;
  struct {
    float x, y, z;
  } vc;
  unsigned short attr;
} shapeTriangle;

typedef struct tag_shape {
  shapeTriangle* triangle;
  size_t capacity;
  size_t size;
} shape_model;

/**
 * @brief Create an instance of shape_model
 *
 * @return      shape_create() returns the pointer of a shape_model instance.
 *              Otherwise, NULL is returned.
 */
shape_model* shape_create();

/**
 * @brief Free the shape_model instance
 *
 * @param[in]   self    the pointer of a shape_model instance
 */
void shape_free(shape_model* self);

/**
 * @brief Return the size of the shape_model instance
 *
 * @param[in]   self    the pointer of a shape_model instance
 * @return      shape_size() returns the size of a shape_model instance.
 */
size_t shape_size(const shape_model* self);

/**
 * @brief Return the pointer of the shape_model triangles
 *
 * @param[in]   self    the pointer of a shape_model instance
 * @return      shape_triangle() returns the pointer of an array of the shape_model triangles.
 */
const shapeTriangle* shape_triangle(const shape_model* self);

/**
 * @brief Return the pointer of the triangle element of the shape_model instance
 *        at user-supplied index.
 * @param[in]   self    the pointer of a shape_model instance
 * @param[in]   index   index of triangle array
 * @return      shape_triangle_at() returns the pointer of index-th element of an array
 *              of the shape_model triangles. Otherwise, 0 is returned.
 */
const shapeTriangle* shape_triangle_at(const shape_model* self, size_t index);

/**
 * @brief Add a triangle to the shape_model instance
 * @param[in]   self    the pointer of a shape_model instance
 * @param[in]   tri     triangle shape array
 * @return      shape_push_triangle() returns SHAPE_TRUE if the triangle is registered.
 *              Otherwise, SHAPE_FALSE is returned.
 */
ShapeBool shape_push_triangle(shape_model* self, const shapeTriangle* tri);


/**
 * @brief Expand the size of the shape_model instance
 * @param[in]   self  the pointer of a shape_model instance
 * @param[in]   size  new size of the shape_model instance
 * @return      shape_push_triangle() returns SHAPE_TRUE if the triangle is registered.
 *              Otherwise, SHAPE_FALSE is returned.
 */
ShapeBool shape_extend(shape_model* self, size_t size);

__END_DECLS

#endif /* __SHAPE_MODEL_H__ */
