/**
 * @file      spice_cell_lua.c
 * @brief     LUA binding for spice_cell
 * @date      2010-03-12
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <math.h>
#include "SpiceUsr.h"
#include "spice_cell_lua.h"

#ifndef NULLCHAR
#define NULLCHAR ('\0')
#endif

#define SPICE_DEFUN(name)        static int luafunc_ ## name(lua_State * L)
#define SPICE_REGFUN(lua, name)  lua_register((lua), # name, luafunc_ ## name)
#define SPICE_FUNC(name)         luafunc_ ## name


SpiceCell* spice_cell_lua_create(lua_State* L, int type, int size, int len)
{
  SpiceCell* cell;
  int unit;
  cell = lua_newuserdata(L, sizeof(SpiceCell));
  printf("SpiceCell(%08X) Allocate.\n", (unsigned)cell);
  cell->length = len;
  cell->size = size;
  cell->card = 0;
  cell->isSet = SPICETRUE;
  cell->adjust = SPICEFALSE;
  cell->init = SPICEFALSE;
  switch (type) {
  case SPICE_CHR: unit = len;              break;
  case SPICE_DP:  unit = sizeof(SpiceDouble); break;
  case SPICE_INT: unit = sizeof(SpiceInt);    break;
  }
  cell->base = malloc(unit*size);
  cell->data = cell->base + unit * SPICE_CELL_CTRLSZ;
  // set metatable
  luaL_getmetatable(L, SpiceCellMetatableKey);
  lua_setmetatable(L, -2);
  return cell;
}

void spice_cell_lua_free(lua_State* L, SpiceCell* cell)
{
  printf("SpiceCell(%08X) GC.\n", (unsigned)cell);
  free(cell->base);
}

/**
 * SpiceCell spice_cell(string type, int size [, int len])
 * Returns SpiceCell
 * @param type  data type. Select SPICE_CHR, SPICE_DP, and SPICE_INT.
 * @param size  maximum size to store cells
 * @param len   length of one element. Set the type is SPICE CHR.
 * @result SpiceCell
 */
SPICE_DEFUN(spice_cell)
{
  const char* options[] = {"SPICE_CHR", "SPICE_DP", "SPICE_INT"};
  SpiceCell* cell;
  int optidx;
  int type;
  int size;
  int len = 0;
  optidx = luaL_checkoption(L, 1, NULL, options);
  switch (optidx) {
  case 0: type = SPICE_CHR; break;
  case 1: type = SPICE_DP; break;
  case 2: type = SPICE_INT; break;
  }
  type = luaL_checkinteger(L, 2);
  if (type == SPICE_CHR) {
    len = luaL_checkinteger(L, 3);
  }
  cell = spice_cell_lua_create(L, type, size, len);
  return 1;
}

SPICE_DEFUN(spice_cell_free)
{
  SpiceCell* cell = luaL_checkudata(L, 1, SpiceCellMetatableKey);
  spice_cell_lua_free(L, cell);
  return 0;
}

/**
 * index subscript starts with 1 like Lua.
 * Use spice_cell_get() or spice_cell_elem() if you want to 0 as start index.
 */
SPICE_DEFUN(spice_cell_index)
{
  SpiceCell* cell = luaL_checkudata(L, 1, SpiceCellMetatableKey);
  int index = luaL_checknumber(L, 2);
  if (cell->card < index) {
    return luaL_argerror(L, 1, "index overflow");
  }
  switch (cell->dtype) {
  case SPICE_CHR:
    lua_pushstring(L, SPICE_CELL_ELEM_C(cell, index-1));
    break;
  case SPICE_DP:
    lua_pushnumber(L, SPICE_CELL_ELEM_D(cell, index-1));
    break;
  case SPICE_INT:
    lua_pushinteger(L, SPICE_CELL_ELEM_I(cell, index-1));
    break;
  default:
    return luaL_argerror(L, 1, "Illegal Cell data.");
  }
  return 1;
}

SPICE_DEFUN(spice_cell_len)
{
  SpiceCell* cell = luaL_checkudata(L, 1, SpiceCellMetatableKey);
  lua_pushinteger(L, card_c(cell));
  return 1;
}

SPICE_DEFUN(spice_size)
{
  SpiceCell* cell = luaL_checkudata(L, 1, SpiceCellMetatableKey);
  lua_pushinteger(L, size_c(cell));
  return 1;
}

SPICE_DEFUN(spice_card)
{
  SpiceCell* cell = luaL_checkudata(L, 1, SpiceCellMetatableKey);
  lua_pushinteger(L, card_c(cell));
  return 1;
}

SPICE_DEFUN(spice_scard)
{
  int card = luaL_checknumber(L, 1);
  SpiceCell* cell = luaL_checkudata(L, 2, SpiceCellMetatableKey);
  scard_c(card, cell);
  return 0;
}

/*
 * implement appndc_c(), appndd_c(), appndi_c().
 */
SPICE_DEFUN(spice_appnd)
{
  SpiceCell* cell;
  cell = luaL_checkudata(L, 2, SpiceCellMetatableKey);
  switch (cell->dtype) {
  case SPICE_CHR:
    appndc_c(luaL_checkstring(L, 1), cell);
    break;
  case SPICE_DP:
    appndd_c(luaL_checknumber(L, 1), cell);
    break;
  case SPICE_INT:
    appndi_c(luaL_checkinteger(L, 1), cell);
    break;
  default:
    return luaL_argerror(L, 2, "Illegal Cell data.");
  }
  lua_settop(L, 2);
  return 1;
}

SPICE_DEFUN(spice_copy)
{
  SpiceCell* cell;
  SpiceCell* c2;
  int i;
  int max;
  cell = luaL_checkudata(L, 1, SpiceCellMetatableKey);
  c2 = spice_cell_lua_create(L, cell->dtype, size_c(cell), card_c(cell));
  for (i = 0; i < max; i++) {
    switch (c2->dtype) {
    case SPICE_CHR:
      appndc_c(SPICE_CELL_ELEM_C(cell, i), c2);
      break;
    case SPICE_DP:
      appndd_c(SPICE_CELL_ELEM_D(cell, i), c2);
      break;
    case SPICE_INT:
      appndi_c(SPICE_CELL_ELEM_I(cell, i), c2);
      break;
    default:
      return luaL_argerror(L, 1, "Illegal Cell data.");
    }
  }
  return 1;
}

SPICE_DEFUN(spice_cell_elem)
{
  SpiceCell* cell;
  int i;
  cell = luaL_checkudata(L, 1, SpiceCellMetatableKey);
  i = luaL_checkinteger(L, 2);
  switch (cell->dtype) {
  case SPICE_CHR:
    lua_pushstring(L, SPICE_CELL_ELEM_C(cell, i));
    break;
  case SPICE_DP:
    lua_pushnumber(L, SPICE_CELL_ELEM_D(cell, i));
    break;
  case SPICE_INT:
    lua_pushinteger(L, SPICE_CELL_ELEM_I(cell, i));
    break;
  default:
    return luaL_argerror(L, 1, "Illegal Cell data.");
  }
  return 1;
}

SPICE_DEFUN(spice_cell_set)
{
  SpiceCell* cell;
  int i;
  const char* s;
  i = luaL_checkinteger(L, 2);
  cell = luaL_checkudata(L, 3, SpiceCellMetatableKey);
  switch (cell->dtype) {
  case SPICE_CHR:
    s = luaL_checkstring(L, 1);
    SPICE_CELL_SET_C(s, i, cell);
    break;
  case SPICE_DP:
    SPICE_CELL_SET_D(luaL_checknumber(L, 1), i, cell);
    break;
  case SPICE_INT:
    SPICE_CELL_SET_I(luaL_checkinteger(L, 1), i, cell);
    break;
  default:
    return luaL_argerror(L, 3, "Illegal Cell data.");
  }
  return 0;
}

/* Window */

SPICE_DEFUN(spice_wncard)
{
  SpiceCell* window;
  int n;
  window = luaL_checkudata(L, 1, SpiceCellMetatableKey);
  n = wncard_c(window);
  lua_pushinteger(L, n);
  return 1;
}

SPICE_DEFUN(spice_wnfetd)
{
  SpiceCell* window;
  int n;
  double left, right;
  window = luaL_checkudata(L, 1, SpiceCellMetatableKey);
  n = luaL_checknumber(L, 2);
  wnfetd_c(window, n, &left, &right);
  lua_pushnumber(L, left);
  lua_pushnumber(L, right);
  return 2;
}

void spice_cell_lua_init(lua_State* L)
{
  SPICE_REGFUN(L, spice_cell);
  SPICE_REGFUN(L, spice_cell_free);
  SPICE_REGFUN(L, spice_cell_len);
  SPICE_REGFUN(L, spice_cell_index);

  // macro
  SPICE_REGFUN(L, spice_cell_elem);
  lua_register(L, "spice_cell_get", SPICE_FUNC(spice_cell_elem));
  SPICE_REGFUN(L, spice_cell_set);

  SPICE_REGFUN(L, spice_card);
  SPICE_REGFUN(L, spice_scard);
  SPICE_REGFUN(L, spice_size);
  SPICE_REGFUN(L, spice_wncard);
  SPICE_REGFUN(L, spice_wnfetd);
  SPICE_REGFUN(L, spice_appnd);
  SPICE_REGFUN(L, spice_copy);

  // create metatable
  luaL_newmetatable(L, SpiceCellMetatableKey);
  // gc
  lua_pushstring(L, "__gc");
  lua_pushcfunction(L, SPICE_FUNC(spice_cell_free));
  lua_settable(L, -3);
  // len
  lua_pushstring(L, "__len");
  lua_pushcfunction(L, SPICE_FUNC(spice_cell_len));
  lua_settable(L, -3);
  // index
  lua_pushstring(L, "__index");
  lua_pushcfunction(L, SPICE_FUNC(spice_cell_index));
  lua_settable(L, -3);

  lua_pop(L, 1);
}
