/**
 * @file      spice_lua.c
 * @brief     LUA SPICE binding
 * @date      2010-03-12
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <math.h>
#include "SpiceUsr.h"
#include "spice_lua.h"
#include "spice_cell_lua.h"

#define SPICE_DEFUN(name)           static int luafunc_ ## name(lua_State *L)
#define SPICE_REGFUN(lua, name)  lua_register((lua), # name, luafunc_ ## name)
#define SPICE_FUNC(name)         luafunc_ ## name

/**
 * Convert the vector of double [] into a table of lua and stuck it on the stack.
 */
static int spice_lua_pushvector(lua_State* L, const double* vec, int dim)
{
  int i;
  lua_createtable(L, dim, 0);
  for (i = 0; i < dim; i++) {
    lua_pushinteger(L, i+1);
    lua_pushnumber(L, vec[i]);
    lua_settable(L, -3);
  }
  return 1;
}

/**
 * Convert Lua table to vector.
 */
static int spice_lua_tovector(lua_State* L, int tblidx, double* vec, int dim)
{
  int i;
  lua_pushvalue(L, tblidx);
  for (i = 0; i < dim; i++) {
    lua_pushinteger(L, i+1);
    lua_gettable(L, -2);
    vec[i] = lua_tonumber(L, -1);
    lua_pop(L, 1);
  }
  lua_pop(L, 1);
  return 1;
}

/**
 * Convert int[][] matrix to lua's table and stack it on the stack.
 */
static int spice_lua_pushmatrix(lua_State* L, const double* mat, int dim1, int dim2)
{
  int i, j, x;
  lua_createtable(L, dim1, 0);
  for (i = 0; i < dim1; i++) {
    lua_pushinteger(L, i+1);
    lua_createtable(L, dim2, 0);
    for (j = 0; j < dim2; j++) {
      x = i*dim1+j;
      lua_pushinteger(L, j+1);
      lua_pushnumber(L, mat[x]);
      lua_settable(L, -3);
    }
    lua_settable(L, -3);
  }
  return 1;
}

/**
 * convert lua table to matrix
 */
static int spice_lua_tomatrix(lua_State* L, int tblidx, double* mat, int dim1, int dim2)
{
  int i, j, x;
  lua_pushvalue(L, tblidx);
  for (i = 0; i < dim1; i++) {
    lua_pushinteger(L, i+1);
    lua_gettable(L, -2);
    for (j = 0; j < dim2; j++) {
      x = i*dim1+j;
      lua_pushinteger(L, j+1);
      lua_gettable(L, -2);
      mat[x] = lua_tonumber(L, -1);
      lua_pop(L, 1);
    }
    lua_pop(L, 1);
  }
  lua_pop(L, 1);
  return 1;
}

static int spice_lua_parse_axis(const char* axis)
{
  int iaxis = 0;
  switch (axis[0]) {
  case '1': case 'x': case 'X': iaxis = 1; break;
  case '2': case 'y': case 'Y': iaxis = 2; break;
  case '3': case 'z': case 'Z': iaxis = 3; break;
  }
  return iaxis;
}



/**
 * furnsh_c() binding.
 */
SPICE_DEFUN(spice_furnsh)
{
  const char* path = luaL_checkstring(L, 1);
  furnsh_c(path);
  return 0;
}

/**
 * unload_c() binding.
 * Unload spice kernel.
 */
SPICE_DEFUN(spice_unload)
{
  const char* path = luaL_checkstring(L, 1);
  unload_c(path);
  return 0;
}

// Converting between Ephemeris Time and Spacecraft Clock (SCLK)

SPICE_DEFUN(spice_scs2e)
{
  int sc;
  const char* sclkch;
  double et;
  sc = (int)floor(luaL_checknumber(L, 1));
  sclkch = luaL_checkstring(L, 2);
  scs2e_c(sc, sclkch, &et);
  lua_pushnumber(L, et);
  return 1;
}

SPICE_DEFUN(spice_sce2s)
{
  int sc;
  double et;
  char sclkch[256];
  sc = (int)floor(luaL_checknumber(L, 1));
  et = luaL_checknumber(L, 2);
  sce2s_c(sc, et, sizeof(sclkch), sclkch);
  lua_pushstring(L, sclkch);
  return 1;
}

SPICE_DEFUN(spice_sct2e)
{
  int sc;
  double sclkdp, et;
  sc = (int)floor(luaL_checknumber(L, 1));
  sclkdp = luaL_checknumber(L, 2);
  sct2e_c(sc, sclkdp, &et);
  lua_pushnumber(L, et);
  return 1;
}

SPICE_DEFUN(spice_sce2c)
{
  int sc;
  double sclkdp, et;
  sc = (int)floor(luaL_checknumber(L, 1));
  et = luaL_checknumber(L, 2);
  sce2c_c(sc, et, &sclkdp);
  lua_pushnumber(L, sclkdp);
  return 1;
}

SPICE_DEFUN(spice_scencd)
{
  int sc;
  const char* sclkch;
  double sclkdp;
  sc = (int)floor(luaL_checknumber(L, 1));
  sclkch = luaL_checkstring(L, 2);
  scencd_c(sc, sclkch, &sclkdp);
  lua_pushnumber(L, sclkdp);
  return 1;
}

SPICE_DEFUN(spice_scdecd)
{
  int sc;
  double sclkdp;
  char sclkch[256];
  sc = (int)floor(luaL_checknumber(L, 1));
  sclkdp = luaL_checknumber(L, 2);
  scdecd_c(sc, sclkdp, sizeof(sclkch), sclkch);
  lua_pushstring(L, sclkch);
  return 1;
}

SPICE_DEFUN(spice_str2et)
{
  const char* time = luaL_checkstring(L, 1);
  lua_Number et;
  str2et_c(time, &et);
  lua_pushnumber(L, et);
  return 1;
}

SPICE_DEFUN(spice_timout)
{
  double et = luaL_checknumber(L, 1);
  const char* fmt = luaL_checkstring(L, 2);
  char output[64];
  timout_c(et, fmt, 64, output);
  lua_pushstring(L, output);
  return 1;
}

SPICE_DEFUN(spice_et2utc)
{
  double et = luaL_checknumber(L, 1);
  const char* fmt = luaL_checkstring(L, 2);
  int prec = (int)floor(luaL_checknumber(L, 3));
  char output[64];
  et2utc_c(et, fmt, prec, 64, output);
  lua_pushstring(L, output);
  return 1;
}

SPICE_DEFUN(spice_et2lst)
{
  double et   = luaL_checknumber(L, 1);
  int body = (int)floor(luaL_checknumber(L, 2));
  double lng  = luaL_checknumber(L, 3);
  const char* type = luaL_checkstring(L, 4);
  SpiceInt hour, min, sec;
  char time[64];
  char ampm[10];
  et2lst_c(et, body, lng, type, sizeof(time), sizeof(ampm), &hour, &min, &sec, time, ampm);
  lua_pushinteger(L, hour);       // hour
  lua_pushinteger(L, min);        // min
  lua_pushinteger(L, sec);        // sec
  lua_pushstring(L, time);        // time
  lua_pushstring(L, time);        // ampm
  return 5;
}

// Constants and Orientation for Natural Bodies (PCK)

SPICE_DEFUN(spice_bodfnd)
{
  int body = (int)floor(luaL_checknumber(L, 1));
  const char* item = luaL_checkstring(L, 2);
  SpiceBoolean flag = bodfnd_c(body, item);
  lua_pushboolean(L, flag);
  return 1;
}

SPICE_DEFUN(spice_bodvrd)
{
  const char* bodynum = luaL_checkstring(L, 1);
  const char* item = luaL_checkstring(L, 2);
  int max = (int)floor(luaL_checknumber(L, 3));
  SpiceInt dim;
  double* values;

  values = malloc(sizeof(double)*max);
  memset(values, 0, sizeof(double)*max);
  bodvrd_c(bodynum, item, max, &dim, values);
  spice_lua_pushvector(L, values, dim);
  free(values);

  return 1;
}

SPICE_DEFUN(spice_pxform)
{

  const char* from = luaL_checkstring(L, 1);
  const char* to = luaL_checkstring(L, 2);
  double et = luaL_checknumber(L, 3);
  double rotate[3][3];
  pxform_c(from, to, et, rotate);
  spice_lua_pushmatrix(L, (double *)rotate, 3, 3);
  return 1;
}

SPICE_DEFUN(spice_sxform)
{
  const char* from = luaL_checkstring(L, 1);
  const char* to = luaL_checkstring(L, 2);
  double et = luaL_checknumber(L, 3);
  double xform[6][6];
  sxform_c(from, to, et, xform);
  spice_lua_pushmatrix(L, (double *)xform, 6, 6);
  return 1;
}

/**
 * udata<SpiceCell> spice_pckform(String pck, udata<SpiceCell> ids)
 */
SPICE_DEFUN(spice_pckfrm)
{
  const char* pck;
  SpiceCell* ids;
  pck = luaL_checkstring(L, 1);
  ids = luaL_checkudata(L, 2, SpiceCellMetatableKey);
  pckfrm_c(pck, ids);
  lua_pushvalue(L, 2);
  return 1;
}

/**
 * table<SpiceCell> spice_pckcov(string pck, int idcode, table<cell> ids)
 */
SPICE_DEFUN(spice_pckcov)
{
  const char* pck;
  int idcode;
  SpiceCell* cover;
  pck = luaL_checkstring(L, 1);
  idcode = luaL_checkinteger(L, 2);
  cover = luaL_checkudata(L, 3, SpiceCellMetatableKey);
  pckcov_c(pck, idcode, cover);
  lua_pushvalue(L, 3);
  return 1;
}

// Computing Positions of Spacecraft and Natural Bodies (SPK)

/**
 * table<vec>, double spice_spkezr(string targ, double et, string ref, string abcorr, string obs)
 */
SPICE_DEFUN(spice_spkezr)
{
  const char* targ   = luaL_checkstring(L, 1);
  double et     = luaL_checknumber(L, 2);
  const char* ref    = luaL_checkstring(L, 3);
  const char* abcorr = luaL_checkstring(L, 4);
  const char* obs    = luaL_checkstring(L, 5);
  SpiceDouble arg[6];
  SpiceDouble lt;
  spkezr_c(targ, et, ref, abcorr, obs, arg, &lt);
  spice_lua_pushvector(L, arg, 6);
  lua_pushnumber(L, lt);
  return 2;
}

/**
 * table<vec>, double = spice_spkpos(string targ, double et, string ref, string abcorr, string obs)
 */
SPICE_DEFUN(spice_spkpos)
{
  const char* targ   = luaL_checkstring(L, 1);
  double et     = luaL_checknumber(L, 2);
  const char* ref    = luaL_checkstring(L, 3);
  const char* abcorr = luaL_checkstring(L, 4);
  const char* obs    = luaL_checkstring(L, 5);
  SpiceDouble arg[3];
  SpiceDouble lt;
  spkpos_c(targ, et, ref, abcorr, obs, arg, &lt);
  spice_lua_pushvector(L, arg, 3);
  lua_pushnumber(L, lt);
  return 2;
}

/**
 * SpiceCell spice_spkobj(string spk, SpiceCell cell)
 */
SPICE_DEFUN(spice_spkobj)
{
  const char* spk;
  SpiceCell* cell;
  spk = luaL_checkstring(L, 1);
  cell = luaL_checkudata(L, 2, SpiceCellMetatableKey);
  spkobj_c(spk, cell);
  lua_pushvalue(L, 1);
  return 1;
}

SPICE_DEFUN(spice_spkcov)
{
  const char* spk;
  int idcode;
  SpiceCell* cell;
  spk = luaL_checkstring(L, 1);
  idcode = luaL_checkinteger(L, 2);
  cell = luaL_checkudata(L, 3, SpiceCellMetatableKey);
  spkcov_c(spk, idcode, cell);
  lua_pushvalue(L, 1);
  return 1;
}

// Computing Orientation for Spacecraft and Instruments (CK)

SPICE_DEFUN(spice_ckobj)
{
  const char* ck;
  SpiceCell* ids;
  ck = luaL_checkstring(L, 1);
  ids = luaL_checkudata(L, 2, SpiceCellMetatableKey);
  ckobj_c(ck, ids);
  lua_pushvalue(L, 2);
  return 1;
}

SPICE_DEFUN(spice_ckcov)
{
  const char* ck;
  int idcode;
  SpiceBoolean needav;
  const char* level;
  double tol;
  const char* timsys;
  SpiceCell* cover;
  ck     = luaL_checkstring(L, 1);
  idcode = luaL_checkinteger(L, 2);
  needav = lua_toboolean(L, 3);
  level  = luaL_checkstring(L, 4);
  tol    = luaL_checknumber(L, 5);
  timsys = luaL_checkstring(L, 6);
  cover  = luaL_checkudata(L, 7, SpiceCellMetatableKey);
  ckcov_c(ck, idcode, needav, level, tol, timsys, cover);
  lua_pushvalue(L, 7);
  return 1;
}

SPICE_DEFUN(spice_ckgp)
{
  int inst;
  double sclkdp;
  double tol;
  const char* ref;
  double cmap[3][3];
  double clkout;
  SpiceBoolean found;
  inst   = luaL_checkinteger(L, 1);
  sclkdp = luaL_checknumber(L, 2);
  tol    = luaL_checknumber(L, 3);
  ref    = luaL_checkstring(L, 4);
  ckgp_c(inst, sclkdp, tol, ref, cmap, &clkout, &found);
  spice_lua_pushmatrix(L, (double *)cmap, 3, 3);
  lua_pushnumber(L, clkout);
  lua_pushnumber(L, found);
  return 3;
}

SPICE_DEFUN(spice_ckgpav)
{
  int inst;
  double sclkdp;
  double tol;
  const char* ref;
  double cmap[3][3];
  double av[3];
  double clkout;
  SpiceBoolean found;
  inst   = luaL_checkinteger(L, 1);
  sclkdp = luaL_checknumber(L, 2);
  tol    = luaL_checknumber(L, 3);
  ref    = luaL_checkstring(L, 4);
  ckgpav_c(inst, sclkdp, tol, ref, cmap, av, &clkout, &found);
  spice_lua_pushmatrix(L, (double *)cmap, 3, 3);
  spice_lua_pushvector(L, av, 3);
  lua_pushnumber(L, clkout);
  lua_pushboolean(L, found);
  return 4;
}

// Retrieving instrument parameters (IK)
SPICE_DEFUN(spice_getfov)
{
  int instid, room;
  char shape[128];
  char frame[128];
  double bsight[3];
  SpiceInt n;
  double* bounds;
  instid = (int)floor(luaL_checknumber(L, 1));
  room = (int)floor(luaL_checknumber(L, 2));
  bounds = malloc(sizeof(double)*room);
  getfov_c(instid, room, sizeof(shape), sizeof(frame), shape, frame, bsight, &n, (void *)bounds);
  lua_pushstring(L, shape);
  lua_pushstring(L, frame);
  spice_lua_pushvector(L, bsight, 3);
  spice_lua_pushmatrix(L, bounds, n, 3);
  free(bounds);
  return 5;
}

SPICE_DEFUN(spice_gdpool)
{
  const char* name;
  int start, room;
  SpiceInt n;
  SpiceDouble value;
  SpiceBoolean found;
  int i;
  name  = luaL_checkstring(L, 1);
  start = (int)floor(luaL_checknumber(L, 2));
  room  = (int)floor(luaL_checknumber(L, 3));
  lua_newtable(L);
  for (i = 0; 0 == room || i < room; i++) {
    gdpool_c(name, start+i, 1, &n, &value, &found);
    if (!found || n == 0) {
      break;
    }
    lua_pushinteger(L, i+1);
    lua_pushnumber(L, value);
    lua_settable(L, -3);
  }
  lua_pushboolean(L, found);
  return 2;
}

SPICE_DEFUN(spice_gipool)
{
  const char* name;
  int start, room;
  SpiceInt n;
  SpiceInt value;
  SpiceBoolean found;
  int i;
  name  = luaL_checkstring(L, 1);
  start = (int)floor(luaL_checknumber(L, 2));
  room  = (int)floor(luaL_checknumber(L, 3));
  lua_newtable(L);
  for (i = 0; 0 == room || i < room; i++) {
    gipool_c(name, start+i, 1, &n, &value, &found);
    if (!found || n == 0) {
      break;
    }
    lua_pushinteger(L, i+1);
    lua_pushinteger(L, value);
    lua_settable(L, -3);
  }
  lua_pushboolean(L, found);
  return 2;
}

SPICE_DEFUN(spice_gcpool)
{
  const char* name;
  int start, room;
  SpiceInt n;
  char value[1024];
  SpiceBoolean found;
  int i;
  name  = luaL_checkstring(L, 1);
  start = (int)floor(luaL_checknumber(L, 2));
  room  = (int)floor(luaL_checknumber(L, 3));
  lua_newtable(L);
  for (i = 0; 0 == room || i < room; i++) {
    gcpool_c(name, start+i, 1, sizeof(value), &n, value, &found);
    if (!found || n == 0) {
      break;
    }
    lua_pushinteger(L, i+1);
    lua_pushstring(L, value);
    lua_settable(L, -3);
  }
  lua_pushboolean(L, found);
  return 2;
}

// Mapping Between Object Names and NAIF IDs

/**
 * string, boolean spice_bodc2n(int code [, int maxlen])
 */
SPICE_DEFUN(spice_bodc2n)
{
  int code;
  int len = 256;
  char* name;
  SpiceBoolean found;
  code = luaL_checkinteger(L, 1);
  if (2 == lua_gettop(L)) {
    len = luaL_checkinteger(L, 2);
  }
  name = malloc(len+1);
  bodc2n_c(code, len+1, name, &found);
  lua_pushstring(L, name);
  lua_pushboolean(L, found);
  free(name);
  return 2;
}

SPICE_DEFUN(spice_bodn2c)
{
  const char* name;
  SpiceInt code;
  SpiceBoolean found;
  name = luaL_checkstring(L, 1);
  bodn2c_c(name, &code, &found);
  lua_pushinteger(L, code);
  lua_pushboolean(L, found);
  return 2;
}

//
// computing derived geometry
//

// Computing Planetocentric, Planetodetic, and Planetographic Coordinates

SPICE_DEFUN(spice_reclat)
{
  double rectan[3];
  double rad, lng, lat;
  spice_lua_tovector(L, 1, rectan, 3);
  reclat_c(rectan, &rad, &lng, &lat);
  lua_pushnumber(L, rad);
  lua_pushnumber(L, lng);
  lua_pushnumber(L, lat);
  return 3;
}

SPICE_DEFUN(spice_latrec)
{
  double rad, lng, lat;
  double rectan[3];
  rad = luaL_checknumber(L, 1);
  lng = luaL_checknumber(L, 2);
  lat = luaL_checknumber(L, 3);
  latrec_c(rad, lng, lat, rectan);
  spice_lua_pushvector(L, rectan, 3);
  return 1;
}

SPICE_DEFUN(spice_srfrec)
{
  int body;
  double lng, lat;
  double rectan[3];
  body = (int)floor(luaL_checknumber(L, 1));
  lng = luaL_checknumber(L, 2);
  lat = luaL_checknumber(L, 3);
  srfrec_c(body, lng, lat, rectan);
  spice_lua_pushvector(L, rectan, 3);
  return 1;
}

SPICE_DEFUN(spice_recgeo)
{
  double rectan[3];
  double re, f, lng, lat, alt;
  spice_lua_tovector(L, 1, rectan, 3);
  re = luaL_checknumber(L, 2);
  f = luaL_checknumber(L, 3);
  recgeo_c(rectan, re, f, &lng, &lat, &alt);
  lua_pushnumber(L, lng);
  lua_pushnumber(L, lat);
  lua_pushnumber(L, alt);
  return 3;
}

SPICE_DEFUN(spice_georec)
{
  double lng, lat, alt;
  double rectan[3];
  double re, f;
  lng = luaL_checknumber(L, 1);
  lat = luaL_checknumber(L, 2);
  alt = luaL_checknumber(L, 3);
  re  = luaL_checknumber(L, 4);
  f   = luaL_checknumber(L, 5);
  georec_c(lng, lat, alt, re, f, rectan);
  spice_lua_pushvector(L, rectan, 3);
  return 1;
}

SPICE_DEFUN(spice_recpgr)
{
  const char* body;
  double rectan[3];
  double re, f, lng, lat, alt;
  body = luaL_checkstring(L, 1);
  spice_lua_tovector(L, 2, rectan, 3);
  re = luaL_checknumber(L, 3);
  f  = luaL_checknumber(L, 4);
  recpgr_c(body, rectan, re, f, &lng, &lat, &alt);
  lua_pushnumber(L, lng);
  lua_pushnumber(L, lat);
  lua_pushnumber(L, alt);
  return 3;
}

SPICE_DEFUN(spice_pgrrec)
{
  const char* body;
  double lng, lat, alt, re, f;
  double rectan[3];
  body = luaL_checkstring(L, 1);
  lng = luaL_checknumber(L, 2);
  lat = luaL_checknumber(L, 3);
  alt = luaL_checknumber(L, 4);
  re  = luaL_checknumber(L, 5);
  f   = luaL_checknumber(L, 6);
  pgrrec_c(body, lng, lat, alt, re, f, rectan);
  spice_lua_pushvector(L, rectan, 3);
  return 1;
}

// Computing Surface Intercept Point

/**
 * table<vec>, double, table<vec>, boolean spice_sincpt(
 *   string method,
 *   string target,
 *   double et,
 *   string fixref,
 *   string abcorr,
 *   string obsrvr,
 *   string dref,
 *   table<vec> dvec)
 */
SPICE_DEFUN(spice_sincpt)
{
  const char* method;
  const char* target;
  double et;
  const char* fixref;
  const char* abcorr;
  const char* obsrvr;
  const char* dref;
  double dvec[3];
  double spoint[3];
  double trgepc;
  double srfvec[3];
  SpiceBoolean found;
  method = luaL_checkstring(L, 1);
  target = luaL_checkstring(L, 2);
  et = luaL_checknumber(L, 3);
  fixref = luaL_checkstring(L, 4);
  abcorr = luaL_checkstring(L, 5);
  obsrvr = luaL_checkstring(L, 6);
  dref = luaL_checkstring(L, 7);
  spice_lua_tovector(L, 8, dvec, 3);
  sincpt_c(method, target, et, fixref, abcorr, obsrvr, dref, dvec,
           spoint, &trgepc, srfvec, &found);
  spice_lua_pushvector(L, spoint, 3);
  lua_pushnumber(L, trgepc);
  spice_lua_pushvector(L, srfvec, 3);
  lua_pushboolean(L, found);
  return 4;
}

// Computing Sub-observer and Sub-solar Points
SPICE_DEFUN(spice_subpnt)
{
  const char* method;
  const char* target;
  double et;
  const char* fixref;
  const char* abcorr;
  const char* obsrvr;
  double spoint[3];
  double trgepc;
  double srfvec[3];
  method = luaL_checkstring(L, 1);
  target = luaL_checkstring(L, 2);
  et = luaL_checknumber(L, 3);
  fixref = luaL_checkstring(L, 4);
  abcorr = luaL_checkstring(L, 5);
  obsrvr = luaL_checkstring(L, 6);
  subpnt_c(method, target, et, fixref, abcorr, obsrvr,
           spoint, &trgepc, srfvec);
  spice_lua_pushvector(L, spoint, 3);
  lua_pushnumber(L, trgepc);
  spice_lua_pushvector(L, srfvec, 3);
  return 3;
}

SPICE_DEFUN(spice_subslr)
{
  const char* method;
  const char* target;
  double et;
  const char* fixref;
  const char* abcorr;
  const char* obsrvr;
  double spoint[3];
  double trgepc;
  double srfvec[3];
  method = luaL_checkstring(L, 1);
  target = luaL_checkstring(L, 2);
  et = luaL_checknumber(L, 3);
  fixref = luaL_checkstring(L, 4);
  abcorr = luaL_checkstring(L, 5);
  obsrvr = luaL_checkstring(L, 6);
  subslr_c(method, target, et, fixref, abcorr, obsrvr,
           spoint, &trgepc, srfvec);
  spice_lua_pushvector(L, spoint, 3);
  lua_pushnumber(L, trgepc);
  spice_lua_pushvector(L, srfvec, 3);
  return 3;
}

// Computing Illumination Angles

SPICE_DEFUN(spice_ilumin)
{
  const char* method;
  const char* target;
  double et;
  const char* fixref;
  const char* abcorr;
  const char* obsrvr;
  double spoint[3];
  double trgepc;
  double srfvec[3];
  double phase;
  double solar;
  double emissn;
  method = luaL_checkstring(L, 1);
  target = luaL_checkstring(L, 2);
  et = luaL_checknumber(L, 3);
  fixref = luaL_checkstring(L, 4);
  abcorr = luaL_checkstring(L, 5);
  obsrvr = luaL_checkstring(L, 6);
  ilumin_c(method, target, et, fixref, abcorr, obsrvr,
           spoint, &trgepc, srfvec, &phase, &solar, &emissn);
  spice_lua_pushvector(L, spoint, 3);
  lua_pushnumber(L, trgepc);
  spice_lua_pushvector(L, srfvec, 3);
  lua_pushnumber(L, phase);
  lua_pushnumber(L, solar);
  lua_pushnumber(L, emissn);
  return 3;
}

// Computing and Propagating Orbital Elements

SPICE_DEFUN(spice_conics)
{
  double elts[8];
  double et;
  double state[6];
  spice_lua_tovector(L, 1, elts, 8);
  et = luaL_checknumber(L, 2);
  conics_c(elts, et, state);
  spice_lua_pushvector(L, state, 6);
  return 1;
}

SPICE_DEFUN(spice_oscelt)
{
  double state[6];
  double et, mu;
  double elts[8];
  spice_lua_tovector(L, 1, state, 6);
  et = luaL_checknumber(L, 2);
  mu = luaL_checknumber(L, 3);
  oscelt_c(state, et, mu, elts);
  spice_lua_pushvector(L, elts, 8);
  return 1;
}

//
// CSPICE APIs for coordinate conversions
//

// Converting from and to rectangular coordinates
SPICE_DEFUN(spice_recrad)
{
  double rectan[3];
  double range, ra, dec;
  spice_lua_tovector(L, 1, rectan, 3);
  recrad_c(rectan, &range, &ra, &dec);
  lua_pushnumber(L, range);
  lua_pushnumber(L, ra);
  lua_pushnumber(L, dec);
  return 3;
}

SPICE_DEFUN(spice_radrec)
{
  double range, ra, dec;
  double rectan[3];
  range = luaL_checknumber(L, 1);
  ra    = luaL_checknumber(L, 2);
  dec   = luaL_checknumber(L, 3);
  radrec_c(range, ra, dec, rectan);
  spice_lua_pushvector(L, rectan, 3);
  return 1;
}

SPICE_DEFUN(spice_recsph)
{
  double rectan[3];
  double r, colat, lng;
  spice_lua_tovector(L, 1, rectan, 3);
  recsph_c(rectan, &r, &colat, &lng);
  lua_pushnumber(L, r);
  lua_pushnumber(L, colat);
  lua_pushnumber(L, lng);
  return 3;
}

SPICE_DEFUN(spice_sphrec)
{
  double r, colat, lng;
  double rectan[3];
  r     = luaL_checknumber(L, 1);
  colat = luaL_checknumber(L, 2);
  lng   = luaL_checknumber(L, 3);
  sphrec_c(r, colat, lng, rectan);
  spice_lua_pushvector(L, rectan, 3);
  return 1;
}

SPICE_DEFUN(spice_reccyl)
{
  double rectan[3];
  double r, lng, z;
  spice_lua_tovector(L, 1, rectan, 3);
  reccyl_c(rectan, &r, &lng, &z);
  lua_pushnumber(L, r);
  lua_pushnumber(L, lng);
  lua_pushnumber(L, z);
  return 3;
}

SPICE_DEFUN(spice_cylrec)
{
  double r, lng, z;
  double rectan[3];
  r   = luaL_checknumber(L, 1);
  lng = luaL_checknumber(L, 2);
  z   = luaL_checknumber(L, 3);
  cylrec_c(r, lng, z, rectan);
  spice_lua_pushvector(L, rectan, 3);
  return 1;
}

// Converting from and to spherical coordinates

SPICE_DEFUN(spice_sphcyl)
{
  double rad, colat, slng, r, lng, z;
  rad   = luaL_checknumber(L, 1);
  colat = luaL_checknumber(L, 2);
  slng  = luaL_checknumber(L, 3);
  sphcyl_c(rad, colat, slng, &r, &lng, &z);
  lua_pushnumber(L, r);
  lua_pushnumber(L, lng);
  lua_pushnumber(L, z);
  return 3;
}

SPICE_DEFUN(spice_cylsph)
{
  double r, lngc, z, rad, colat, lng;
  r    = luaL_checknumber(L, 1);
  lngc = luaL_checknumber(L, 2);
  z    = luaL_checknumber(L, 3);
  cylsph_c(r, lngc, z, &rad, &colat, &lng);
  lua_pushnumber(L, r);
  lua_pushnumber(L, lngc);
  lua_pushnumber(L, z);
  return 3;
}

SPICE_DEFUN(spice_sphlat)
{
  double r, colat, lngs, rad, lng, lat;
  r     = luaL_checknumber(L, 1);
  colat = luaL_checknumber(L, 2);
  lngs  = luaL_checknumber(L, 3);
  sphlat_c(r, colat, lngs, &rad, &lng, &lat);
  lua_pushnumber(L, rad);
  lua_pushnumber(L, lng);
  lua_pushnumber(L, lat);
  return 3;
}

SPICE_DEFUN(spice_latsph)
{
  double rad, lng, lat, rho, colat, lngs;
  rad = luaL_checknumber(L, 1);
  lng = luaL_checknumber(L, 2);
  lat = luaL_checknumber(L, 3);
  latsph_c(rad, lng, lat, &rho, &colat, &lngs);
  lua_pushnumber(L, rho);
  lua_pushnumber(L, colat);
  lua_pushnumber(L, lngs);
  return 3;
}

// Converting from and to cylindrical coordinates

SPICE_DEFUN(spice_cyllat)
{
  double r, lngc, z, rad, lng, lat;
  r    = luaL_checknumber(L, 1);
  lngc = luaL_checknumber(L, 2);
  z    = luaL_checknumber(L, 3);
  cyllat_c(r, lngc, z, &rad, &lng, &lat);
  lua_pushnumber(L, rad);
  lua_pushnumber(L, lng);
  lua_pushnumber(L, lat);
  return 3;
}

SPICE_DEFUN(spice_latcyl)
{
  double rad, lng, lat, r, lngc, z;
  rad = luaL_checknumber(L, 1);
  lng = luaL_checknumber(L, 2);
  lat = luaL_checknumber(L, 3);
  latcyl_c(rad, lng, lat, &r, &lngc, &z);
  lua_pushnumber(L, r);
  lua_pushnumber(L, lngc);
  lua_pushnumber(L, z);
  return 3;
}

// Converting from and to latitudinal coordinates
// Converting from and to R, RA, DEC
// Converting from and to geodetic coordinates


// Performing simple operations on 3D vectors

SPICE_DEFUN(spice_vadd)
{
  double v1[3];
  double v2[3];
  double v3[3];
  spice_lua_tovector(L, 1, v1, 3);
  spice_lua_tovector(L, 2, v2, 3);
  vadd_c(v1, v2, v3);
  spice_lua_pushvector(L, v3, 3);
  return 1;
}

SPICE_DEFUN(spice_vsub)
{
  double v1[3];
  double v2[3];
  double v3[3];
  spice_lua_tovector(L, 1, v1, 3);
  spice_lua_tovector(L, 2, v2, 3);
  vsub_c(v1, v2, v3);
  spice_lua_pushvector(L, v3, 3);
  return 1;
}

SPICE_DEFUN(spice_vcrss)
{
  double v1[3];
  double v2[3];
  double v3[3];
  spice_lua_tovector(L, 1, v1, 3);
  spice_lua_tovector(L, 2, v2, 3);
  vcrss_c(v1, v2, v3);
  spice_lua_pushvector(L, v3, 3);
  return 1;
}

SPICE_DEFUN(spice_vdot)
{
  double v1[3];
  double v2[3];
  double dot;
  spice_lua_tovector(L, 1, v1, 3);
  spice_lua_tovector(L, 2, v2, 3);
  dot = vdot_c(v1, v2);
  lua_pushnumber(L, dot);
  return 1;
}

SPICE_DEFUN(spice_vrel)
{
  double v1[3];
  double v2[3];
  double rel;
  spice_lua_tovector(L, 1, v1, 3);
  spice_lua_tovector(L, 2, v2, 3);
  rel = vrel_c(v1, v2);
  lua_pushnumber(L, rel);
  return 1;
}

SPICE_DEFUN(spice_vscl)
{
  double v1[3];
  double v2[3];
  double s;
  s = luaL_checknumber(L, 1);
  spice_lua_tovector(L, 2, v1, 3);
  vscl_c(s, v1, v2);
  spice_lua_pushvector(L, v2, 3);
  return 1;
}

SPICE_DEFUN(spice_vminus)
{
  double v1[3];
  double v2[3];
  spice_lua_tovector(L, 1, v1, 3);
  vminus_c(v1, v2);
  spice_lua_pushvector(L, v2, 3);
  return 1;
}

SPICE_DEFUN(spice_vequ)
{
  double v1[3];
  double v2[3];
  spice_lua_tovector(L, 1, v1, 3);
  vequ_c(v1, v2);
  spice_lua_pushvector(L, v2, 3);
  return 1;
}

SPICE_DEFUN(spice_vzero)
{
  double v[3];
  vzero_c(v);
  spice_lua_pushvector(L, v, 3);
  return 1;
}

SPICE_DEFUN(spice_vsep)
{
  double v1[3];
  double v2[3];
  double sep;
  spice_lua_tovector(L, 1, v1, 3);
  spice_lua_tovector(L, 2, v2, 3);
  sep = vsep_c(v1, v2);
  lua_pushnumber(L, sep);
  return 1;
}

SPICE_DEFUN(spice_vdist)
{
  double v1[3];
  double v2[3];
  double dist;
  spice_lua_tovector(L, 1, v1, 3);
  spice_lua_tovector(L, 2, v2, 3);
  dist = vdist_c(v1, v2);
  lua_pushnumber(L, dist);
  return 1;
}

SPICE_DEFUN(spice_vnorm)
{
  double v1[3];
  double norm;
  spice_lua_tovector(L, 1, v1, 3);
  norm = vnorm_c(v1);
  lua_pushnumber(L, norm);
  return 1;
}

SPICE_DEFUN(spice_vhat)
{
  double v1[3];
  double v2[3];
  spice_lua_tovector(L, 1, v1, 3);
  vhat_c(v1, v2);
  spice_lua_pushvector(L, v2, 3);
  return 1;
}

SPICE_DEFUN(spice_ucrss)
{
  double v1[3];
  double v2[3];
  double v3[3];
  spice_lua_tovector(L, 1, v1, 3);
  spice_lua_tovector(L, 2, v2, 3);
  ucrss_c(v1, v2, v3);
  spice_lua_pushvector(L, v3, 3);
  return 1;
}

SPICE_DEFUN(spice_unorm)
{
  double v1[3];
  double v2[3];
  SpiceDouble mag;
  spice_lua_tovector(L, 1, v1, 3);
  unorm_c(v1, v2, &mag);
  spice_lua_pushvector(L, v2, 3);
  lua_pushnumber(L, mag);
  return 2;
}

SPICE_DEFUN(spice_mxm)
{
  double m1[3][3];
  double m2[3][3];
  double m3[3][3];
  spice_lua_tomatrix(L, 1, (double *)m1, 3, 3);
  spice_lua_tomatrix(L, 2, (double *)m2, 3, 3);
  mxm_c(m1, m2, m3);
  spice_lua_pushmatrix(L, (double *)m3, 3, 3);
  return 1;
}

SPICE_DEFUN(spice_mxmt)
{
  double m1[3][3];
  double m2[3][3];
  double m3[3][3];
  spice_lua_tomatrix(L, 1, (double *)m1, 3, 3);
  spice_lua_tomatrix(L, 2, (double *)m2, 3, 3);
  mxmt_c(m1, m2, m3);
  spice_lua_pushmatrix(L, (double *)m3, 3, 3);
  return 1;
}

SPICE_DEFUN(spice_mxv)
{
  double m1[3][3];
  double v1[3];
  double v2[3];
  spice_lua_tomatrix(L, 1, (double *)m1, 3, 3);
  spice_lua_tovector(L, 2, v1, 3);
  mxv_c(m1, v1, v2);
  spice_lua_pushvector(L, v2, 3);
  return 1;
}

SPICE_DEFUN(spice_mtxm)
{
  double m1[3][3];
  double m2[3][3];
  double m3[3][3];
  spice_lua_tomatrix(L, 1, (double *)m1, 3, 3);
  spice_lua_tomatrix(L, 2, (double *)m2, 3, 3);
  mtxm_c(m1, m2, m3);
  spice_lua_pushmatrix(L, (double *)m3, 3, 3);
  return 1;
}

SPICE_DEFUN(spice_mtxv)
{
  double m1[3][3];
  double v1[3];
  double v2[3];
  spice_lua_tomatrix(L, 1, (double *)m1, 3, 3);
  spice_lua_tovector(L, 2, v1, 3);
  mtxv_c(m1, v1, v2);
  spice_lua_pushvector(L, v2, 3);
  return 1;
}

SPICE_DEFUN(spice_vtmv)
{
  double m1[3][3];
  double v1[3];
  double v2[3];
  double value;
  spice_lua_tovector(L, 1, v1, 3);
  spice_lua_tomatrix(L, 2, (double *)m1, 3, 3);
  value = vtmv_c(v1, m1, v2);
  lua_pushnumber(L, value);
  return 1;
}

SPICE_DEFUN(spice_xpose)
{
  double m1[3][3];
  spice_lua_tomatrix(L, 1, (double *)m1, 3, 3);
  xpose_c(m1, m1);
  spice_lua_pushmatrix(L, (double *)m1, 3, 3);
  return 1;
}

SPICE_DEFUN(spice_mequ)
{
  double m1[3][3];
  double m2[3][3];
  spice_lua_tomatrix(L, 1, (double *)m1, 3, 3);
  mequ_c(m1, m2);
  spice_lua_pushmatrix(L, (double *)m2, 3, 3);
  return 1;
}

SPICE_DEFUN(spice_det)
{
  double m1[3][3];
  double value;
  spice_lua_tomatrix(L, 1, (double *)m1, 3, 3);
  value = det_c(m1);
  lua_pushnumber(L, value);
  return 1;
}

SPICE_DEFUN(spice_trace)
{
  double m1[3][3];
  double value;
  spice_lua_tomatrix(L, 1, (double *)m1, 3, 3);
  value = trace_c(m1);
  lua_pushnumber(L, value);
  return 1;
}


// Projecting, Combining and Rotating 3D vectors

SPICE_DEFUN(spice_vperp)
{
  double v1[3];
  double v2[3];
  double v3[3];
  spice_lua_tovector(L, 1, v1, 3);
  spice_lua_tovector(L, 2, v2, 3);
  vperp_c(v1, v2, v3);
  spice_lua_pushvector(L, v3, 3);
  return 1;
}

SPICE_DEFUN(spice_vproj)
{
  double v1[3];
  double v2[3];
  double v3[3];
  spice_lua_tovector(L, 1, v1, 3);
  spice_lua_tovector(L, 2, v2, 3);
  vproj_c(v1, v2, v3);
  spice_lua_pushvector(L, v3, 3);
  return 1;
}

SPICE_DEFUN(spice_vrotv)
{
  double v[3];
  double axis[3];
  double theta;
  double r[3];
  spice_lua_tovector(L, 1, v, 3);
  spice_lua_tovector(L, 2, axis, 3);
  theta = luaL_checknumber(L, 3);
  vrotv_c(v, axis, theta, r);
  spice_lua_pushvector(L, r, 3);
  return 1;
}

SPICE_DEFUN(spice_rotvec)
{
  double v1[3];
  double angle;
  const char* c;
  int iaxis;
  double v2[3];
  spice_lua_tovector(L, 1, v1, 3);
  angle = luaL_checknumber(L, 2);
  c = luaL_checkstring(L, 3);
  iaxis = spice_lua_parse_axis(c);
  rotvec_c(v1, angle, iaxis, v2);
  spice_lua_pushvector(L, v2, 3);
  return 1;
}

/**
 * sum = a*v1 + b*v2
 */
SPICE_DEFUN(spice_vlcom)
{
  double a;
  double v1[3];
  double b;
  double v2[3];
  double sum[3];
  a = luaL_checknumber(L, 1);
  spice_lua_tovector(L, 2, v1, 3);
  b = luaL_checknumber(L, 3);
  spice_lua_tovector(L, 4, v2, 3);
  vlcom_c(a, v1, b, v2, sum);
  spice_lua_pushvector(L, sum, 3);
  return 1;
}

/**
 * sum = a*v1 + b*v2 + c*v3
 */
SPICE_DEFUN(spice_vlcom3)
{
  double a;
  double v1[3];
  double b;
  double v2[3];
  double c;
  double v3[3];
  double sum[3];
  a = luaL_checknumber(L, 1);
  spice_lua_tovector(L, 2, v1, 3);
  b = luaL_checknumber(L, 3);
  spice_lua_tovector(L, 4, v2, 3);
  c = luaL_checknumber(L, 5);
  spice_lua_tovector(L, 6, v2, 3);
  vlcom3_c(a, v1, b, v2, c, v3, sum);
  spice_lua_pushvector(L, sum, 3);
  return 1;
}

// Creating and Converting Transformation Matrices

SPICE_DEFUN(spice_rotate)
{
  double angle;
  const char* axis;
  double iaxis;
  double m[3][3];
  angle = luaL_checknumber(L, 1);
  axis = luaL_checkstring(L, 2);
  iaxis = spice_lua_parse_axis(axis);
  rotate_c(angle, iaxis, m);
  spice_lua_pushmatrix(L, (double *)m, 3, 3);
  return 1;
}

SPICE_DEFUN(spice_rotmat)
{
  double m1[3][3];
  double angle;
  const char* axis;
  double iaxis;
  double m2[3][3];
  spice_lua_tomatrix(L, 1, (double *)m1, 3, 3);
  angle = luaL_checknumber(L, 2);
  axis = luaL_checkstring(L, 3);
  iaxis = spice_lua_parse_axis(axis);
  rotmat_c(m1, angle, iaxis, m2);
  spice_lua_pushmatrix(L, (double *)m2, 3, 3);
  return 1;
}

SPICE_DEFUN(spice_twovec)
{
  double axdef[3];
  int indexa;
  double plndef[3];
  int indexp;
  double m[3][3];
  spice_lua_tovector(L, 1, axdef, 3);
  indexa = luaL_checknumber(L, 2);
  spice_lua_tovector(L, 3, plndef, 3);
  indexp = luaL_checknumber(L, 2);
  twovec_c(axdef, indexa, plndef, indexp, m);
  spice_lua_pushmatrix(L, (double *)m, 3, 3);
  return 1;
}

SPICE_DEFUN(spice_eul2m)
{
  double angle3, angle2, angle1;
  const char* s;
  int axis3, axis2, axis1;
  double r[3][3];
  angle3 = luaL_checknumber(L, 1);
  angle2 = luaL_checknumber(L, 2);
  angle1 = luaL_checknumber(L, 3);
  s = luaL_checkstring(L, 4);
  axis3 = spice_lua_parse_axis(s);
  s = luaL_checkstring(L, 5);
  axis2 = spice_lua_parse_axis(s);
  s = luaL_checkstring(L, 6);
  axis1 = spice_lua_parse_axis(s);
  eul2m_c(angle3, angle2, angle1, axis3, axis2, axis1, r);
  spice_lua_pushmatrix(L, (double *)r, 3, 3);
  return 1;
}

SPICE_DEFUN(spice_m2eul)
{
  double m[3][3];
  const char* s;
  int axis3, axis2, axis1;
  double angle3, angle2, angle1;
  spice_lua_tomatrix(L, 1, (double *)m, 3, 3);
  s = luaL_checkstring(L, 2);
  axis3 = spice_lua_parse_axis(s);
  s = luaL_checkstring(L, 3);
  axis2 = spice_lua_parse_axis(s);
  s = luaL_checkstring(L, 4);
  axis1 = spice_lua_parse_axis(s);
  m2eul_c(m, axis3, axis2, axis1, &angle3, &angle2, &angle1);
  lua_pushnumber(L, angle3);
  lua_pushnumber(L, angle2);
  lua_pushnumber(L, angle1);
  return 3;
}

SPICE_DEFUN(spice_raxisa)
{
  double m[3][3];
  double ax[3];
  double angle;
  spice_lua_tomatrix(L, 1, (double *)m, 3, 3);
  raxisa_c(m, ax, &angle);
  spice_lua_pushvector(L, ax, 3);
  lua_pushnumber(L, angle);
  return 1;
}

SPICE_DEFUN(spice_axisar)
{
  double ax[3];
  double angle;
  double r[3][3];
  spice_lua_tovector(L, 1, ax, 3);
  angle = luaL_checknumber(L, 2);
  axisar_c(ax, angle, r);
  spice_lua_pushmatrix(L, (double *)r, 3, 3);
  return 1;
}

SPICE_DEFUN(spice_m2q)
{
  double r[3][3];
  double q[4];
  spice_lua_tomatrix(L, 1, (double *)r, 3, 3);
  m2q_c(r, q);
  spice_lua_pushvector(L, q, 4);
  return 1;
}

SPICE_DEFUN(spice_q2m)
{
  double q[4];
  double r[3][3];
  spice_lua_tovector(L, 1, q, 4);
  q2m_c(q, r);
  spice_lua_pushmatrix(L, (double *)r, 3, 3);
  return 1;
}

// Accessing Physical and Mathematical Constants

SPICE_DEFUN(spice_halfpi)
{
  lua_pushnumber(L, halfpi_c());
  return 1;
}

SPICE_DEFUN(spice_pi)
{
  lua_pushnumber(L, pi_c());
  return 1;
}

SPICE_DEFUN(spice_twopi)
{
  lua_pushnumber(L, twopi_c());
  return 1;
}

SPICE_DEFUN(spice_dpr)
{
  lua_pushnumber(L, dpr_c());
  return 1;
}

SPICE_DEFUN(spice_rpd)
{
  lua_pushnumber(L, rpd_c());
  return 1;
}

SPICE_DEFUN(spice_spd)
{
  lua_pushnumber(L, spd_c());
  return 1;
}

SPICE_DEFUN(spice_jyear)
{
  lua_pushnumber(L, jyear_c());
  return 1;
}

SPICE_DEFUN(spice_tyear)
{
  lua_pushnumber(L, tyear_c());
  return 1;
}

SPICE_DEFUN(spice_clight)
{
  lua_pushnumber(L, clight_c());
  return 1;
}

SPICE_DEFUN(spice_b1900)
{
  lua_pushnumber(L, b1900_c());
  return 1;
}

SPICE_DEFUN(spice_b1950)
{
  lua_pushnumber(L, b1950_c());
  return 1;
}

SPICE_DEFUN(spice_j1900)
{
  lua_pushnumber(L, j1900_c());
  return 1;
}

SPICE_DEFUN(spice_j1950)
{
  lua_pushnumber(L, j1950_c());
  return 1;
}

SPICE_DEFUN(spice_j2000)
{
  lua_pushnumber(L, j2000_c());
  return 1;
}

SPICE_DEFUN(spice_j2100)
{
  lua_pushnumber(L, j2100_c());
  return 1;
}

SPICE_DEFUN(spice_convrt)
{
  double x, y;
  const char* in;
  const char* out;
  x   = luaL_checknumber(L, 1);
  in  = luaL_checkstring(L, 2);
  out = luaL_checkstring(L, 3);
  convrt_c(x, in, out, &y);
  lua_pushnumber(L, y);
  return 1;
}

void spice_lua_init(lua_State* L)
{
  // init spice_cell_lua
  spice_cell_lua_init(L);

  //
  // accessing SPICE kernel data
  //

  // Loading and Unloading SPICE Kernels
  SPICE_REGFUN(L, spice_furnsh);
  SPICE_REGFUN(L, spice_unload);
  // Converting between UTC and Ephemeris Time (LSK)
  SPICE_REGFUN(L, spice_str2et);
  SPICE_REGFUN(L, spice_timout);
  SPICE_REGFUN(L, spice_et2utc);
  SPICE_REGFUN(L, spice_et2lst);
  // Converting between Ephemeris Time and Spacecraft Clock (SCLK)
  SPICE_REGFUN(L, spice_scs2e);
  SPICE_REGFUN(L, spice_sce2s);
  SPICE_REGFUN(L, spice_sct2e);
  SPICE_REGFUN(L, spice_sce2c);
  SPICE_REGFUN(L, spice_scencd);
  SPICE_REGFUN(L, spice_scdecd);
  // Constants and Orientation for Natural Bodies (PCK)
  SPICE_REGFUN(L, spice_bodfnd);
  SPICE_REGFUN(L, spice_bodvrd);
  SPICE_REGFUN(L, spice_pxform);
  SPICE_REGFUN(L, spice_sxform);
  SPICE_REGFUN(L, spice_pckfrm);
  SPICE_REGFUN(L, spice_pckcov);
  // Computing Positions of Spacecraft and Natural Bodies (SPK)
  SPICE_REGFUN(L, spice_spkezr);
  SPICE_REGFUN(L, spice_spkpos);
  SPICE_REGFUN(L, spice_spkobj);
  SPICE_REGFUN(L, spice_spkcov);
  // Computing Orientation for Spacecraft and Instruments (CK)
  SPICE_REGFUN(L, spice_ckobj);
  SPICE_REGFUN(L, spice_ckcov);
  SPICE_REGFUN(L, spice_ckgp);
  SPICE_REGFUN(L, spice_ckgpav);
  // Retrieving instrument parameters (IK)
  SPICE_REGFUN(L, spice_getfov);
  SPICE_REGFUN(L, spice_gdpool);
  SPICE_REGFUN(L, spice_gipool);
  SPICE_REGFUN(L, spice_gcpool);
  // Mapping Between Object Names and NAIF IDs
  SPICE_REGFUN(L, spice_bodc2n);
  SPICE_REGFUN(L, spice_bodn2c);

  //
  // computing derived geometry
  //

  // Computing Planetocentric, Planetodetic, and Planetographic Coordinates
  SPICE_REGFUN(L, spice_reclat);
  SPICE_REGFUN(L, spice_latrec);
  SPICE_REGFUN(L, spice_srfrec);
  SPICE_REGFUN(L, spice_recgeo);
  SPICE_REGFUN(L, spice_georec);
  SPICE_REGFUN(L, spice_recpgr);
  SPICE_REGFUN(L, spice_pgrrec);
  // Computing Surface Intercept Point
  SPICE_REGFUN(L, spice_sincpt);
  // Computing Sub-observer and Sub-solar Points
  SPICE_REGFUN(L, spice_subpnt);
  SPICE_REGFUN(L, spice_subslr);
  // Computing Illumination Angles
  SPICE_REGFUN(L, spice_ilumin);
  // Computing and Propagating Orbital Elements
  SPICE_REGFUN(L, spice_conics);
  SPICE_REGFUN(L, spice_oscelt);

  //
  // finding times of geometric events
  //
  // TODO: gfdist, gfposc, gfsep ...

  //
  // coordinate conversions
  //
  SPICE_REGFUN(L, spice_recrad);
  SPICE_REGFUN(L, spice_radrec);
  SPICE_REGFUN(L, spice_recsph);
  SPICE_REGFUN(L, spice_sphrec);
  SPICE_REGFUN(L, spice_reccyl);
  SPICE_REGFUN(L, spice_cylrec);
  SPICE_REGFUN(L, spice_sphcyl);
  SPICE_REGFUN(L, spice_cylsph);
  SPICE_REGFUN(L, spice_sphlat);
  SPICE_REGFUN(L, spice_latsph);
  SPICE_REGFUN(L, spice_cyllat);
  SPICE_REGFUN(L, spice_latcyl);

  //
  // operations with 3D vectors and matrices
  //

  // Performing simple operations on 3D vectors
  SPICE_REGFUN(L, spice_vadd);
  SPICE_REGFUN(L, spice_vsub);
  SPICE_REGFUN(L, spice_vcrss);
  SPICE_REGFUN(L, spice_vdot);
  SPICE_REGFUN(L, spice_vrel);
  SPICE_REGFUN(L, spice_vscl);
  SPICE_REGFUN(L, spice_vminus);
  SPICE_REGFUN(L, spice_vequ);
  SPICE_REGFUN(L, spice_vzero);
  SPICE_REGFUN(L, spice_vsep);
  SPICE_REGFUN(L, spice_vdist);
  SPICE_REGFUN(L, spice_vnorm);
  SPICE_REGFUN(L, spice_vhat);
  SPICE_REGFUN(L, spice_ucrss);
  SPICE_REGFUN(L, spice_unorm);

  // Performing Simple Operations on 3x3 Matrices
  SPICE_REGFUN(L, spice_mxm);
  SPICE_REGFUN(L, spice_mxmt);
  SPICE_REGFUN(L, spice_mxv);
  SPICE_REGFUN(L, spice_mtxm);
  SPICE_REGFUN(L, spice_mtxv);
  SPICE_REGFUN(L, spice_vtmv);
  SPICE_REGFUN(L, spice_xpose);
  SPICE_REGFUN(L, spice_mequ);
  SPICE_REGFUN(L, spice_det);
  SPICE_REGFUN(L, spice_trace);

  // Projecting, Combining and Rotating 3D vectors
  SPICE_REGFUN(L, spice_vperp);
  SPICE_REGFUN(L, spice_vproj);
  SPICE_REGFUN(L, spice_vrotv);
  SPICE_REGFUN(L, spice_rotvec);
  SPICE_REGFUN(L, spice_vlcom);
  SPICE_REGFUN(L, spice_vlcom3);

  // Creating and Converting Transformation Matrices
  SPICE_REGFUN(L, spice_rotate);
  SPICE_REGFUN(L, spice_rotmat);
  SPICE_REGFUN(L, spice_twovec);
  SPICE_REGFUN(L, spice_eul2m);
  SPICE_REGFUN(L, spice_m2eul);
  SPICE_REGFUN(L, spice_raxisa);
  SPICE_REGFUN(L, spice_axisar);
  SPICE_REGFUN(L, spice_m2q);
  SPICE_REGFUN(L, spice_q2m);

  // Accessing Physical and Mathematical Constants
  SPICE_REGFUN(L, spice_halfpi);
  SPICE_REGFUN(L, spice_pi);
  SPICE_REGFUN(L, spice_twopi);
  SPICE_REGFUN(L, spice_dpr);
  SPICE_REGFUN(L, spice_rpd);
  SPICE_REGFUN(L, spice_spd);
  SPICE_REGFUN(L, spice_jyear);
  SPICE_REGFUN(L, spice_tyear);
  SPICE_REGFUN(L, spice_clight);
  SPICE_REGFUN(L, spice_b1900);
  SPICE_REGFUN(L, spice_b1950);
  SPICE_REGFUN(L, spice_j1900);
  SPICE_REGFUN(L, spice_j1950);
  SPICE_REGFUN(L, spice_j2000);
  SPICE_REGFUN(L, spice_j2100);

  SPICE_REGFUN(L, spice_convrt);
}
