/**
 * @file      stl.c
 * @brief     STL model loader
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2011 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include "stl.h"
#include "strb.h"

static STLBool stl_parse_data_binary(shape_model* self, const char* data);

extern STLBool stl_parse_data_text(shape_model* self, const char* data);

STLBool stl_parse_file(shape_model* self, const char* path)
{
  FILE* fh = fopen(path, "rb");
  if (!fh) {
    return STL_FALSE;
  }
  return stl_parse_std(self, fh);
}

STLBool stl_parse_std(shape_model* self, FILE* io)
{
  char magic[6];
  char header[256];
  size_t read;
  strb* str;
  enum STLFormatType format = STLFormatTypeText;

  memset(magic, 0, sizeof(magic));
  read = fread(magic, 1, 5, io);
  if (5 != read) {
    fprintf(stderr, "too short file.\n");
    return STL_FALSE;
  }
  if (0 != strncmp(magic, "solid", 5)) {
    format = STLFormatTypeBinary;
  }

  if (format == STLFormatTypeText) {
    /* header */
    str = strb_create(256);
    strb_set_cstring(str, magic);
    if (!strb_append_fgets(str, io)) {
      strb_free(str);
      return STL_FALSE;
    }
    /* parse body */
    stl_parse_std_text(self, strb_ptr(str), io);
    strb_free(str);
  }
  else {
    /* header */
    memcpy(header, magic, 5);
    read = fread(&header[5], 1, 75, io);
    if (75 != read) {
      return STL_FALSE;
    }
    /* parse body */
    if (!stl_parse_std_binary(self, header, io)) {
      return STL_FALSE;
    }
  }
  return STL_TRUE;
}


STLBool stl_parse_data(shape_model* self, const char* data)
{
  enum STLFormatType format = STLFormatTypeText;

  if (0 != strncmp(data, "solid", 5)) {
    format = STLFormatTypeBinary;
  }

  if (format == STLFormatTypeText) {
    stl_parse_data_text(self, data);
  }
  else {
    stl_parse_data_binary(self, data);
  }

  return STL_TRUE;
}

STLBool stl_parse_std_text(shape_model* self, const char* readed, FILE* io)
{
  strb* str = strb_create(8192);
  const char* p;
  size_t count = 0;
  strb_set_cstring(str, readed);
  strb_append_fread(str, io);
  p = strb_ptr(str);
  while (NULL != (p = strstr(p, "normal"))) {
    p += 6;
    count++;
  }
  shape_extend(self, count);
  stl_parse_data_text(self, strb_ptr(str));
  strb_free(str);
  return STL_TRUE;
}

STLBool stl_parse_std_binary(shape_model* self, const char* head, FILE* io)
{
  shapeTriangle tri;
  uint32_t i, num;

  /* num of triangles */
  fread(&num, sizeof(num), 1, io);

  shape_extend(self, num);
  /* normals and vertices */
  for (i = 0; i < num; i++) {
    fread(&tri.normal.x, sizeof(float), 1, io);
    fread(&tri.normal.y, sizeof(float), 1, io);
    fread(&tri.normal.z, sizeof(float), 1, io);
    fread(&tri.va.x, sizeof(float), 1, io);
    fread(&tri.va.y, sizeof(float), 1, io);
    fread(&tri.va.z, sizeof(float), 1, io);
    fread(&tri.vb.x, sizeof(float), 1, io);
    fread(&tri.vb.y, sizeof(float), 1, io);
    fread(&tri.vb.z, sizeof(float), 1, io);
    fread(&tri.vc.x, sizeof(float), 1, io);
    fread(&tri.vc.y, sizeof(float), 1, io);
    fread(&tri.vc.z, sizeof(float), 1, io);
    fread(&tri.attr, sizeof(unsigned short), 1, io);
    self->triangle[i] = tri;
  }
  return STL_TRUE;
}

static STLBool stl_parse_data_binary(shape_model* self, const char* data)
{
  uint32_t i, num;
  shapeTriangle tri;
  const char* p = data;
  /* num of triangles */
  p += 80;
  memcpy(&num, p, sizeof(num));
  shape_extend(self, num);
  /* normals and vertices */
  for (i = 0; i < num; i++) {
    memcpy(&tri.normal.x, p, sizeof(float)); p += sizeof(float);
    memcpy(&tri.normal.y, p, sizeof(float)); p += sizeof(float);
    memcpy(&tri.normal.z, p, sizeof(float)); p += sizeof(float);
    memcpy(&tri.va.x, p, sizeof(float)); p += sizeof(float);
    memcpy(&tri.va.y, p, sizeof(float)); p += sizeof(float);
    memcpy(&tri.va.z, p, sizeof(float)); p += sizeof(float);
    memcpy(&tri.vb.x, p, sizeof(float)); p += sizeof(float);
    memcpy(&tri.vb.y, p, sizeof(float)); p += sizeof(float);
    memcpy(&tri.vb.z, p, sizeof(float)); p += sizeof(float);
    memcpy(&tri.vc.x, p, sizeof(float)); p += sizeof(float);
    memcpy(&tri.vc.y, p, sizeof(float)); p += sizeof(float);
    memcpy(&tri.vc.z, p, sizeof(float)); p += sizeof(float);
    memcpy(&tri.attr, p, sizeof(unsigned short)); p += sizeof(unsigned short);
    self->triangle[i] = tri;
  }
  return STL_TRUE;
}

