/**
 * @file      stl.h
 * @brief     STL model loader
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2011 Japan Aerospace Exploration Agency
 *
 */

#ifndef _STL_H_
#define _STL_H_

#include <stdio.h>
#include <stdint.h>
#include "shape_model.h"

typedef int STLBool;
#define STL_TRUE  1
#define STL_FALSE 0

enum STLFormatType {
  STLFormatTypeText,
  STLFormatTypeBinary
};

__BEGIN_DECLS

/**
 * @brief Load STL file (both binary/text format)
 *
 * @param[in]  self  STL instance
 * @param[in]  path  filepath
 * @return     Upon successful completion, stl_parse_file() returns STL_TRUE.
 *             Otherwise, STL_FALSE is returned.
 */
STLBool stl_parse_file(shape_model* self, const char* path);

/**
 * @brief Load STL data from FILE pointer
 *
 * @param[in]  self  STL instance
 * @param[in]  io    FILE pointer
 * @return     Upon successful completion, stl_parse_std() returns STL_TRUE.
 *             Otherwise, STL_FALSE is returned.
 */
STLBool stl_parse_std(shape_model* self, FILE* io);

/**
 * @brief Load STL data from memory
 *
 * @param[in]  self  STL instance
 * @param[in]  data  STL memory data
 * @return Upon successful completion, stl_parse_data() returns STL_TRUE.
 *         Otherwise, STL_FALSE is returned.
 */
STLBool stl_parse_data(shape_model* self, const char* data);

/**
 * @brief Load STL TEXT data from FILE pointer.
 *        This function is internal use only.
 *        Use stl_parse_std() instead.
 *
 * @param[in]  self    STL instance
 * @param[in]  readed  already read data
 * @return Upon successful completion, stl_parse_std_text() returns STL_TRUE.
 *         Otherwise, STL_FALSE is returned.
 * @see stl_parse_std()
 */
STLBool stl_parse_std_text(shape_model* self, const char* readed, FILE* io);

/**
 * @brief Load STL Binary data from FILE pointer.
 *        This function is internal use only.
 *        Use stl_parse_std() instead.
 *
 * @param[in]  self  STL instance
 * @param[in]  head  STL header (80 bytes)
 * @param[in]  io    FILE pointer
 * @return Upon successful completion, stl_parse_std_binary() returns STL_TRUE.
 *         Otherwise, STL_FALSE is returned.
 * @see stl_parse_std()
 */
STLBool stl_parse_std_binary(shape_model* self, const char* head, FILE* io);

__END_DECLS

#endif /* _STL_H_ */
