/**
 * @file      strb.c
 * @brief     Flexible length c-string
 * @date      2010-03-12
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#ifdef ASPRINTF_NEED_GNU_SOURCE
#  define _GNU_SOURCE
#endif
#include <stdio.h>
#include "strb.h"

static char _urlencode_table[] = {
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,
  1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,0,
  1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,1,1,1,0,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
};
static char _urldecode_table[] = {
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,
  0,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
};

ANY_BUFFER_DEFINE_FUNCTION(strb, char)
ANY_BUFFER_DEFINE_FUNCTION(strbary, strbPtr)

strb *strb_create_with_cstring(const char* str)
{
  return strb_create_with_data(str, strlen(str)+1);
}

strb* strb_create_substring(strb* self, size_t start, size_t len)
{
  strb* str = strb_create(strb_size(self));
  char* ptr = strb_ptr(self);
  char* s = "";
  if (self->size-1 < start) {
    return str;
  }
  if (self->size < start+len+1) {
    len = self->size-1-start;
  }
  strb_append(str, ptr+start, len);
  strb_append(str, s, 1);
  return str;
}

strb* strb_create_sprintf(const char* format, ...)
{
  va_list ap;
  va_start(ap, format);
  return strb_create_vsprintf(format, ap);
}

strb* strb_create_vsprintf(const char* format, va_list ap)
{
  char* str = NULL;
  strb* self = NULL;
  vasprintf(&str, format, ap);
  if (!str) {
    return NULL;
  }
  self = strb_create_with_cstring(str);
  free(str);
  return self;
}

strb* strb_convert_urlencode(strb* self)
{
  strb* str = NULL;
  unsigned char* p = strb_ptr(self);
  size_t len = 0;
  char es[4];
  while (*p) {
    len += (_urlencode_table[*p]) ? 3 : 1;
    p++;
  }
  str = strb_create(len+1);
  if (!str) {
    return NULL;
  }
  p = strb_ptr(self);
  while (*p) {
    if (_urlencode_table[*p]) {
      sprintf(es, "%%%02X", (int)*p);
      strb_append(str, es, 3);
    }
    else if (*p == ' ') {
      strb_append(str, "+", 1);
    }
    else {
      strb_append(str, (char *)p, 1);
    }
    p++;
  }
  es[0] = '\0';
  strb_append(str, es, 1);
  return str;
}

strb* strb_convert_urldecode(strb* self)
{
  strb* str = NULL;
  unsigned char* p = NULL;
  char* p2 = NULL;
  size_t len = 0;
  unsigned char c;
  p = strb_ptr(self);
  while (*p) {
    len++;
    if (*p=='%' && _urldecode_table[p[1]] && _urldecode_table[p[2]]) {
      p += 3;
    }
    else {
      p++;
    }
  }
  str = strb_create(len+1);
  if (!str) {
    return NULL;
  }
  p = strb_ptr(self);
  while (*p) {
    if (*p=='%' && _urldecode_table[p[1]] && _urldecode_table[p[2]]) {
      p++;
      c = (unsigned char)strtol((char *)p, &p2, 16);
      strb_append(str, (char *)&c, 1);
      p += 2;
    }
    else if (*p == '+') {
      strb_append(str, " ", 1);
    }
    else {
      strb_append(str, (char *)p, 1);
      p++;
    }
  }
  c = '\0';
  strb_append(str, (char *)&c, 1);
  return str;
}

int strb_set_cstring(strb* self, const char* str)
{
  return strb_set(self, str, strlen(str)+1);
}

int strb_sprintf(strb* self, const char* format, ...)
{
  va_list ap;
  va_start(ap, format);
  return strb_vsprintf(self, format, ap);
}

int strb_vsprintf(strb* self, const char* format, va_list ap)
{
  char* str = NULL;
  int err;
  vasprintf(&str, format, ap);
  if (!str) return 0;
  err = strb_set_cstring(self, str);
  free(str);
  return err;
}

int strb_append_cstring(strb* self, const char* str)
{
  strb_truncate(self, self->size-1);
  return strb_append(self, str, strlen(str)+1);
}

int strb_append_sprintf(strb* self, const char* format, ...)
{
  va_list ap;
  va_start(ap, format);
  return strb_append_vsprintf(self, format, ap);
}

int strb_append_vsprintf(strb* self, const char* format, va_list ap)
{
  char* str = NULL;
  int err;
  vasprintf(&str, format, ap);
  if (!str) return 0;
  err = strb_append_cstring(self, str);
  free(str);
  return err;
}

int strb_truncate_cstring(strb* self, size_t size)
{
  char* s = "";
  strb_truncate(self, size);
  return strb_append(self, s, 1);
}

int strb_fgets(strb* self, FILE* io)
{
  char c[] = {' ', '\0'};
  size_t r;
  size_t count = 0;
  while (0 < (r = fread(c, 1, 1, io))) {
    if (0 == count) {
      strb_set_cstring(self, c);
    }
    else {
      strb_append_cstring(self, c);
    }
    count++;
    if (c[0] == '\n') {
      break;
    }
  }
  return 1;
}

int strb_append_fgets(strb* self, FILE* io)
{
  char c[] = {' ', '\0'};
  size_t r;
  while (0 < (r = fread(c, 1, 1, io))) {
    strb_append_cstring(self, c);
    if (c[0] == '\n') {
      break;
    }
  }
  return 1;
}

int strb_fread(strb* self, FILE* io)
{
  char c[1025];
  size_t r;
  size_t count = 0;
  c[1024] = '\0';
  while (0 < (r = fread(c, 1, 1024, io))) {
    c[r] = '\0';
    if (0 == count) {
      strb_set_cstring(self, c);
    }
    else {
      strb_append_cstring(self, c);
    }
    count++;
  }
  return 1;
}

int strb_append_fread(strb* self, FILE* io)
{
  char c[1025];
  size_t r;
  c[1024] = '\0';
  while (0 < (r = fread(c, 1, 1024, io))) {
    c[r] = '\0';
    strb_append_cstring(self, c);
  }
  return 1;
}

void strbary_free_items(strbary* self)
{
  strbary_each(self, strb_free);
}
