/**
 * @file      strb.h
 * @brief     Flexible length c-string and its array
 * @date      2010-03-12
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

#ifndef _STRB_H
#define _STRB_H

#include <stdarg.h>
#include "any_buffer.h"

/** any_buffer の文字列拡張。*/
ANY_BUFFER_DEFINE_STRUCT(strb, char)

/** strb のポインタ型。strbary の要素として利用。*/
typedef strb *  strbPtr;

/**
 * 各要素が strb の配列。
 * 配列を解放する前に必ず要素を解放する strbary_free_items() を呼び出し
 * てください。
 */
ANY_BUFFER_DEFINE_STRUCT(strbary, strbPtr)

__BEGIN_DECLS
  ANY_BUFFER_DECLARE_FUNCTION(strb, char)

/**
 * @brief Create instance and set C string
 *
 * @param[in]  str C string. The string must be terminated by \0.
 * @return Upon successful completion, strb_create_with_cstring() returns strb string.
 *         Otherwise, NULL is returned.
 */
strb *strb_create_with_cstring(const char* str);

/**
 * @brief Create strb instance and set a formatted C string
 * @param[in]  format Specifies the string and how to format the variables in it
 * @param[in]  ...    Argument(s) to be inserted n-th %-sign in the format string
 * @result Upon successful completion, strb_create_sprintf() returns a formatted
 *         strb string. Otherwise, NULL is returned.
 */
strb* strb_create_sprintf(const char* format, ...);

/**
 * @brief Create strb instance and set a formatted C string
 *
 * @param[in] format formatted string
 * @param[in] ap     A value identifying a variable arguments list initialized with va_start
 * @result Upon successful completion, strb_create_vsprintf() returns a formatted strb
 *         string. Otherwise, NULL is returned.
 */
strb* strb_create_vsprintf(const char* format, va_list ap);

/**
 * @brief Extracts the characters from a strb string
 *
 * @param[in] self  strb instance
 * @param[in] start Position where to start the extraction.
 * @param[in] len   Length of the extracted characters
 * @return strb_create_substring() returns the instance of the extracted string.
 *         strb_free() must be called when the instance is unnecessary.
 */
strb* strb_create_substring(strb* self, size_t start, size_t len);

/**
 * @brief Create URL encoded string from strb string
 *
 * @param[in] self  strb instance
 * @return strb_convert_urlencode() returns the instance of URL encoded string.
 *         strb_free() must be called when the returned instance is unnecessary.
 */
strb* strb_convert_urlencode(strb* self);

/**
 * @brief Create URL decoded string from strb string
 *
 * @param[in] self  strb instance
 * @result strb_convert_urldecode() returns the instance of URL decoded string.
 *         strb_free() must be called when the returned instance is unnecessary.
 */
strb* strb_convert_urldecode(strb* self);

/**
 * @brief Set C string into strb instance
 *
 * @param[in] self  strb instance
 * @param[in] str   C string terminated by \0.
 * @return Upon successful completion, strb_set_cstring() return 1.
 *         Otherwise, 0 is returned.
 */
int strb_set_cstring(strb* self, const char* str);

/**
 * @brief Set formatted C string into strb instance
 *
 * @param[in] format format string
 * @param[in] ...    Argument(s) to be inserted n-th %-sign in the format string
 * @return Upon successful completion, strb_sprintf() return 1.
 *         Otherwise, 0 is returned.
 */
int strb_sprintf(strb* self, const char* format, ...);

/**
 * @brief Set formatted C string into strb instance
 *
 * @param[in] format formatted string
 * @param[in] ap     A value identifying a variable arguments list initialized with va_start
 * @result Upon successful completion, strb_vsprintf() returns 1.
 *         Otherwise, 0 is returned.
 */
int strb_vsprintf(strb* self, const char* format, va_list ap);

/**
 * @brief Extends the string by appending additional characters at the end of its current value
 *
 * @param[in] self  strb instance
 * @param[in] str   C string terminated by \0
 * @return Upon successful completion, strb_append_cstring() returns 1.
 *         Otherwise, 0 is returned.
 */
int strb_append_cstring(strb* self, const char* str);

/**
 * @brief Extends the string by appending a formatted string at the end of its current value
 *
 * @param[in] self   strb instance
 * @param[in] format Specifies the string and how to format the variables in it
 * @param[in] ...    Argument(s) to be inserted n-th %-sign in the format string
 * @return Upon successful completion, strb_append_sprintf() returns 1.
 *         Otherwise, 0 is returned.
 */
int strb_append_sprintf(strb* self, const char* format, ...);

/**
 * @brief Extends the string by appending a formatted string at the end of its current value
 *
 * @param[in] self   strb instance
 * @param[in] format Specifies the string and how to format the variables in it
 * @param[in] ap     A value identifying a variable arguments list initialized with va_start
 * @return Upon successful completion, strb_append_vsprintf() returns 1.
 *         Otherwise, 0 is returned.
 */
int strb_append_vsprintf(strb* self, const char* format, va_list ap);

/**
 * @brief Truncate strb string. Unlike strb_truncate(), the string is terminated by \0.
 *
 * @param[in] self  strb instance
 * @param[in] size  length of the truncated string
 * @return Upon successful completion, strb_truncate_cstring() returns 1.
 *         Otherwise, 0 is returned.
 */
int strb_truncate_cstring(strb* self, size_t size);

/**
 * @brief fgets(3) interface with unlimited string length.
 *        The current string is overwritten.
 *
 * @param[in] self  strb instance.
 * @param[in] io    file handle.
 * @return Upon successful completion, strb_fgets() returns 1.
 *         Otherwise, 0 is returned.
 */
int strb_fgets(strb* self, FILE* io);

/**
 * @brief fgets(3) interface with unlimited string length.
 * Extends the string by appending characters at the end of its current value.
 *
 * @param[in] self  strb instance
 * @param[in] io    file handle
 * @return Upon successful completion, strb_append_fgets() returns 1.
 *         Otherwise, 0 is returned.
 */
int strb_append_fgets(strb* self, FILE* io);

/**
 * @brief fread(3) interface with unlimited string length.
 *        The current string is overwritten.
 * @param self  strb instance
 * @param io    file handle.
 * @return Upon successful completion, strb_fread() returns 1.
 *         Otherwise, 0 is returned.
 */
int strb_fread(strb* self, FILE* io);

/**
 * @brief fread(3) interface with unlimited string length.
 *        Extends the string by appending additional characters at the end of its current value.
 *
 * @param self  strb instance
 * @param io    file handle.
 * @return Upon successful completion, strb_append_fread() returns 1.
 *         Otherwise, 0 is returned.
 */
int strb_append_fread(strb* self, FILE* io);

ANY_BUFFER_DECLARE_FUNCTION(strbary, strbPtr)

/**
 * @brief Free each elements of array
 *
 * @param self  strbary instance
 */
void strbary_free_items(strbary* self);
__END_DECLS

#endif /* _STRB_H */
