/**
 * @file      sv_doc.c
 * @brief     svdoc structure
 * @date      2010-03-12
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include "sv_doc.h"
#include "flow_common.h"

////////////////////////////////////////////////////////////////////////
// SV_HEAD
////////////////////////////////////////////////////////////////////////

// constructor
SV_HEAD* new_sv_head()
{
  SV_HEAD* sv_head = (SV_HEAD *)malloc(sizeof(SV_HEAD));
  if (sv_head == NULL) {
    fprintf(stderr, "%s: insufficient memory\n", app_name);
    return NULL;
  }
  sv_head->title = NULL;

  return sv_head;
}

// destructor
void destroy_sv_head(SV_HEAD* sv_head)
{
  if (sv_head != NULL) {
    if (sv_head->title != NULL) {
      free(sv_head->title);
    }
    free(sv_head);
  }
}

////////////////////////////////////////////////////////////////////////
// SV_VIEW
////////////////////////////////////////////////////////////////////////

// constructor
SV_VIEW* new_sv_view()
{
  SV_VIEW* sv_view = (SV_VIEW *)malloc(sizeof(SV_VIEW));
  if (sv_view == NULL) {
    fprintf(stderr, "%s: insufficient memory\n", app_name);
    return NULL;
  }

  sv_view->date = NULL;

  sv_view->location[0] = 0.0;
  sv_view->location[1] = 0.0;
  sv_view->location[2] = 0.0;

  sv_view->boresight[0] = 0.0;
  sv_view->boresight[1] = 0.0;
  sv_view->boresight[2] = 0.0;

  sv_view->nbounds = 0;
  sv_view->bounds = NULL;

  sv_view->fov       = 0.0;
  sv_view->pos_angle = 0.0;
  sv_view->angle_res = 1.0;
  sv_view->shape     = NULL;

  sv_view->center[0] = 0.0;
  sv_view->center[1] = 0.0;

  sv_view->image_size[0] = -1;
  sv_view->image_size[1] = -1;

  return sv_view;
}

BOOL sv_view_add_bound(SV_VIEW* self, const double* vec)
{
  double* new_bounds;
  int index = self->nbounds;
  int count = index + 1;

  if ((new_bounds = realloc(self->bounds, count * sizeof(double) * 3)) == NULL) {
    return FALSE;
  }
  self->bounds = new_bounds;
  self->nbounds = count;

  self->bounds[index*3  ] = vec[0];
  self->bounds[index*3+1] = vec[1];
  self->bounds[index*3+2] = vec[2];

  return TRUE;
}

void sv_view_get_bound(SV_VIEW* self, int index, double* vec)
{
  assert(0 <= index && index <= self->nbounds);
  vec[0] = self->bounds[index*3  ];
  vec[1] = self->bounds[index*3+1];
  vec[2] = self->bounds[index*3+2];
}

// destructor
void destroy_sv_view(SV_VIEW* sv_view)
{
  if (sv_view != NULL) {
    if (sv_view->date != NULL) {
      free(sv_view->date);
    }
    if (sv_view->shape != NULL) {
      free(sv_view->shape);
    }
    if (sv_view->bounds != NULL) {
      free(sv_view->bounds);
    }
    free(sv_view);
  }
}

////////////////////////////////////////////////////////////////////////
// SV_OBJECT
////////////////////////////////////////////////////////////////////////

// constructor
SV_OBJECT* new_sv_object(int type)
{
  int i, j;

  SV_OBJECT* sv_object = (SV_OBJECT *)malloc(sizeof(SV_OBJECT));
  if (sv_object == NULL) {
    fprintf(stderr, "%s: insufficient memory\n", app_name);
    return NULL;
  }

  sv_object->type = type;
  sv_object->id = NULL;
  sv_object->name = NULL;
  sv_object->position[0] = 0.0;
  sv_object->position[1] = 0.0;
  sv_object->position[2] = 0.0;
  sv_object->magnitude = 0.0;
  sv_object->distance = -1.0;
  sv_object->texture_count = 0;
  sv_object->textures = NULL;
  sv_object->model = NULL;
  sv_object->image_pos[0] = 0;
  sv_object->image_pos[1] = 0;

  switch (type) {
  case SV_OBJECT_TYPE_INVALID:
    break;
  case SV_OBJECT_TYPE_SOLAR:
    sv_object->solar.radius[0] = 0.0;
    sv_object->solar.radius[1] = 0.0;
    sv_object->solar.radius[2] = 0.0;
    for (i = 0; i < 3; i++) {
      for (j = 0; j < 3; j++) {
        sv_object->solar.rotation[i][j] = 0.0;
      }
    }
    break;
  case SV_OBJECT_TYPE_STAR:
    sv_object->star.spectral = NULL;
    sv_object->star.color = 0xffffffff;
    break;
  default:
    assert(FALSE);
    return NULL;
  }

  return sv_object;
}

// destructor
void destroy_sv_object(SV_OBJECT* sv_object)
{
  int i;

  if (sv_object != NULL) {
    if (sv_object->id != NULL) {
      free(sv_object->id);
    }
    if (sv_object->name != NULL) {
      free(sv_object->name);
    }

    for (i = 0; i < sv_object->texture_count; i++) {
      destroy_sv_texture(sv_object->textures[i]);
    }
    free(sv_object->textures);

    if (sv_object->model != NULL) {
      destroy_sv_model(sv_object->model);
    }

    switch (sv_object->type) {
    case SV_OBJECT_TYPE_SOLAR:
      break;
    case SV_OBJECT_TYPE_STAR:
      if (sv_object->star.spectral != NULL) {
        free(sv_object->star.spectral);
      }
      break;
    case SV_OBJECT_TYPE_INVALID:
      assert(FALSE);
      break;
    }
    free(sv_object);
  }
}

// add new texture into sv_object
BOOL sv_object_add_texture(SV_OBJECT* sv_object, SV_TEXTURE* sv_texture)
{
  int new_size = sizeof(SV_TEXTURE **) * (sv_object->texture_count + 1);
  SV_TEXTURE** sv_textures_new
    = (SV_TEXTURE * *)realloc(sv_object->textures, new_size);
  if (sv_textures_new == NULL) {
    fprintf(stderr, "%s: insufficient memory\n", app_name);
    return FALSE;
  }
  sv_object->textures = sv_textures_new;
  sv_object->textures[sv_object->texture_count] = sv_texture;
  sv_object->texture_count++;

  return TRUE;
}

////////////////////////////////////////////////////////////////////////
// SV_TEXTURE
////////////////////////////////////////////////////////////////////////

// constructor
SV_TEXTURE* new_sv_texture()
{
  int i;

  SV_TEXTURE* sv_texture = (SV_TEXTURE *)malloc(sizeof(SV_TEXTURE));
  if (sv_texture == NULL) {
    fprintf(stderr, "%s: insufficient memory\n", app_name);
    return NULL;
  }

  sv_texture->component = SV_TEXTURE_COMPONENT_INVALID;
  sv_texture->src = NULL;
  sv_texture->type = SV_TEXTURE_TYPE_INVALID;
  sv_texture->title = NULL;
  sv_texture->region_string = NULL;
  for (i = 0; i < 4; i++) {
    sv_texture->region[i] = 0.0;
  }
  sv_texture->resolution = 1.0;
  sv_texture->ring_radius = -1.0;

  return sv_texture;
}

// destructor
void destroy_sv_texture(SV_TEXTURE* sv_texture)
{
  if (sv_texture != NULL) {
    if (sv_texture->src != NULL) {
      free(sv_texture->src);
    }
    if (sv_texture->title != NULL) {
      free(sv_texture->title);
    }
    if (sv_texture->region_string != NULL) {
      free(sv_texture->region_string);
    }
    free(sv_texture);
  }
}

////////////////////////////////////////////////////////////////////////
// SV_MODEL
////////////////////////////////////////////////////////////////////////

// constructor
SV_MODEL* new_sv_model()
{
  SV_MODEL* sv_model = (SV_MODEL *)malloc(sizeof(SV_MODEL));
  if (sv_model == NULL) {
    fprintf(stderr, "%s: insufficient memory\n", app_name);
    return NULL;
  }

  sv_model->src = NULL;
  sv_model->scale = 1.0;
  sv_model->color = 0xffffffff;

  return sv_model;
}

// destructor
void destroy_sv_model(SV_MODEL* sv_model)
{
  if (sv_model != NULL) {
    if (sv_model->src != NULL) {
      free(sv_model->src);
    }
    free(sv_model);
  }
}

////////////////////////////////////////////////////////////////////////
// SV_FRAME
////////////////////////////////////////////////////////////////////////

// constructor
SV_FRAME* new_sv_frame()
{
  SV_FRAME* sv_frame = (SV_FRAME *)malloc(sizeof(SV_FRAME));
  if (sv_frame == NULL) {
    fprintf(stderr, "%s: insufficient memory\n", app_name);
    return NULL;
  }

  if ((sv_frame->view = new_sv_view()) == NULL) {
    free(sv_frame);
    return NULL;
  }
  sv_frame->object_count = 0;
  sv_frame->objects = NULL;

  return sv_frame;
}

// destructor
void destroy_sv_frame(SV_FRAME* sv_frame)
{
  int i;

  if (sv_frame != NULL) {

    // view
    destroy_sv_view(sv_frame->view);
    sv_frame->view = NULL;

    // objects
    for (i = 0; i < sv_frame->object_count; i++) {
      destroy_sv_object(sv_frame->objects[i]);
    }
    if (sv_frame->objects != NULL) {
      free(sv_frame->objects);
      sv_frame->objects = NULL;
    }
    sv_frame->object_count = 0;

  }
}

// add new object into sv_frame
BOOL sv_frame_add_object(SV_FRAME* sv_frame, SV_OBJECT* sv_object)
{
  int new_size = sizeof(SV_OBJECT **) * (sv_frame->object_count + 1);
  SV_OBJECT** sv_objects_new
    = (SV_OBJECT * *)realloc(sv_frame->objects, new_size);
  if (sv_objects_new == NULL) {
    fprintf(stderr, "%s: insufficient memory\n", app_name);
    return FALSE;
  }
  sv_frame->objects = sv_objects_new;
  sv_frame->objects[sv_frame->object_count] = sv_object;
  sv_frame->object_count++;

  return TRUE;
}

////////////////////////////////////////////////////////////////////////
// SV_DOC
////////////////////////////////////////////////////////////////////////

// constructor
SV_DOC* new_sv_doc()
{
  SV_DOC* sv_doc = (SV_DOC *)malloc(sizeof(SV_DOC));
  if (sv_doc == NULL) {
    fprintf(stderr, "%s: insufficient memory\n", app_name);
    return NULL;
  }

  if ((sv_doc->head = new_sv_head()) == NULL) {
    free(sv_doc);
    return NULL;
  }
  sv_doc->frame_count = 0;
  sv_doc->frames = NULL;

  return sv_doc;
}

// destructor
void destroy_sv_doc(SV_DOC* sv_doc)
{
  int i;

  if (sv_doc != NULL) {

    // head
    destroy_sv_head(sv_doc->head);
    sv_doc->head = NULL;

    // frames
    for (i = 0; i < sv_doc->frame_count; i++) {
      destroy_sv_frame(sv_doc->frames[i]);
    }
    if (sv_doc->frames != NULL) {
      free(sv_doc->frames);
    }
    sv_doc->frame_count = 0;

    free(sv_doc);
  }
}

// add new frame into sv_doc
BOOL sv_doc_add_frame(SV_DOC* sv_doc, SV_FRAME* sv_frame)
{
  int new_size = sizeof(SV_FRAME **) * (sv_doc->frame_count + 1);
  SV_FRAME** sv_frames_new
    = (SV_FRAME * *)realloc(sv_doc->frames, new_size);
  if (sv_frames_new == NULL) {
    fprintf(stderr, "%s: insufficient memory\n", app_name);
    return FALSE;
  }
  sv_doc->frames = sv_frames_new;
  sv_doc->frames[sv_doc->frame_count] = sv_frame;
  sv_doc->frame_count++;

  return TRUE;
}
