/**
 * @file      sv_doc.h
 * @brief     svdoc structure
 * @date      2010-03-12
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

//
// sv_doc.c - 'sv' is solar view, space view or whatever
//
#ifndef _SV_DOC_H
#define _SV_DOC_H

#ifdef TRUE
#  undef TRUE
#endif
#ifdef FALSE
#  undef FALSE
#endif

typedef int BOOL;
#define FALSE 0
#define TRUE (!FALSE)

#define XML_ENCODING    "UTF-8"

//
// head
//

typedef struct tagSV_HEAD {
  char* title;
} SV_HEAD;

//
// view
//

typedef struct tagSV_VIEW {
  char* date;
  double location[3];
  double boresight[3];
  int nbounds;
  double* bounds;
  double fov;
  double pos_angle;
  double angle_res;
  char* shape;
  double center[2];
  int image_size[2];
  double sky_color[4];
} SV_VIEW;

//
// texture
//

typedef enum {
  SV_TEXTURE_TYPE_INVALID = 0,
  SV_TEXTURE_TYPE_FILE,
  SV_TEXTURE_TYPE_WMS,
  SV_TEXTURE_TYPE_UNKNOWN,
} SV_TEXTURE_TYPE;

typedef enum {
  SV_TEXTURE_COMPONENT_INVALID = 0,
  SV_TEXTURE_COMPONENT_BODY,
  SV_TEXTURE_COMPONENT_RING,
} SV_TEXTURE_COMPONENT;

typedef struct tagSV_TEXTURE {
  SV_TEXTURE_COMPONENT component;       // BODY or RING
  char* src;                  // filename or URI
  SV_TEXTURE_TYPE type;       // FILE or WMS
  char* title;                // WMS title
  char* region_string;        // region (longitude/latitude double pair)
  double region[4];           // region in double
  double resolution;          // region resolution [pixel/degree]
  double ring_radius;         // ring radius
} SV_TEXTURE;

//
// model
//

typedef struct tagSV_MODEL {
  char* src;        // filename
  double scale;
  unsigned long color;
  char* format;        // filename
} SV_MODEL;

//
// object
//

typedef enum {
  SV_OBJECT_TYPE_INVALID = 0,
  SV_OBJECT_TYPE_SOLAR,
  SV_OBJECT_TYPE_STAR,
} SV_OBJECT_TYPE;

typedef struct tagSV_OBJECT {
  // attributes (required)
  SV_OBJECT_TYPE type;
  char* id;
  // elements
  char* name;
  double position[3];
  double magnitude;
  double distance;
  int texture_count;
  SV_TEXTURE** textures;
  SV_MODEL* model;
  int image_pos[2];
  union {
    struct {
      double radius[3];
      double rotation[3][3];
    } solar;
    struct {
      char* spectral;
      unsigned long color;
    } star;
  };
} SV_OBJECT;

//
// frame
//

typedef struct tagSV_FRAME {
  SV_VIEW* view;
  int object_count;
  SV_OBJECT** objects;
} SV_FRAME;

//
// svdoc
//

typedef struct tagSV_DOC {
  SV_HEAD* head;
  int frame_count;
  SV_FRAME** frames;
} SV_DOC;

//
// sv_doc.c
//
extern SV_HEAD* new_sv_head();
extern void destroy_sv_head(SV_HEAD* sv_head);

extern SV_VIEW* new_sv_view();
extern void destroy_sv_view(SV_VIEW* sv_view);
extern BOOL sv_view_add_bound(SV_VIEW* self, const double* vec);
extern void sv_view_get_bound(SV_VIEW* self, int index, double* vec);

extern SV_OBJECT* new_sv_object(int type);
extern void destroy_sv_object(SV_OBJECT* sv_object);
extern BOOL sv_object_add_texture(SV_OBJECT* sv_object, SV_TEXTURE* sv_texture);

extern SV_TEXTURE* new_sv_texture();
extern void destroy_sv_texture(SV_TEXTURE* sv_texture);

extern SV_MODEL* new_sv_model();
extern void destroy_sv_model(SV_MODEL* sv_model);

extern SV_FRAME* new_sv_frame();
extern void destroy_sv_frame(SV_FRAME* sv_frame);
extern BOOL sv_frame_add_object(SV_FRAME* sv_frame, SV_OBJECT* sv_object);

extern SV_DOC* new_sv_doc();
extern void destroy_sv_doc(SV_DOC* sv_doc);
extern BOOL sv_doc_add_frame(SV_DOC* sv_doc, SV_FRAME* sv_frame);

//
// sv_read.c
//
extern BOOL read_sv_doc(char* uri, SV_DOC* sv_doc, BOOL show_error);

//
// sv_write.c
//
extern BOOL write_sv_doc(char* uri, SV_DOC* sv_doc);

#endif // _SV_DOC_H
