/**
 * @file      sv_read.c
 * @brief     read svdoc XML
 * @date      2010-03-12
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <assert.h>
#include <libxml/parser.h>
#include "sv_doc.h"
#include "flow_common.h"

////////////////////////////////////////////////////////////////////////
// common
////////////////////////////////////////////////////////////////////////

static void dump_node_name(xmlNodePtr cur)
{
  if (cur == NULL) {
    return;
  }
  fprintf(stderr, "** %s", cur->name);
  cur = cur->parent;
  while (cur->parent != NULL) {
    fprintf(stderr, " < %s", cur->name);
    if (xmlStrcmp(cur->name, (const xmlChar *)"object") == 0) {
      xmlChar* id = xmlGetProp(cur, (const xmlChar *)"id");
      if (id != NULL) {
        fprintf(stderr, "(%s)", id);
        xmlFree(id);
      }
    }
    cur = cur->parent;
  }
  fprintf(stderr, "\n");
}

static BOOL is_negligible_node(const xmlChar* name)
{
  return xmlStrcmp(name, (const xmlChar *)"text") == 0
         || xmlStrcmp(name, (const xmlChar *)"comment") == 0;
}

static xmlChar* get_node_string(xmlDocPtr doc, xmlNodePtr cur)
{
  xmlChar* p = NULL;
  xmlChar* value = xmlNodeListGetString(doc, cur->xmlChildrenNode, 1);
  p = xmlStrdup(value);
  xmlFree(value);
  return p;
}

static SV_OBJECT_TYPE get_node_object_type(xmlNodePtr cur)
{
  xmlChar* type_name;
  SV_OBJECT_TYPE type;

  // cur should be an "object" element
  assert(xmlStrcmp(cur->name, (const xmlChar *)"object") == 0);

  if ((type_name = xmlGetProp(cur, (const xmlChar *)"type")) == NULL) {
    fprintf(stderr, "%s: no type attribute in <%s>\n", app_name, cur->name);
    return SV_OBJECT_TYPE_INVALID;
  }
  type = SV_OBJECT_TYPE_INVALID;
  if (xmlStrcasecmp(type_name, (const xmlChar *)"solar") == 0) {
    type = SV_OBJECT_TYPE_SOLAR;
  }
  else if (xmlStrcasecmp(type_name, (const xmlChar *)"star") == 0) {
    type = SV_OBJECT_TYPE_STAR;
  }
  xmlFree(type_name);
  if (type == SV_OBJECT_TYPE_INVALID) {
    fprintf(stderr, "%s: invalid object type: %s\n", app_name, type_name);
  }
  return type;
}

static int get_node_integer(xmlDocPtr doc, xmlNodePtr cur)
{
  int i;
  xmlChar* value = xmlNodeListGetString(doc, cur->xmlChildrenNode, 1);
  i = atoi((char *)value);
  xmlFree(value);
  return i;
}

static double get_node_double(xmlDocPtr doc, xmlNodePtr cur)
{
  double f;
  xmlChar* value = xmlNodeListGetString(doc, cur->xmlChildrenNode, 1);
  f = atof((char *)value);
  xmlFree(value);
  return f;
}

static BOOL get_node_xyz(double xyz[3], xmlDocPtr doc, xmlNodePtr cur)
{
  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
    if (xmlStrcmp(cur->name, (const xmlChar *)"x") == 0) {
      xyz[0] = get_node_double(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"y") == 0) {
      xyz[1] = get_node_double(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"z") == 0) {
      xyz[2] = get_node_double(doc, cur);
    }
    else if (!is_negligible_node(cur->name)) {
      fprintf(stderr, "%s: invalid node <%s> found\n", app_name, cur->name);
      dump_node_name(cur);
      return FALSE;
    }
    cur = cur->next;
  }
  return TRUE;
}

static BOOL get_node_radec(double radec[2], xmlDocPtr doc, xmlNodePtr cur)
{
  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
    if (xmlStrcmp(cur->name, (const xmlChar *)"ra") == 0) {
      radec[0] = get_node_double(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"dec") == 0) {
      radec[1] = get_node_double(doc, cur);
    }
    else if (!is_negligible_node(cur->name)) {
      fprintf(stderr, "%s: invalid node <%s> found\n", app_name, cur->name);
      dump_node_name(cur);
      return FALSE;
    }
    cur = cur->next;
  }
  return TRUE;
}

static BOOL get_node_image_size(int size[2], xmlDocPtr doc, xmlNodePtr cur)
{
  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
    if (xmlStrcmp(cur->name, (const xmlChar *)"width") == 0) {
      size[0] = get_node_integer(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"height") == 0) {
      size[1] = get_node_integer(doc, cur);
    }
    else if (!is_negligible_node(cur->name)) {
      fprintf(stderr, "%s: invalid node <%s> found\n", app_name, cur->name);
      dump_node_name(cur);
      return FALSE;
    }
    cur = cur->next;
  }
  return TRUE;
}

static BOOL get_node_sky_color(double color[4], xmlDocPtr doc, xmlNodePtr cur)
{
  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
    if (xmlStrcmp(cur->name, (const xmlChar *)"red") == 0) {
      color[0] = get_node_double(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"green") == 0) {
      color[1] = get_node_double(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"blue") == 0) {
      color[2] = get_node_double(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"alpha") == 0) {
      color[3] = get_node_double(doc, cur);
    }
    else if (!is_negligible_node(cur->name)) {
      fprintf(stderr, "%s: invalid node <%s> found\n", app_name, cur->name);
      dump_node_name(cur);
      return FALSE;
    }
    cur = cur->next;
  }
  return TRUE;
}

static BOOL get_node_image_pos(int image_pos[3], xmlDocPtr doc, xmlNodePtr cur)
{
  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
    if (xmlStrcmp(cur->name, (const xmlChar *)"x") == 0) {
      image_pos[0] = get_node_integer(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"y") == 0) {
      image_pos[1] = get_node_integer(doc, cur);
    }
    else if (!is_negligible_node(cur->name)) {
      fprintf(stderr, "%s: invalid node <%s> found\n", app_name, cur->name);
      dump_node_name(cur);
      return FALSE;
    }
    cur = cur->next;
  }
  return TRUE;
}

static BOOL get_node_matrix(double mtx[3][3], xmlDocPtr doc, xmlNodePtr cur)
{
  int i, n;
  char* p;

  cur = cur->xmlChildrenNode;
  i = 0;
  while (cur != NULL) {
    if (xmlStrcmp(cur->name, (const xmlChar *)"A3") == 0) {
      if (i >= 3) {
        fprintf(stderr, "%s: too many <A3> node in <%s> found\n", app_name, cur->name);
        dump_node_name(cur);
        return FALSE;
      }
      p = (char *)get_node_string(doc, cur);
      n = 0;
      if (p != NULL) {
        n = sscanf(p, "%lf,%lf,%lf", &mtx[i][0], &mtx[i][1], &mtx[i][2]);
        free(p);
      }
      if (p == NULL || n != 3) {
        fprintf(stderr, "%s: invalid <A3> format\n", app_name);
        dump_node_name(cur);
        return FALSE;
      }
      i++;
    }
    else if (!is_negligible_node(cur->name)) {
      fprintf(stderr, "%s: invalid node <%s> found\n", app_name, cur->name);
      dump_node_name(cur);
      return FALSE;
    }
    cur = cur->next;
  }
  return TRUE;
}

////////////////////////////////////////////////////////////////////////
// SV_HEAD
////////////////////////////////////////////////////////////////////////
static BOOL read_sv_head(SV_HEAD* sv_head, xmlDocPtr doc, xmlNodePtr cur)
{
  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
    if (xmlStrcmp(cur->name, (const xmlChar *)"title") == 0) {
      sv_head->title = (char *)get_node_string(doc, cur);
    }
    else if (!is_negligible_node(cur->name)) {
      fprintf(stderr, "%s: invalid node <%s> found\n", app_name, cur->name);
      dump_node_name(cur);
      return FALSE;
    }
    cur = cur->next;
  }
  return TRUE;
}

////////////////////////////////////////////////////////////////////////
// SV_VIEW
////////////////////////////////////////////////////////////////////////
static BOOL read_sv_view(SV_VIEW* sv_view, xmlDocPtr doc, xmlNodePtr cur)
{
  xmlNodePtr parent;
  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
    if (xmlStrcmp(cur->name, (const xmlChar *)"date") == 0) {
      sv_view->date = (char *)get_node_string(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"location") == 0) {
      if (!get_node_xyz(sv_view->location, doc, cur)) {
        return FALSE;
      }
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"boresight") == 0) {
      if (!get_node_xyz(sv_view->boresight, doc, cur)) {
        return FALSE;
      }
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"fov") == 0) {
      sv_view->fov = get_node_double(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"pos_angle") == 0) {
      sv_view->pos_angle = get_node_double(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"angle_res") == 0) {
      sv_view->angle_res = get_node_double(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"shape") == 0) {
      sv_view->shape = (char *)get_node_string(doc, cur);
    }

    else if (xmlStrcmp(cur->name, (const xmlChar *)"bounds") == 0) {
      // <bounds>
      parent = cur;
      cur = cur->xmlChildrenNode;
      // <point>
      do {
        double bound[3];
        if (xmlStrcmp(cur->name, (const xmlChar *)"point") == 0) {
          get_node_xyz(bound, doc, cur);
          if (!sv_view_add_bound(sv_view, bound)) {
            return FALSE;
          }
        }
        else if (!is_negligible_node(cur->name)) {
          fprintf(stderr, "%s: invalid node <%s> found\n", app_name, cur->name);
          dump_node_name(cur);
          return FALSE;
        }
      } while ((NULL != (cur = cur->next)));
      cur = parent;
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"center") == 0) {
      if (!get_node_radec(sv_view->center, doc, cur)) {
        return FALSE;
      }
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"image_size") == 0) {
      if (!get_node_image_size(sv_view->image_size, doc, cur)) {
        return FALSE;
      }
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"sky_color") == 0) {
      if (!get_node_sky_color(sv_view->sky_color, doc, cur)) {
        return FALSE;
      }
    }
    else if (!is_negligible_node(cur->name)) {
      fprintf(stderr, "%s: invalid node <%s> found\n", app_name, cur->name);
      dump_node_name(cur);
      return FALSE;
    }
    cur = cur->next;
  }
  return TRUE;
}

////////////////////////////////////////////////////////////////////////
// SV_TEXTURE
////////////////////////////////////////////////////////////////////////
static BOOL read_sv_texture(SV_TEXTURE* sv_texture, xmlDocPtr doc, xmlNodePtr cur)
{
  xmlChar* str;

  // component attribute
  if ((str = xmlGetProp(cur, (const xmlChar *)"component")) == NULL) {
    fprintf(stderr, "%s: no `component' attribute in <%s>\n", app_name, cur->name);
    return FALSE;
  }
  if (xmlStrcasecmp(str, (const xmlChar *) "body") == 0) {
    sv_texture->component = SV_TEXTURE_COMPONENT_BODY;
  } else if (xmlStrcasecmp(str, (const xmlChar *) "ring") == 0) {
    sv_texture->component = SV_TEXTURE_COMPONENT_RING;
  } else {
    fprintf(stderr, "%s: invalid component `%s' in <%s>\n",
            app_name, str, cur->name);
    free(str);
    return FALSE;
  }
  free(str);

  // get elements
  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
    if (xmlStrcmp(cur->name, (const xmlChar *)"src") == 0) {
      // type
      if ((str = xmlGetProp(cur, (const xmlChar *)"type")) == NULL) {
        fprintf(stderr, "%s: no `type' in <%s>\n", app_name, cur->name);
        return FALSE;
      }
      if (xmlStrcasecmp(str, (const xmlChar *) "file") == 0) {
        sv_texture->type = SV_TEXTURE_TYPE_FILE;
      } else if (xmlStrcasecmp(str, (const xmlChar *) "wms") == 0) {
        sv_texture->type = SV_TEXTURE_TYPE_WMS;
      } else {
        fprintf(stderr, "%s: invalid type `%s' in <%s>\n",
                app_name, str, cur->name);
        free(str);
        return FALSE;
      }
      free(str);
      // title
      sv_texture->title = NULL;
      if ((str = xmlGetProp(cur, (const xmlChar *)"title")) != NULL) {
        sv_texture->title = (char *)str;
      }
      // src
      sv_texture->src = (char *)get_node_string(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"region") == 0) {
      double d[4];
      int i;
      sv_texture->region_string = NULL;
      for (i = 0; i < 4; i++) {
        sv_texture->region[i] = 0.0;
      }
      if ((str = get_node_string(doc, cur)) != NULL) {
        if (sscanf((char *)str, "%lf,%lf,%lf,%lf", &d[0], &d[1], &d[2], &d[3]) == 4) {
          for (i = 0; i < 4; i++) {
            sv_texture->region[i] = d[i];
          }
        }
        sv_texture->region_string = (char *)str;
      }
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"ring_radius") == 0) {
      sv_texture->ring_radius = get_node_double(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"resolution") == 0) {
      sv_texture->resolution = get_node_double(doc, cur);
    }
    else if (!is_negligible_node(cur->name)) {
      fprintf(stderr, "%s: invalid node <%s> found\n", app_name, cur->name);
      dump_node_name(cur);
      return FALSE;
    }
    cur = cur->next;
  }
  return TRUE;
}

////////////////////////////////////////////////////////////////////////
// SV_MODEL
////////////////////////////////////////////////////////////////////////
static BOOL read_sv_model(SV_MODEL* sv_model, xmlDocPtr doc, xmlNodePtr cur)
{
  xmlChar* str;
  memset(sv_model, 0, sizeof(SV_MODEL));
  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
    if (xmlStrcmp(cur->name, (const xmlChar *)"src") == 0) {
      // type
      if ((str = xmlGetProp(cur, (const xmlChar *)"type")) == NULL) {
        fprintf(stderr, "%s: no `type' in <%s>\n", app_name, cur->name);
        return FALSE;
      }
      if (xmlStrcasecmp(str, (const xmlChar *) "file") != 0) {
        fprintf(stderr, "%s: invalid type `%s' in <%s>\n",
                app_name, str, cur->name);
        free(str);
        return FALSE;
      }
      free(str);
      // src
      sv_model->src = (char *)get_node_string(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"scale") == 0) {
      sv_model->scale = get_node_double(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"color") == 0) {
      int r,g,b;
      char* rgb = (char *)get_node_string(doc, cur);
      if (sscanf(rgb, "%d,%d,%d", &r, &g, &b) == 3) {
        sv_model->color = (r << 16) | (g << 8) | b;
      }
      free(rgb);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"format") == 0) {
      sv_model->format = (char *)get_node_string(doc, cur);
    }
    else if (!is_negligible_node(cur->name)) {
      fprintf(stderr, "%s: invalid node <%s> found\n", app_name, cur->name);
      dump_node_name(cur);
      return FALSE;
    }
    cur = cur->next;
  }
  return TRUE;
}

////////////////////////////////////////////////////////////////////////
// SV_OBJECT
////////////////////////////////////////////////////////////////////////
static BOOL read_sv_object(SV_OBJECT* sv_object, xmlDocPtr doc, xmlNodePtr cur)
{
  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
    // common
    if (xmlStrcmp(cur->name, (const xmlChar *)"name") == 0) {
      sv_object->name = (char *)get_node_string(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"position") == 0) {
      if (!get_node_xyz(sv_object->position, doc, cur)) {
        return FALSE;
      }
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"magnitude") == 0) {
      sv_object->magnitude = get_node_double(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"texture") == 0) {
      SV_TEXTURE* sv_texture = new_sv_texture();
      if (!read_sv_texture(sv_texture, doc, cur) ||
          !sv_object_add_texture(sv_object, sv_texture)) {
        destroy_sv_texture(sv_texture);
        return FALSE;
      }
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"model") == 0) {
      if (sv_object->model != NULL) {
        fprintf(stderr, "%s: duplicate model node found\n", app_name);
        return FALSE;
      }
      SV_MODEL* sv_model = new_sv_model();
      if (read_sv_model(sv_model, doc, cur)) {
        sv_object->model = sv_model;
      } else {
        destroy_sv_model(sv_model);
        return FALSE;
      }
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"distance") == 0) {
      sv_object->distance = get_node_double(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"image_pos") == 0) {
      get_node_image_pos(sv_object->image_pos, doc, cur);
    }
    // solar
    else if (xmlStrcmp(cur->name, (const xmlChar *)"radius") == 0) {
      if (!get_node_xyz(sv_object->solar.radius, doc, cur)) {
        return FALSE;
      }
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"rotation") == 0) {
      if (!get_node_matrix(sv_object->solar.rotation, doc, cur)) {
        return FALSE;
      }
    }
    // star
    else if (xmlStrcmp(cur->name, (const xmlChar *)"spectral") == 0) {
      sv_object->star.spectral = (char *)get_node_string(doc, cur);
    }
    else if (xmlStrcmp(cur->name, (const xmlChar *)"color") == 0) {
      int r,g,b;
      char* rgb = (char *)get_node_string(doc, cur);
      if (sscanf(rgb, "%d,%d,%d", &r, &g, &b) == 3) {
        sv_object->star.color = (r << 16) | (g << 8) | b;
      }
      free(rgb);
    }
    // ignore
    else if (!is_negligible_node(cur->name)) {
      fprintf(stderr, "%s: invalid node <%s> found\n", app_name, cur->name);
      dump_node_name(cur);
      return FALSE;
    }
    cur = cur->next;
  }
  return TRUE;
}

////////////////////////////////////////////////////////////////////////
// SV_FRAME
////////////////////////////////////////////////////////////////////////
BOOL read_sv_frame(SV_FRAME* sv_frame, xmlDocPtr doc, xmlNodePtr cur)
{
  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
    if ( xmlStrcmp(cur->name, (const xmlChar *)"view") == 0) {
      if (!read_sv_view(sv_frame->view, doc, cur)) {
        break;
      }
    }
    else if ( xmlStrcmp(cur->name, (const xmlChar *)"object") == 0) {
      // attributes
      SV_OBJECT* sv_object;
      SV_OBJECT_TYPE type;
      xmlChar* id;
      if ((type = get_node_object_type(cur)) == SV_OBJECT_TYPE_INVALID) {
        return FALSE;
      }
      if ((id = xmlGetProp(cur, (const xmlChar *)"id")) == NULL) {
        fprintf(stderr, "%s: no id in <%s>\n", app_name, cur->name);
        return FALSE;
      }
      sv_object = new_sv_object(type);
      sv_object->id = (char *)id;
      // elements
      if (!read_sv_object(sv_object, doc, cur)) {
        break;
      }
      if (!sv_frame_add_object(sv_frame, sv_object)) {
        break;
      }
    }
    else if (!is_negligible_node(cur->name)) {
      fprintf(stderr, "%s: invalid node <%s> found\n", app_name, cur->name);
      dump_node_name(cur);
      return FALSE;
    }

    cur = cur->next;
  }
  return TRUE;
}

////////////////////////////////////////////////////////////////////////
// suppress xml error message
////////////////////////////////////////////////////////////////////////
void XMLErrorFunction(void* ctx, const char* msg, ...)
{
  // do nothing

  //va_list arg_ptr;
  //va_start(arg_ptr, msg);
  //vfprintf(stderr, msg, arg_ptr);
  //va_end(arg_ptr);
}

////////////////////////////////////////////////////////////////////////
// SV_DOC (public function)
////////////////////////////////////////////////////////////////////////
BOOL read_sv_doc(char* uri, SV_DOC* sv_doc, BOOL show_error)
{
  xmlDocPtr doc;
  xmlNodePtr cur;

  // sv_doc should be allocated
  assert(sv_doc != NULL);
  assert(sv_doc->head != NULL);

  if (!show_error) {
    xmlSetGenericErrorFunc( NULL, XMLErrorFunction );
  }

  if ((doc = xmlParseFile(uri)) == NULL) {
    fprintf(stderr, "%s: invalid or empty xml document.\n", app_name);
    return FALSE;
  }

  if ((cur = xmlDocGetRootElement(doc)) == NULL) {
    fprintf(stderr,"%s: xml document %s is empty\n", app_name, uri);
    xmlFreeDoc(doc);
    return FALSE;
  }

  if (xmlStrcmp(cur->name, (const xmlChar *) "svdoc")) {
    fprintf(stderr,"%s: %s: wrong xml document, root node is not <svdoc>\n",
            app_name, uri);
    xmlFreeDoc(doc);
    return FALSE;
  }

  //
  // walk through document
  //
  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
    if ( xmlStrcmp(cur->name, (const xmlChar *)"head") == 0) {
      if (!read_sv_head(sv_doc->head, doc, cur)) {
        break;
      }
    }
    else if ( xmlStrcmp(cur->name, (const xmlChar *)"frame") == 0) {
      // attributes
      SV_FRAME* sv_frame;
      sv_frame = new_sv_frame();
      // elements
      if (!read_sv_frame(sv_frame, doc, cur)) {
        break;
      }
      if (!sv_doc_add_frame(sv_doc, sv_frame)) {
        break;
      }
    }
    else if (!is_negligible_node(cur->name)) {
      fprintf(stderr, "%s: invalid node <%s> found\n", app_name, cur->name);
      dump_node_name(cur);
      return FALSE;
    }

    cur = cur->next;
  }

  xmlFreeDoc(doc);

  if (cur != NULL) {
    return FALSE;
  }

  return TRUE;
}
