/**
 * @file      sv_write.c
 * @brief     write svdoc XML
 * @date      2010-03-12
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <math.h>
#include <assert.h>
#include <libxml/xmlwriter.h>
#include "sv_doc.h"
#include "flow_common.h"

////////////////////////////////////////////////////////////////////////
// common
////////////////////////////////////////////////////////////////////////
static void write_element_xyz(xmlTextWriterPtr writer, const xmlChar* element, double xyz[3], int nFrac)
{
  xmlTextWriterStartElement(writer, element);
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"x", "%.*f", nFrac, xyz[0]);
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"y", "%.*f", nFrac, xyz[1]);
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"z", "%.*f", nFrac, xyz[2]);
  xmlTextWriterEndElement(writer);
}

static void write_element_radec(xmlTextWriterPtr writer, const xmlChar* element, double radec[2], int nFrac)
{
  xmlTextWriterStartElement(writer, element);
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"ra",  "%.*f", nFrac, radec[0]);
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"dec", "%.*f", nFrac, radec[1]);
  xmlTextWriterEndElement(writer);
}

static void write_element_image_pos(xmlTextWriterPtr writer, const xmlChar* element, int image_pos[2])
{
  xmlTextWriterStartElement(writer, element);
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"x", "%d", image_pos[0]);
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"y", "%d", image_pos[1]);
  xmlTextWriterEndElement(writer);
}

static void write_element_matrix(xmlTextWriterPtr writer, const xmlChar* element, double mtx[3][3])
{
  int i;

  xmlTextWriterStartElement(writer, element);
  for (i = 0; i < 3; i++) {
    xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"A3", "%.10f, %.10f, %.10f",
                                    mtx[i][0], mtx[i][1], mtx[i][2]);
  }
  xmlTextWriterEndElement(writer);
}

static BOOL write_element_object_type(xmlTextWriterPtr writer, SV_OBJECT_TYPE type)
{
  char* type_name;
  switch (type) {
  case SV_OBJECT_TYPE_SOLAR:
    type_name = "solar";
    break;
  case SV_OBJECT_TYPE_STAR:
    type_name = "star";
    break;
  default:
    fprintf(stderr, "%s: invalid object type: %d\n", app_name, type);
    return FALSE;
  }
  xmlTextWriterWriteAttribute(writer, (const xmlChar *)"type", (const xmlChar *)type_name);
  return TRUE;
}

static BOOL write_element_texture_type(xmlTextWriterPtr writer, SV_TEXTURE_TYPE type)
{
  char* type_name;
  switch (type) {
  case SV_TEXTURE_TYPE_FILE:
    type_name = "file";
    break;
  case SV_TEXTURE_TYPE_WMS:
    type_name = "wms";
    break;
  default:
    fprintf(stderr, "%s: invalid texture type: %d\n", app_name, type);
    return FALSE;
  }
  xmlTextWriterWriteAttribute(writer, (const xmlChar *)"type", (const xmlChar *)type_name);
  return TRUE;
}

static BOOL write_element_texture_component(xmlTextWriterPtr writer,
                                            SV_TEXTURE_COMPONENT component)
{
  char* component_name;
  switch (component) {
  case SV_TEXTURE_COMPONENT_BODY:
    component_name = "body";
    break;
  case SV_TEXTURE_COMPONENT_RING:
    component_name = "ring";
    break;
  default:
    fprintf(stderr, "%s: invalid texture component: %d\n", app_name, component);
    return FALSE;
  }
  xmlTextWriterWriteAttribute(writer, (const xmlChar *)"component", (const xmlChar *)component_name);
  return TRUE;
}

////////////////////////////////////////////////////////////////////////
// SV_HEAD
////////////////////////////////////////////////////////////////////////
static BOOL write_sv_head(xmlTextWriterPtr writer, SV_HEAD* head)
{
  // <head>
  xmlTextWriterStartElement(writer, (const xmlChar *)"head");
  if (head->title != NULL) {
    xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"title", "%s", head->title);
  }

  // </head>
  xmlTextWriterEndElement(writer);

  return TRUE;
}

////////////////////////////////////////////////////////////////////////
// SV_VIEW
////////////////////////////////////////////////////////////////////////
static BOOL write_sv_view(xmlTextWriterPtr writer, SV_VIEW* view)
{
  int i;
  // <view>
  xmlTextWriterStartElement(writer, (const xmlChar *)"view");

  if (view->date != NULL) {
    xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"date", "%s", view->date);
  }
  write_element_xyz(writer, (const xmlChar *)"location", view->location, 6);
  write_element_xyz(writer, (const xmlChar *)"boresight", view->boresight, 6);
  // <bounds>
  if (view->nbounds) {
    xmlTextWriterStartElement(writer, (const xmlChar *)"bounds");
    for (i = 0; i < view->nbounds; i++) {
      write_element_xyz(writer, (const xmlChar *)"point", &view->bounds[i*3], 6);
    }
    xmlTextWriterEndElement(writer);
  }
  // </bounds>
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"fov", "%.6f", view->fov);
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"pos_angle", "%.6f", view->pos_angle);
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"angle_res", "%.6e", view->angle_res);
  if (view->shape != NULL) {
    xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"shape", "%s", view->shape);
  }
  write_element_radec(writer, (const xmlChar *)"center", view->center, 6);

  // <image_size>
  xmlTextWriterStartElement(writer, (const xmlChar *)"image_size");
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"width",  "%d", view->image_size[0]);
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"height", "%d", view->image_size[1]);
  xmlTextWriterEndElement(writer);

  // <sky_color>
  xmlTextWriterStartElement(writer, (const xmlChar *)"sky_color");
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"red",  "%f", view->sky_color[0]);
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"green", "%f", view->sky_color[1]);
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"blue", "%f", view->sky_color[2]);
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"alpha", "%f", view->sky_color[3]);
  xmlTextWriterEndElement(writer);

  // </view>
  xmlTextWriterEndElement(writer);

  return TRUE;
}

////////////////////////////////////////////////////////////////////////
// SV_TEXTURE
////////////////////////////////////////////////////////////////////////
static BOOL write_sv_texture(xmlTextWriterPtr writer, SV_TEXTURE* texture)
{
  // <texture>
  xmlTextWriterStartElement(writer, (const xmlChar *)"texture");

  if (texture->component != SV_TEXTURE_COMPONENT_INVALID) {
    write_element_texture_component(writer, texture->component);
  }

  // <texture> -> <src>
  xmlTextWriterStartElement(writer, (const xmlChar *)"src");

  if (texture->type != SV_TEXTURE_TYPE_INVALID) {
    write_element_texture_type(writer, texture->type);
  }
  if (texture->title != NULL) {
    xmlTextWriterWriteAttribute(writer, (const xmlChar *)"title",
                                (const xmlChar *)texture->title);
  }
  xmlTextWriterWriteString(writer, (const xmlChar *)texture->src);

  // <texture> -> </src>
  xmlTextWriterEndElement(writer);

  // <region>
  if (texture->region_string != NULL) {
    xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"region",
                                    "%s", (const xmlChar *)texture->region_string);
  }

  // <resolution> -- slave of <title>
  if (texture->title != NULL) {
    xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"resolution",
                                    "%g", texture->resolution);
  }

  // <ring_radius>
  if (texture->ring_radius > 0.0) {
    xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"ring_radius",
                                    "%g", texture->ring_radius);
  }

  // </texture>
  xmlTextWriterEndElement(writer);

  return TRUE;
}

////////////////////////////////////////////////////////////////////////
// SV_MODEL
////////////////////////////////////////////////////////////////////////
static BOOL write_sv_model(xmlTextWriterPtr writer, SV_MODEL* model)
{
  if (model == NULL) {
    return TRUE;
  }

  // <model>
  xmlTextWriterStartElement(writer, (const xmlChar *)"model");

  // src
  xmlTextWriterStartElement(writer, (const xmlChar *)"src");
  xmlTextWriterWriteAttribute(writer, (const xmlChar *)"type", (const xmlChar *)"file");
  xmlTextWriterWriteString(writer, (const xmlChar *)model->src);
  xmlTextWriterEndElement(writer);

  // scale
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"scale",
                                  "%g", model->scale);

  // color
  if (model->color != 0xffffffff) {
    char rgb[32];
    sprintf(rgb, "%d,%d,%d",
            (int)((model->color & 0x00ff0000) >> 16),
            (int)((model->color & 0x0000ff00) >>  8),
            (int)((model->color & 0x000000ff)));
    xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"color", "%s", rgb);
  }

  // format
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"format",
                                  "%s", model->format);

  // </model>
  xmlTextWriterEndElement(writer);

  return TRUE;
}

////////////////////////////////////////////////////////////////////////
// SV_OBJECT
////////////////////////////////////////////////////////////////////////
static BOOL write_sv_object(xmlTextWriterPtr writer, SV_OBJECT* object)
{
  int i;

  xmlTextWriterStartElement(writer, (const xmlChar *)"object");
  // attributes
  if (!write_element_object_type(writer, object->type)) {
    return FALSE;
  }
  xmlTextWriterWriteAttribute(writer, (const xmlChar *)"id", (const xmlChar *)object->id);
  // elements
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"name", "%s", object->name);
  write_element_xyz(writer, (const xmlChar *)"position", object->position, 6);
  xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"magnitude", "%.2f", object->magnitude);
  if (object->distance > 0.0) {
    // 9.460730e+12 km = 1 l.y.
    xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"distance",
                                    (object->distance > 9.460730e+12 ? "%.6e" : "%.6f"),
                                    object->distance);
  }
  for (i = 0; i < object->texture_count; i++) {
    write_sv_texture(writer, object->textures[i]);
  }
  write_sv_model(writer, object->model);
  write_element_image_pos(writer, (const xmlChar *)"image_pos", object->image_pos);

  switch (object->type) {
  case SV_OBJECT_TYPE_SOLAR:
    write_element_xyz(writer, (const xmlChar *)"radius", object->solar.radius, 4);
    write_element_matrix(writer, (const xmlChar *)"rotation", object->solar.rotation);
    break;
  case SV_OBJECT_TYPE_STAR:
    if (object->star.spectral != NULL) {
      xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"spectral", "%s", object->star.spectral);
    }
    if (object->star.color != 0xffffffff) {
      char rgb[32];
      sprintf(rgb, "%d,%d,%d",
              (int)((object->star.color & 0x00ff0000) >> 16),
              (int)((object->star.color & 0x0000ff00) >>  8),
              (int)((object->star.color & 0x000000ff)));
      xmlTextWriterWriteFormatElement(writer, (const xmlChar *)"color", "%s", rgb);
    }
    break;
  default:
    assert(FALSE);
    break;
  }
  xmlTextWriterEndElement(writer);

  return TRUE;
}

////////////////////////////////////////////////////////////////////////
// SV_FRAME
////////////////////////////////////////////////////////////////////////
static BOOL write_sv_frame(xmlTextWriterPtr writer, SV_FRAME* sv_frame)
{
  int i;

  // <frame>
  xmlTextWriterStartElement(writer, (const xmlChar *)"frame");

  // <view>
  if (!write_sv_view(writer, sv_frame->view)) {
    return FALSE;
  }

  // <object>
  for (i = 0; i < sv_frame->object_count; i++) {
    if (!write_sv_object(writer, sv_frame->objects[i])) {
      return FALSE;
    }
  }

  // </frame>
  xmlTextWriterEndElement(writer);

  return TRUE;
}

////////////////////////////////////////////////////////////////////////
// SV_DOC
////////////////////////////////////////////////////////////////////////
static BOOL do_write_sv_doc(xmlTextWriterPtr writer, SV_DOC* sv_doc)
{
  int i;

  xmlTextWriterSetIndent(writer, 1);
  xmlTextWriterSetIndentString(writer, (const xmlChar *)"  ");

  xmlTextWriterStartDocument(writer, NULL, XML_ENCODING, NULL);

  // <svdoc>
  xmlTextWriterStartElement(writer, (const xmlChar *)"svdoc");

  // <head>
  if (!write_sv_head(writer, sv_doc->head)) {
    return FALSE;
  }

  // <frame>
  for (i = 0; i < sv_doc->frame_count; i++) {
    if (!write_sv_frame(writer, sv_doc->frames[i])) {
      return FALSE;
    }
  }

  // </svdoc>
  xmlTextWriterEndElement(writer);

  xmlTextWriterEndDocument(writer);

  return TRUE;
}

//
// public entry
//
BOOL write_sv_doc(char* uri, SV_DOC* sv_doc)
{
  BOOL ret;
  xmlTextWriterPtr writer;

  // sv_doc should be allocated
  assert(sv_doc != NULL);
  assert(sv_doc->head != NULL);

  if ((writer = xmlNewTextWriterFilename(uri, 0)) == NULL) {
    fprintf(stderr,"%s: error creating the xml writer\n", app_name);
    return FALSE;
  }

  ret = do_write_sv_doc(writer, sv_doc);

  xmlFreeTextWriter(writer);

  return ret;
}
