/**
 * @file      tge.h
 * @brief     tiny 3D Graphics Engine
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2010-2011 Japan Aerospace Exploration Agency
 *
 */

#ifndef _TGE_H
#define _TGE_H

#include <stdio.h>
#include <stdint.h>
#include "canvas.h"
#include "triangle.h"

enum TGEenum {
  TGE_NO_ERROR           = 0,
  TGE_INVALID_ENUM,
  TGE_INVALID_VALUE,
  TGE_INVALID_OPERATION,
  TGE_STACK_OVERFLOW,
  TGE_STACK_UNDERFLOW,
  TGE_OUT_OF_MEMORY,

  // data types
  TGE_BYTE               = 0x100,
  TGE_UNSIGNED_BYTE,
  TGE_SHORT,
  TGE_UNSIGNED_SHORT,
  TGE_INT,
  TGE_UNSIGNED_INT,
  TGE_FLOAT,
  TGE_DOUBLE,

  // tge_enable, tge_disable
  TGE_ALPHA_TEST         = 0x110,
  TGE_AUTO_NORMAL,
  TGE_CULL_FACE,
  TGE_DEPTH_TEST,
  TGE_LIGHT0,
  TGE_TEXTURE_2D,
  TGE_BLEND,
  TGE_POLYGON_OFFSET_FILL,
  TGE_POLYGON_OFFSET_LINE,
  TGE_POLYGON_OFFSET_POINT,
  TGE_RESCALE_NORMAL,

  // tgeLightfv
  TGE_LIGHT1             = 0x130,
  TGE_POSITION,

  // tge_set_cull_face, tge_cull_face
  TGE_FRONT              = 0x140,
  TGE_BACK,
  TGE_FRONT_AND_BACK,

  // tgeEnableClientState
  TGE_COLOR_ARRAY        = 0x150,
  TGE_NORMAL_ARRAY,
  TGE_TEXTURE_COORD_ARRAY,
  TGE_VERTEX_ARRAY,

  // tgeDrawArrays
  TGE_POINTS             = 0x160,
  TGE_LINE_STRIP,
  TGE_LINE_LOOP,
  TGE_LINES,
  TGE_TRIANGLE_STRIP,
  TGE_TRIANGLE_FAN,
  TGE_TRIANGLES,

  // matrix mode
  TGE_MODELVIEW          = 0x170,
  TGE_PROJECTION,

  // blend function
  TGE_ZERO               = 0x180,
  TGE_ONE,
  TGE_SRC_COLOR,
  TGE_ONE_MINUS_SRC_COLOR,
  TGE_DST_COLOR,
  TGE_ONE_MINUS_DST_COLOR,
  TGE_SRC_ALPHA,
  TGE_ONE_MINUS_SRC_ALPHA,
  TGE_DST_ALPHA,
  TGE_ONE_MINUS_DST_ALPHA,

  // get
  TGE_COLOR_ARRAY_SIZE   = 0x200,
  TGE_COLOR_ARRAY_STRIDE,
  TGE_COLOR_ARRAY_TYPE,
  TGE_CULL_FACE_MODE,
  TGE_CURRENT_COLOR,
  TGE_DEPTH_FUNC,
  TGE_MATRIX_MODE,
  TGE_MAX_LIGHTS,
  TGE_MAX_PROJECTION_STACK_DEPTH,
  TGE_NORMAL_ARRAY_STRIDE,
  TGE_NORMAL_ARRAY_TYPE,
  TGE_POLYGON_OFFSET_FACTOR,
  TGE_POLYGON_OFFSET_UNITS,
  TGE_PROJECTION_MATRIX,
  TGE_PROJECTION_STACK_DEPTH,
  TGE_TEXTURE_COORD_ARRAY_SIZE,
  TGE_TEXTURE_COORD_ARRAY_STRIDE,
  TGE_TEXTURE_COORD_ARRAY_TYPE,
  TGE_VERTEX_ARRAY_SIZE,
  TGE_VERTEX_ARRAY_STRIDE,
  TGE_VERTEX_ARRAY_TYPE,
  TGE_VIEWPORT,

};

typedef uint32_t TGEenum;
typedef unsigned char TGEboolean;
typedef uint32_t TGEbitfield;
typedef char TGEbyte;
typedef unsigned char TGEubyte;
typedef short TGEshort;
typedef unsigned short TGEushort;
typedef int32_t TGEint;
typedef uint32_t TGEsizei;
typedef uint32_t TGEuint;
typedef float TGEfloat;
typedef float TGEclampf;
typedef double TGEdouble;
typedef double TGEclampd;
typedef void TGEvoid;
typedef intptr_t TGEContext;

// Boolean
#define TGE_FALSE              0
#define TGE_TRUE               1

// tgeClear
#define TGE_COLOR_BUFFER_BIT   1
#define TGE_DEPTH_BUFFER_BIT   (1<<1)

typedef struct _TGEItessellator {
  TGEenum type;
  TGEsizei size;
  TGEsizei stride;
  const void* data;
  TGEint* index;
  TGEsizei nindex;
} TGEItessellator;

__BEGIN_DECLS

//////////////////////////////////////////////////////////////////////
// コンテキスト作成・開放
//////////////////////////////////////////////////////////////////////
/**
 * TGE コンテキストを作成します。
 * この関数は OpenGL に合わせてありますが、最内部ではグローバル変
 * 数を使っているため、コンテキストを複数作成することはできません。
 */
TGEContext* tgeCreateContext(void);

/**
 * TGE コンテキストを開放します。
 */
void tgeDestroyContext(TGEContext* self);

/**
 * TGE コンテキストを描画対象にします。
 * @param screen  描画対象 canvas。
 * @param ctx     TGE コンテキスト。
 */
TGEboolean tgeMakeCurrent(canvas* screen, TGEContext* ctx);


//////////////////////////////////////////////////////////////////////
// レンダリングオプション
//////////////////////////////////////////////////////////////////////

/**
 * レンダリング設定を有効にします。
 */
void tgeEnable(TGEenum type);

/**
 * レンダリング設定を無効にします。
 */
void tgeDisable(TGEenum type);

/**
 * 光源の設定を行います。
 * 現在は light=TGE_LIGHT0, pname=TGE_POSION 固定です。
 * params には光源ベクトルを設定します。
 */
void tgeLightfv(TGEenum light, TGEenum pname, const TGEfloat* params);

/**
 * カリング面を設定します。
 * @param face  カリング面。TGE_BACK, TGE_FRONT, TGE_FROMT_AND_BACK が指定できます。
 */
void tgeCullFace(TGEenum face);

/**
 * ブレンド条件を設定します。
 * @param sfactor  描画色のブレンド条件。
 * @param dfactor  既存色のブレンド条件。
 */
void tgeBlendFunc(TGEenum sfactor, TGEenum dfactor);

/**
 * clear color を設定します。
 */
void tgeClearColor(TGEclampf red, TGEclampf green, TGEclampf blue, TGEclampf alpha);

/**
 * 頂点カラーを設定します。
 */
void tgeColor4f(TGEfloat red, TGEfloat green, TGEfloat blue, TGEfloat alpha);
void tgeColor4i(TGEint red, TGEint green, TGEint blue, TGEint alpha);
#define tgeColor3f(r,g,b)  tgeColor4f((r),(g),(b),1.0)
#define tgeColor3i(r,g,b)  tgeColor4i((r),(g),(b),255)
#define tgeColor4d         tgeColor4f
#define tgeColor3d         tgeColor3f
#define tgeColor4s         tgeColor4i
#define tgeColor3s         tgeColor3i
void tgeColor4dv(const TGEdouble* v);
void tgeColor4fv(const TGEfloat* v);
void tgeColor4iv(const TGEint* v);

//////////////////////////////////////////////////////////////////////
// Projection matrix
//////////////////////////////////////////////////////////////////////

/**
 * アクティブにする matrix を指定します。
 * @param mode  TGE_PROJECTION, TGE_MODELVIEW のいずれか。
 */
void tgeMatrixMode(TGEenum mode);

/**
 * 投影方法を設定します。
 */
void tgeFrustum(TGEdouble left, TGEdouble right, TGEdouble bottom, TGEdouble top, TGEdouble near, TGEdouble far);

/**
 * 投影マトリクスに正投影マトリクスを乗算します。
 */
void tgeOrtho(TGEdouble left, TGEdouble right, TGEdouble bottom, TGEdouble top, TGEdouble near, TGEdouble far);

/**
 * スクリーンマトリクスに unit matrix を設定します。
 */
void tgeLoadIdentity(void);

/**
 * スクリーンマトリクスを直接設定します。double 版。
 * @param m  4x4 マトリクス。
 */
void tgeLoadMatrixd(const TGEdouble* m);

/**
 * スクリーンマトリクスを直接設定します。float 版。
 * @param m  4x4 マトリクス。
 */
void tgeLoadMatrixf(const TGEfloat* m);

/**
 * スクリーンマトリクスを乗算します。(test)
 */
void tgeMultMatrixd(const TGEdouble* m);

/**
 * 回転マトリクスをスクリーンマトリクスに乗算します。
 */
void tgeRotated(TGEdouble anTGEe, TGEdouble x, TGEdouble y, TGEdouble z);
#define tgeRotatef tgeRotated

/**
 * 移動マトリクスをスクリーンマトリクスに乗算します。
 */
void tgeTranslated(TGEdouble x, TGEdouble y, TGEdouble z);
#define tgeTranslatef tgeTranslated

void tgeScaled(TGEdouble x, TGEdouble y, TGEdouble z);
#define tgeScalef tgeScaled



//////////////////////////////////////////////////////////////////////
// utility
//////////////////////////////////////////////////////////////////////
void tgeClear(TGEbitfield mask);
TGEenum tgeGetError(void);
void tgePolygonOffset(TGEfloat factor, TGEfloat units);
void tgeDepthMask(TGEboolean flag);
void tgeGetBooleanv(TGEenum pname, TGEboolean* params);
void tgeGetDoublev(TGEenum pname, TGEdouble* params);
void tgeGetFloatv(TGEenum pname, TGEfloat* params);
void tgeGetIntegerv(TGEenum pname, TGEint* params);

//////////////////////////////////////////////////////////////////////
// texture
//////////////////////////////////////////////////////////////////////

/**
 * テクスチャをセットします。
 */
void tgeTexImage2D(canvas* data);

//////////////////////////////////////////////////////////////////////
// drawarray
//////////////////////////////////////////////////////////////////////

/**
 * client capability を有効にします。
 * @param cap  capability. TGE_COLOR_ARRAY, TGE_NORMAL_ARRAY,
 *             TGE_TEXTURE_COORD_ARRAY, TGE_VERTEX_ARRAY を指定できます。
 */
void tgeEnableClientState(TGEenum cap);

/**
 * client capability を無効にします。
 * @param cap  capability. TGE_COLOR_ARRAY, TGE_NORMAL_ARRAY,
 *             TGE_TEXTURE_COORD_ARRAY, TGE_VERTEX_ARRAY を指定できます。
 */
void tgeDisableClientState(TGEenum cap);

void tgeVertexPointer(TGEint size, TGEenum type, TGEsizei stride, const TGEvoid* pointer);
void tgeColorPointer(TGEint size, TGEenum type, TGEsizei stride, const TGEvoid* pointer);
void tgeNormalPointer(TGEenum type, TGEsizei stride, const TGEvoid* pointer );
void tgeTexCoordPointer(TGEint size, TGEenum type, TGEsizei stride, const TGEvoid* pointer);

/**
 * array data をレンダリングします。
 * @param mode   基本形状。TGE_TRIANGLE_STRIP, TGE_TRIANGLE_FAN, TGE_TRIANGLES
 *               を指定できます。
 * @param first  描画開始オフセット。
 * @param count  描画に使用する頂点数。
 */
void tgeDrawArrays(TGEenum mode, TGEint first, TGEsizei count);

/**
 * array data をレンダリングします。
 * @param mode    基本形状。TGE_TRIANGLE_STRIP, TGE_TRIANGLE_FAN, TGE_TRIANGLES
 *                を指定できます。
 * @param count   頂点数。
 * @param type    indices のデータ型。TGE_UNSIGNED_INT が最も高速に動作します。
 * @param indices 頂点インデックスの配列。
 */
void tgeDrawElements(TGEenum mode, TGEsizei count, TGEenum type, const TGEvoid* indices);

void tgePushMatrix(void);
void tgePopMatrix(void);

void tgeInvertMatrixd(const TGEdouble* m, TGEdouble* mout);

/* **********************************************************************
* ベクトル演算
* **********************************************************************/
void tgeVectorAddd(TGEsizei size, const TGEdouble* va, const TGEdouble* vb, TGEdouble* vc);
void tgeVectorSubd(TGEsizei size, const TGEdouble* va, const TGEdouble* vb, TGEdouble* vc);
void tgeVectorMuld(TGEsizei size, const TGEdouble* va, TGEdouble f, TGEdouble* vb);
void tgeVectorDivd(TGEsizei size, const TGEdouble* va, TGEdouble f, TGEdouble* vb);
TGEdouble tgeVectorCrossd(const TGEdouble* va, const TGEdouble* vb);
TGEdouble tgeVectorDotd(TGEsizei size, const TGEdouble* va, const TGEdouble* vb);
TGEdouble tgeVectorNormd(TGEsizei size, const TGEdouble* va, TGEdouble* vb);
TGEdouble tgeVectorDistd(TGEsizei size, const TGEdouble* va);

/* **********************************************************************
* TGEI - TGE internal function (tgei*).
* This functions are not defined in OpenGL.
* **********************************************************************/

/**
 * 頂点座標を現在の設定に従って Viewport にプロットしたときの変換を行い
 * ます。
 * @param size  vertex の要素数。2 or 3.
 * @param type  vertex の型。TGE_INT, TGE_UNSIGNED_INT, TGE_FLOAT, TGE_DOUBLE
 *              が利用できます。
 * @param iv    入力 vertex.
 * @param ov    変換後の vertex. iv と同じアドレスを指定できます。
 */
void tgeiVertexToViewport(TGEsizei size, TGEenum type, const TGEvoid* iv, TGEvoid* ov);

/**
 * tessellator を作成します。
 * @result  tge tessellator。
 */
TGEItessellator* tgeiNewTess(void);

/**
 * tessellator にポリゴンを設定し、分割を行います。
 * @param self    tge tessellator.
 * @param type    vertex data type.
 * @param stride  byte offset of vertex.
 * @param vertex  vertex data (XY dimension only).
 */
void tgeiTessVertexPointer(TGEItessellator* self, TGEenum type, TGEint stride, TGEint size, const TGEvoid* vertex);

/**
 * tessellator から分割後の頂点インデックスのポインタを取得します。
 * @param self  tge tesselator.
 * @param size  インデックスのサイズ。
 * @result インデックスポインタ。
 */
const TGEint* tgeiTessIndex(TGEItessellator* self, TGEsizei* size);

/**
 * tessellator を開放します。
 * @param self  tge tessellator.
 */
void tgeiDeleteTess(TGEItessellator* self);

__END_DECLS

#endif /* _TGE_H */
