/**
 * @file      timepair.c
 * @brief     time pair utility
 * @date      2015-05-14
 *
 * @copyright
 * Copyright 2010-2015 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "timepair.h"

int comp(const void* c1, const void* c2) {
  double diff;
  time_pair* t1 = (time_pair *)c1;
  time_pair* t2 = (time_pair *)c2;
  diff = t1->start - t2->start;
  if ( diff > 0 ) return 1;
  if ( diff < 0 ) return -1;
  return 0;
}

time_pair_list* create_time_pair_list(void) {
  time_pair_list* list = (time_pair_list *)malloc(sizeof(time_pair_list));
  if (list != NULL) {
    list->pairs = (time_pair *)malloc(sizeof(time_pair)*DEFAULT_BLOCK);
    if (list->pairs == NULL) {
      free(list);
      list = NULL;
    } else {
      list->max = DEFAULT_BLOCK;
      list->used = 0;
    }
  }
  return list;
}

void* destroy_time_pair_list(time_pair_list* list) {
  if (list != NULL) {
    if (list->pairs != NULL) {
      free(list->pairs);
    }
    free(list);
  }
  return list;
}

int get_time_pair_used(time_pair_list* list) {
  return list->used;
}

int get_time_pair_max(time_pair_list* list) {
  return list->max;
}

int add_time_pair_with_comment(time_pair_list* list, double start, double end, const char* comment) {
  time_pair p;
  time_pair* tmp;
  p.start = start;
  p.end = end;
  if (list->max == list->used) {
    list->max += DEFAULT_BLOCK;
    tmp = (time_pair *)realloc(list->pairs, sizeof(time_pair)*list->max);
    if (tmp == NULL) {
      return -1;
    }
    list->pairs = tmp;
  }
  list->pairs[list->used] = p;
  if (strlen(comment) > 0) {
    strncpy(list->pairs[list->used].comment, comment, SIZE_COMMENT-1);
    list->pairs[list->used].comment[SIZE_COMMENT-1] = '\0';
  } else {
    list->pairs[list->used].comment[0] = '\0';
  }
  list->used++;
  return list->used - 1;
}

int add_time_pair(time_pair_list* list, double start, double end) {
  return add_time_pair_with_comment(list, start, end, "");
}

time_pair get_time_pair(time_pair_list* list, int index) {
  return list->pairs[index];
}

void sort_time_pair(time_pair_list* list) {
  qsort(list->pairs, list->used, sizeof(time_pair), comp);
}

int remove_time_pair(time_pair_list* list, int index) {
  int i;
  if (index >= list->used) {
    return 0;
  }
  if (list->used == 0) {
    return 0;
  }
  for(i=index; i<list->used-1; ++i) {
    list->pairs[i] = list->pairs[i+1];
  }
  list->used--;
  return list->used;
}

int reduce_time_pair_list(time_pair_list* list) {
  int i, ret, flag = 0;
  if (list->used <= 1) {
    return list->used;
  }
  sort_time_pair(list);
  for(i=0; i<list->used-1; ++i) {
    if (list->pairs[i].end >= list->pairs[i+1].start) {
      if (list->pairs[i].end < list->pairs[i+1].end) {
        list->pairs[i].end = list->pairs[i+1].end;
        list->pairs[i].comment[0] = '\0';
      }
      remove_time_pair(list, i+1);
      flag = 1;
      break;
    }
  }
  if (flag) {
    ret = reduce_time_pair_list(list);
  }
  return ret;
}

time_pair_list* merge_time_pair_list(time_pair_list* list1, time_pair_list* list2) {
  int i;
  time_pair_list* list = create_time_pair_list();
  for(i=0; i<list1->used; ++i) {
    add_time_pair(list, list1->pairs[i].start, list1->pairs[i].end);
  }
  for(i=0; i<list2->used; ++i) {
    add_time_pair(list, list2->pairs[i].start, list2->pairs[i].end);
  }
  return list;
}

time_pair_list* logical_and_time_pair_list(time_pair_list* list1, time_pair_list* list2) {
  int i, j;
  time_pair_list* list = create_time_pair_list();
  for(i=0; i<list1->used; ++i) {
    for(j=0; j<list2->used; ++j) {
      time_pair p1 = list1->pairs[i];
      time_pair p2 = list2->pairs[j];
      if (p1.start <= p2.start) {
        if (p2.start <= p1.end) {
          if (p1.end <= p2.end) {
            // case 1: p1.start <= p2.start <= p1.end <= p2.end
            add_time_pair(list, p2.start, p1.end);
          } else {
            // case 2: p1.start <= p2.start <= p2.end <= p1.end
            add_time_pair(list, p2.start, p2.end);
          }
        }
      } else {
        if (p1.start <= p2.end) {
          if (p2.end <= p1.end) {
            // case 3: p2.start < p1.start <= p2.end <= p1.end
            add_time_pair(list, p1.start, p2.end);
          } else {
            // case 4: p2.start < p1.start <= p1.end <= p2.end
            add_time_pair(list, p1.start, p1.end);
          }
        }
      }
    }
  }
  return list;
}

void dump_time_pair_list(time_pair_list* list) {
  int i;
  for(i=0; i<list->used; ++i) {
    printf("%d: %f %f %s\n", i,
           list->pairs[i].start,
           list->pairs[i].end,
           list->pairs[i].comment);
  }
}
