/**
 * @file      timepair.h
 * @brief     time pair utility
 * @date      2015-05-14
 *
 * @copyright
 * Copyright 2010-2015 Japan Aerospace Exploration Agency
 *
 */

#ifndef __TIMEPAIR_H__
#define __TIMEPAIR_H__

#define DEFAULT_BLOCK 100
#define SIZE_COMMENT 256

typedef struct tag_time_pair {
  double start;
  double end;
  char comment[SIZE_COMMENT];
} time_pair;

typedef struct tag_time_pair_list {
  int max;
  int used;
  time_pair* pairs;
} time_pair_list;

__BEGIN_DECLS

/**
 * @brief Create a new time-pair list
 * @return Upon successful comment, create_time_pair_list() returns a pointer of time-pair list.
 *         Otherwise, NULL is returned.
 */
time_pair_list* create_time_pair_list(void);

/**
 * @brief Destroy the time-pair list
 * @param[in]   list  time-pair list
 * @return destroy_time_pair_list() returns a pointer of input time-pair list.
 */
void* destroy_time_pair_list(time_pair_list* list);

/**
 * @brief Get the used size in time-pair list
 * @param[in]   list  time-pair list
 * @return get_time_pair_used() returns the used size of time-pair list.
 */
int get_time_pair_used(time_pair_list* list);

/**
 * @brief Get the maximum size in time-pair list
 * @param[in]   list  time-pair list
 * @return get_time_pair_max() returns the maximum size of time-pair list.
 */
int get_time_pair_max(time_pair_list* list);

/**
 * @brief Add time-pair into the specified time-pair list
 * @param[in]   list  time-pair list
 * @param[in]   start start time
 * @param[in]   end   end time
 * @return add_time_pair() added time pair into the time-pair list.
 *         If block size is too small to store the pair, time-pair list is automatically
 *         expanded, then the time pair is added.
 */
int add_time_pair(time_pair_list* list, double start, double end);

/**
 * @brief Add time-pair into the specified time-pair list
 * @param[in]   list    time-pair list
 * @param[in]   start   start time
 * @param[in]   end     end time
 * @param[in]   comment comment
 * @return add_time_pair_with_comment() added time pair as well as add_time_pair().
 *         In addition, the comment can be added.
 */
int add_time_pair_with_comment(time_pair_list* list, double start, double end, const char* comment);

/**
 * @brief Get time pair from the specifid time-pair list using index.
 * @param[in]   list  time-pair list
 * @param[in]   index index
 * @return get_time_pair() returns the time-pair structure.
 */
time_pair get_time_pair(time_pair_list* list, int index);

/**
 * @brief Reduce the duplicated coverage in the time-pair list
 * @param[in]   list  time-pair list
 * @return reduce_time_pair_list() returns the number of the time-pair list which is reduced.
 */
int reduce_time_pair_list(time_pair_list* list);

/**
 * @brief Merge two time-pair lists into one time-pair list
 * @param[in]   list1  time-pair list1
 * @param[in]   list2  time-pair list2
 * @return merge_time_pair_list() returns merged time-pair list.
 */
time_pair_list* merge_time_pair_list(time_pair_list* list1, time_pair_list* list2);

/**
 * @brief Dump the contents of the time-pair list
 * @param[in]   list  time-pair list
 */
void dump_time_pair_list(time_pair_list* list);

/**
 * @brief Merge two time-pair lists using logical AND operater
 * @param[in]   list1  time-pair list1
 * @param[in]   list2  time-pair list2
 * @return logical_and_time_pair_list() returns merged time-pair list, whcih method is
 *         logical AND operator.
 */
time_pair_list* logical_and_time_pair_list(time_pair_list* list1, time_pair_list* list2);

__END_DECLS

#endif
