/**
 * @file      triangle.c
 * @brief     drawing triangle polygon
 * @date      2012-08-23
 *
 * @copyright
 * Copyright 2010-2012 AstroArts Inc.
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include <stdint.h>

#include "canvas.h"
#include "triangle.h"
#include "zbuf.h"
#include "float_util.h"

#define TOFX(x)      ((x)<<16)
#define TOFL(x)      ((x)>>16)

#define RGB2BGR(x)   (x)
#define min(a, b)    (((a) < (b)) ? (a) : (b))
#define max(a, b)    (((a) > (b)) ? (a) : (b))
#define GetAValue(x) ((unsigned char)(((x)&0xFF000000)>>24))
#define GetRValue(x) ((unsigned char)(((x)&0x00FF0000)>>16))
#define GetGValue(x) ((unsigned char)(((x)&0x0000FF00)>>8))
#define GetBValue(x) ((unsigned char)((x) &0x000000FF))
#define MakeARGB(a,r,g,b) (((DWORD)(a)<<24)|((DWORD)(r)<<16)|((DWORD)(g)<<8)|(DWORD)((b)))

typedef uint32_t DWORD;
typedef int BOOL;
typedef unsigned char BYTE;

// parameters for DrawTriangle
typedef struct tagDRAWTRI {
  double fX[3];
  double fY[3];
  double fZ[3];

  COLORREF clrVtx[3];
  COLORREF clrSky[3];
  BYTE nAlpha[3];

  int nUU[3];                           // 16.16 fixed dicimal
  int nVV[3];                           // 16.16 fixed dicimal
  double fUU[3];
  double fVV[3];

  BOOL bBiLinear;
  int nStep;
} DRAWTRI, * PDRAWTRI;

// DrawTriangle Drawing mode
typedef enum {
  BL_NORMAL,                    // normal
  BL_ADDSKY,                    // add skycolor
  BL_BRIGHTERSKY,               // skycolor and brighter
  BL_ADD,                       // Add original value of pixel (Enable alpha)
  BL_ALPHA,                     // alpha blending
  BL_ALPHA_ADD,                 // Add original value of pixel (Disable alpha)
  BL_BRIGHTER,                  // brighter
  BL_MULTIPLICATION,            // multiplication

  BL_PRNSUB,                    // subtract for printer (Disable alpha)
  BL_ALPHA_PRNSUB,              // subtract for printer (Enable alpha)
} BLEND_METHOD;


// DrawTriangle internal use
typedef struct tagDRAWTRI_SETUP {
  int nX1, nX2;

  int nRR1, nRR2;
  int nGG1, nGG2;
  int nBB1, nBB2;
  int nAA1, nAA2;


  int nFxAlpha1, nFxAlpha2;

  int nUU1, nUU2;
  int nVV1, nVV2;
  float fZZ1, fZZ2;
} DRAWTRI_SETUP, * PDRAWTRI_SETUP;

typedef struct tagRECT {
  int top;
  int left;
  int bottom;
  int right;
} RECT, * PRECT;


// Texture-enabled triangle drawing
static PDRAWTRI_SETUP m_pTriSetupBuffer = NULL;
static int m_nTriSY;
static int m_nTriEY;
static PDRAWTRI m_pDrawTri;
#if 0
static int m_nByteCount = 1;
#endif
static RECT m_rectScreen;
static canvas* tex_cv = NULL;
static canvas* screen_cv = NULL;
static zbuf* gZBuffer = NULL;
static int gEnableZBuffer;
static int gEnableBlend;
static int gBlendSrcFactor;
static int gBlendDstFactor;
static int gDepthMask;

void tri_SetPixelGray8(int nX, int nY, BYTE clr);
void tri_SetPixelARGB8(int nX, int nY, DWORD clr);
BYTE tri_GetTexelFxpGray8(int nUU, int nVV);
DWORD tri_GetTexelFxpARGB8(int nUU, int nVV);
BYTE tri_GetTexelGray8(int nU, int nV);
DWORD tri_GetTexelARGB8(int nU, int nV);
BYTE tri_ColorMulGray8(BYTE clr, BYTE nMulC1);
DWORD tri_ColorMulARGB8(DWORD clr, DWORD mclr);


//////////////////////////////////////////////////////////////////////
// Start/Stop
//////////////////////////////////////////////////////////////////////

/**
 * Constructor
 * Start drawing
 */
void tri_begin(canvas* screen, zbuf* zbuffer, canvas* texture)
{
  int nDIBHeight;
  int nHeight;

  assert(screen_cv == NULL);
  assert(tex_cv == NULL);
  assert(m_pTriSetupBuffer == NULL);

  // screen rect
  m_rectScreen.top = 0;
  m_rectScreen.left = 0;
  m_rectScreen.right = screen->width;
  m_rectScreen.bottom = screen->height;

  nHeight = screen->height;
  nDIBHeight = (nHeight + 1) & ~1;
  m_pTriSetupBuffer = malloc(sizeof(DRAWTRI_SETUP) * nDIBHeight);

  tri_set_blend(0);
  tri_set_blend_func(CVBlendOne, CVBlendZero);

  tex_cv = texture;
  screen_cv = screen;
  gZBuffer = zbuffer;
  gDepthMask = 1;
}

/**
 * Destructor
 */
void tri_end()
{
  if (m_pTriSetupBuffer != NULL) free(m_pTriSetupBuffer);
  m_pTriSetupBuffer = NULL;
  tex_cv = NULL;
  screen_cv = NULL;
  gZBuffer = NULL;
}

BYTE tri_ColorMulGray8(BYTE clr, BYTE nMulC1)
{
  BYTE nC1 = ((((DWORD)clr & 0x000000ff) * (DWORD)nMulC1) >> 8);
  return nC1;
}

DWORD tri_ColorMulARGB8(DWORD clr, DWORD mclr)
{
  DWORD a = (GetAValue(clr) * GetAValue(mclr)) >> 8;
  DWORD r = (GetRValue(clr) * GetRValue(mclr)) >> 8;
  DWORD g = (GetGValue(clr) * GetGValue(mclr)) >> 8;
  DWORD b = (GetBValue(clr) * GetBValue(mclr)) >> 8;
  return MakeARGB(a,r,g,b);
}

//////////////////////////////////////////////////////////////////////////////
// Width of texture
//////////////////////////////////////////////////////////////////////////////
static int GetTexWidth()
{
  return (NULL != tex_cv) ? tex_cv->width : 0;
}
//////////////////////////////////////////////////////////////////////////////
// Height of texture
//////////////////////////////////////////////////////////////////////////////
static int GetTexHeight()
{
  return (NULL != tex_cv) ? tex_cv->height : 0;
}

//////////////////////////////////////////////////////////////////////////////
// color of texture nU is pixel coordinate.
//////////////////////////////////////////////////////////////////////////////
BYTE    tri_GetTexelGray8(int nU, int nV)
{
  BYTE nRet;

  if (nU < 0) nU = 0;
  if (nV < 0) nV = 0;
  if (nU >= GetTexWidth()) nU = GetTexWidth() -1;
  if (nV >= GetTexHeight()) nV = GetTexHeight() -1;

  if (nU < 0 || nU >= GetTexWidth() || nV < 0 || nV >= GetTexHeight()) {
    nRet = (BYTE)0;
  } else {
    nRet = cv_get_pixel_gray8(tex_cv, nU, nV);
  }
  return nRet;
}

DWORD   tri_GetTexelARGB8(int nU, int nV)
{
  DWORD color;

  if (nU < 0) nU = 0;
  if (nV < 0) nV = 0;
  if (nU >= GetTexWidth()) nU = GetTexWidth() -1;
  if (nV >= GetTexHeight()) nV = GetTexHeight() -1;

  if (nU < 0 || nU >= GetTexWidth() || nV < 0 || nV >= GetTexHeight()) {
    color = 0x00FFFFFF;
  } else {
    color = cv_get_pixel_argb8(tex_cv, nU, nV);
  }
  return color;
}

//////////////////////////////////////////////////////////////////////////////
// texture color nUU is 16.16 fixed decimal
//////////////////////////////////////////////////////////////////////////////
BYTE tri_GetTexelFxpGray8(int nUU, int nVV)
{
  nUU -= 1 << 15;
  nVV -= 1 << 15;
  nUU = nUU >> 16;
  nVV = nVV >> 16;
  return tri_GetTexelGray8(nUU, nVV);
}
DWORD tri_GetTexelFxpARGB8(int nUU, int nVV)
{
  nUU -= 1 << 15;
  nVV -= 1 << 15;
  nUU = nUU >> 16;
  nVV = nVV >> 16;
  return tri_GetTexelARGB8(nUU, nVV);

}

//////////////////////////////////////////////////////////////////////////////
// set pixel
//////////////////////////////////////////////////////////////////////////////
void tri_SetPixelGray8(int nX, int nY, BYTE clr)
{
  cv_set_pixel_gray8(screen_cv, nX, nY, clr);
}

void tri_SetPixelARGB8(int nX, int nY, DWORD clr)
{
  if (gEnableBlend) {
    BYTE c[4];
    c[0] = GetAValue(clr);
    c[1] = GetRValue(clr);
    c[2] = GetGValue(clr);
    c[3] = GetBValue(clr);
    cv_blend_pixel2_ex(screen_cv, nX, nY, CVDepthARGB8, c,
                       gBlendSrcFactor, gBlendDstFactor);
  }
  else {
    cv_set_pixel_argb8(screen_cv, nX, nY, clr);
  }
}



#if 0
//////////////////////////////////////////////////////////////////////////////
// DrawTriangle_ALPHA8_Normal
//   PF_ALPHA8, enable color gouraud, disable alpha gouraud
//   normal
static void DrawTriangle_Normal()
{
  int nStep;
  int nAdrStep;
  int nX, nY;
  DWORD clrVtx;

  clrVtx = RGB2BGR(m_pDrawTri->clrVtx[0]);

  nStep    = m_pDrawTri->nStep;
  nAdrStep = nStep * m_nByteCount;

  for(nY = m_nTriSY; nY < m_nTriEY; nY += nStep) {

    int nX1;
    int nX2;
    int nDX;
    int nUU1, nUU2, nVV1, nVV2, nAUU, nAVV, nUU, nVV;

    nX1 = m_pTriSetupBuffer[nY].nX1;
    nX2 = m_pTriSetupBuffer[nY].nX2;
    nDX = nX2 - nX1;
    if(nX1 >= m_rectScreen.right || nX2 < 0 || nDX == 0) {
      continue;
    }
    nUU1 = m_pTriSetupBuffer[nY].nUU1;
    nUU2 = m_pTriSetupBuffer[nY].nUU2;
    nVV1 = m_pTriSetupBuffer[nY].nVV1;
    nVV2 = m_pTriSetupBuffer[nY].nVV2;
    nAUU = (nUU2 - nUU1) / nDX;
    nAVV = (nVV2 - nVV1) / nDX;
    nUU  = nUU1;
    nVV  = nVV1;


    for(nX = nX1; nX < nX2; nX++) {
      DWORD clrVtx;
      clrVtx = tri_GetTexelFxpARGB8(nUU, nVV);
      tri_SetPixelARGB8(nX, nY, clrVtx);

      nUU += nAUU;
      nVV += nAVV;
    }
  }
}
#endif

//////////////////////////////////////////////////////////////////////////////
// DrawTriangle_BGRA8888_ClrGouraud_Normal
//   PF_BGRA8888, enable color gouraud, disable alpha gouraud
//   normal
//////////////////////////////////////////////////////////////////////////////
static void DrawTriangle_ClrGouraud_Normal()
{
  int nX, nY;
  DWORD clr;

  for(nY = m_nTriSY; nY < m_nTriEY; nY++) {

    int nX1 = m_pTriSetupBuffer[nY].nX1;
    int nX2 = m_pTriSetupBuffer[nY].nX2;
    int nDX = nX2 - nX1;

    int nUU1, nUU2, nVV1, nVV2, nRR1, nGG1, nBB1, nAA1;
    int nRR2, nGG2, nBB2, nAA2, nARR, nAGG, nABB, nAAA;
    int nRR, nGG, nBB, nAA;
    int nAUU, nAVV, nUU, nVV;
    float fZZ1, fZZ2, fAZZ, fZZ;

    if(nX1 >= m_rectScreen.right || nX2 < 0 || nDX == 0) {
      continue;
    }

    nRR1 = m_pTriSetupBuffer[nY].nRR1;
    nGG1 = m_pTriSetupBuffer[nY].nGG1;
    nBB1 = m_pTriSetupBuffer[nY].nBB1;
    nAA1 = m_pTriSetupBuffer[nY].nAA1;
    nRR2 = m_pTriSetupBuffer[nY].nRR2;
    nGG2 = m_pTriSetupBuffer[nY].nGG2;
    nBB2 = m_pTriSetupBuffer[nY].nBB2;
    nAA2 = m_pTriSetupBuffer[nY].nAA2;
    nARR = (nRR2 - nRR1) / nDX;
    nAGG = (nGG2 - nGG1) / nDX;
    nABB = (nBB2 - nBB1) / nDX;
    nAAA = (nAA2 - nAA1) / nDX;
    nRR  = nRR1;
    nGG  = nGG1;
    nBB  = nBB1;
    nAA  = nAA1;
    if (tex_cv) {
      nUU1 = m_pTriSetupBuffer[nY].nUU1 + (1 << 15);
      nUU2 = m_pTriSetupBuffer[nY].nUU2 + (1 << 15);
      nVV1 = m_pTriSetupBuffer[nY].nVV1 + (1 << 15);
      nVV2 = m_pTriSetupBuffer[nY].nVV2 + (1 << 15);
      nAUU = (nUU2 - nUU1) / nDX;
      nAVV = (nVV2 - nVV1) / nDX;
      nUU  = nUU1;
      nVV  = nVV1;
    }
    if (gEnableZBuffer) {
      fZZ1 = m_pTriSetupBuffer[nY].fZZ1;
      fZZ2 = m_pTriSetupBuffer[nY].fZZ2;
      fAZZ = (fZZ2 - fZZ1) / nDX;
      fZZ  = fZZ1;
    }

    for(nX = nX1; nX < nX2; nX++) {
      if (!gEnableZBuffer ||
          ((gDepthMask) ? zbuf_set(gZBuffer, nX, nY, fZZ) :
           zbuf_test(gZBuffer, nX, nY, fZZ))) {
        if (tex_cv) {
          clr = tri_GetTexelFxpARGB8(nUU, nVV);
          clr = tri_ColorMulARGB8(clr, MakeARGB(nAA>>16, nRR>>16, nGG>>16, nBB>>16));
        }
        else {
          clr = MakeARGB(nAA>>16, nRR>>16, nGG>>16, nBB>>16);
        }
        tri_SetPixelARGB8(nX, nY, clr);
      }
      nRR += nARR;
      nGG += nAGG;
      nBB += nABB;
      nAA += nAAA;
      if (tex_cv) {
        nUU += nAUU;
        nVV += nAVV;
      }
      if (gEnableZBuffer) {
        fZZ += fAZZ;
      }
    }
  }
}


//////////////////////////////////////////////////////////////////////////////
// TriangleSetup_Tex_ClrGouraud
//   Enable texture, enable color gouraud, disable alpha gouraud
static void TriangleSetup_Tex_ClrGouraud()
{
  double fX[3], fY[3];
  double fNX[3], fNY[3];
  int nNX[3], nNY[3];
  double fVX[2], fVY[2];
  double fD;
  double fU_[3], fV_[3];
  double fVU[2], fVV[2];
  double fK[3][2];
  double fU0, fV0, fAUX, fAVX, fAUY, fAVY;
  int nUU0, nVV0, nAUUX, nAVVX, nAUUY, nAVVY, nX0, nY0;
  int y;
  int i, i1, i2;
  int j;
  int nSignX, nAX, nARR, nAGG, nABB, nAAA;

  fX[0] = m_pDrawTri->fX[0];
  fY[0] = m_pDrawTri->fY[0];
  fX[1] = m_pDrawTri->fX[1];
  fY[1] = m_pDrawTri->fY[1];
  fX[2] = m_pDrawTri->fX[2];
  fY[2] = m_pDrawTri->fY[2];

  fNX[0] = floor(fX[0]);
  fNY[0] = floor(fY[0]);
  fNX[1] = floor(fX[1]);
  fNY[1] = floor(fY[1]);
  fNX[2] = floor(fX[2]);
  fNY[2] = floor(fY[2]);

  nNX[0] = (int)fNX[0];
  nNY[0] = (int)fNY[0];
  nNX[1] = (int)fNX[1];
  nNY[1] = (int)fNY[1];
  nNX[2] = (int)fNX[2];
  nNY[2] = (int)fNY[2];

  fVX[0] = fX[1] - fX[0];
  fVY[0] = fY[1] - fY[0];
  fVX[1] = fX[2] - fX[0];
  fVY[1] = fY[2] - fY[0];

  fD = fVX[0] * fVY[1] - fVY[0] * fVX[1];
  if (isDoubleAlmostEquals(fD,0.0)) {
    return;
  }

  //// Texture coordinate correction
  //// TODO: slightly faster when converting to Fxp?
  if (tex_cv) {
    fU_[0] = (1.0 / (double)(1 << 16)) * (double)(m_pDrawTri->nUU[0]);
    fV_[0] = (1.0 / (double)(1 << 16)) * (double)(m_pDrawTri->nVV[0]);
    fU_[1] = (1.0 / (double)(1 << 16)) * (double)(m_pDrawTri->nUU[1]);
    fV_[1] = (1.0 / (double)(1 << 16)) * (double)(m_pDrawTri->nVV[1]);
    fU_[2] = (1.0 / (double)(1 << 16)) * (double)(m_pDrawTri->nUU[2]);
    fV_[2] = (1.0 / (double)(1 << 16)) * (double)(m_pDrawTri->nVV[2]);

    fVU[0] = fU_[1] - fU_[0];
    fVV[0] = fV_[1] - fV_[0];
    fVU[1] = fU_[2] - fU_[0];
    fVV[1] = fV_[2] - fV_[0];

    fK[0][0] = ( fVY[1] * (fNX[0] - fX[0]) - fVX[1] * (fNY[0] - fY[0])) / fD;
    fK[0][1] = (-fVY[0] * (fNX[0] - fX[0]) + fVX[0] * (fNY[0] - fY[0])) / fD;
    fK[1][0] = ( fVY[1] * (1.0)            - fVX[1] * (0.0)           ) / fD;
    fK[1][1] = (-fVY[0] * (1.0)            + fVX[0] * (0.0)           ) / fD;
    fK[2][0] = ( fVY[1] * (0.0)            - fVX[1] * (1.0)           ) / fD;
    fK[2][1] = (-fVY[0] * (0.0)            + fVX[0] * (1.0)           ) / fD;

    fU0  = fU_[0] + fK[0][0] * fVU[0] + fK[0][1] * fVU[1];
    fV0  = fV_[0] + fK[0][0] * fVV[0] + fK[0][1] * fVV[1];
    fAUX =          fK[1][0] * fVU[0] + fK[1][1] * fVU[1];
    fAVX =          fK[1][0] * fVV[0] + fK[1][1] * fVV[1];
    fAUY =          fK[2][0] * fVU[0] + fK[2][1] * fVU[1];
    fAVY =          fK[2][0] * fVV[0] + fK[2][1] * fVV[1];

    // Fine adjustment of texture position in multiple pixel unit drawing
    fU0 += ((double)(m_pDrawTri->nStep - 1) * 0.5) * fAUX + ((double)(m_pDrawTri->nStep - 1) * 0.5) * fAUY;
    fV0 += ((double)(m_pDrawTri->nStep - 1) * 0.5) * fAVX + ((double)(m_pDrawTri->nStep - 1) * 0.5) * fAVY;


    nUU0  = (int)((double)(1 << 16) * fU0);
    nVV0  = (int)((double)(1 << 16) * fV0);
    nAUUX = (int)((double)(1 << 16) * fAUX);
    nAVVX = (int)((double)(1 << 16) * fAVX);
    nAUUY = (int)((double)(1 << 16) * fAUY);
    nAVVY = (int)((double)(1 << 16) * fAVY);
  }

  nX0 = nNX[0];
  nY0 = nNY[0];


  //// Line move at each line

  y = m_nTriSY;
  for(y = m_nTriSY; y < m_nTriEY; y++) {
    m_pTriSetupBuffer[y].nX1 = m_rectScreen.right;
    m_pTriSetupBuffer[y].nX2 = -1;
  }


  for(i = 0; i < 3; i++) {

    int nX1, nY1, nRR1, nGG1, nBB1, nAA1, nX2, nY2, nRR2, nGG2, nBB2, nAA2;
    int nDX, nDY, nDRR, nDGG, nDBB, nDAA;
    float fZZ1;             // z of p1
    float fZZ2;             // z of p2
    float fDZZ;             // Z difference from p1 to p 2
    float fAZZ;             // Z difference per 1 px

    i1 = i;
    i2 = (i+1 < 3) ? i+1 : 0;
    if(nNY[i1] > nNY[i2]) {
      int w = i1;
      i1 = i2;
      i2 = w;
    }

    nX1 = nNX[i1];
    nY1 = nNY[i1];
    nRR1 = GetRValue(m_pDrawTri->clrVtx[i1]) << 16;
    nGG1 = GetGValue(m_pDrawTri->clrVtx[i1]) << 16;
    nBB1 = GetBValue(m_pDrawTri->clrVtx[i1]) << 16;
    nAA1 = GetAValue(m_pDrawTri->clrVtx[i1]) << 16;
    nX2 = nNX[i2];
    nY2 = nNY[i2];
    nRR2 = GetRValue(m_pDrawTri->clrVtx[i2]) << 16;
    nGG2 = GetGValue(m_pDrawTri->clrVtx[i2]) << 16;
    nBB2 = GetBValue(m_pDrawTri->clrVtx[i2]) << 16;
    nAA2 = GetAValue(m_pDrawTri->clrVtx[i2]) << 16;
    nDX = nX2 - nX1;
    nDY = nY2 - nY1;
    nDRR = nRR2 - nRR1;
    nDGG = nGG2 - nGG1;
    nDBB = nBB2 - nBB1;
    nDAA = nAA2 - nAA1;
    if (gEnableZBuffer) {
      fZZ1 = m_pDrawTri->fZ[i1];
      fZZ2 = m_pDrawTri->fZ[i2];
      fDZZ = fZZ2 - fZZ1;
    }

    if(nY1 >= m_rectScreen.bottom || nY2 <= 0) {
      // Line out of screen
      continue;
    }

    nSignX = (nDX > 0) ? 1 : -1;
    nDX = abs(nDX);
    nDY = abs(nDY);
    nAX = 0; nARR = 0; nAGG = 0; nABB = 0; nAAA = 0;
    if(nDY > 0) {
      nAX = nSignX * (nDX << 16) / nDY;
      nARR = nDRR / nDY;
      nAGG = nDGG / nDY;
      nABB = nDBB / nDY;
      nAAA = nDAA / nDY;
      if (gEnableZBuffer) {
        fAZZ = fDZZ / nDY;
      }
    }
    // Clip at the top and bottom of the screen
    if(nY1 < 0 || nY2 > m_rectScreen.bottom) {
      if(nY1 < 0) {
        nX1 -= (nY1 * nAX) >> 16;
        nRR1 -= nY1 * nARR;
        nGG1 -= nY1 * nAGG;
        nBB1 -= nY1 * nABB;
        nAA1 -= nY1 * nAAA;
        if (gEnableZBuffer) {
          fZZ1 -= nY1 * fAZZ;
        }
        nY1 = 0;
      }
      if(nY2 > m_rectScreen.bottom) {
        nX2 -= ((nY2 - m_rectScreen.bottom) * nAX) >> 16;
        nRR2 -= (nY2 - m_rectScreen.bottom) * nARR;
        nGG2 -= (nY2 - m_rectScreen.bottom) * nAGG;
        nBB2 -= (nY2 - m_rectScreen.bottom) * nABB;
        nAA2 -= (nY2 - m_rectScreen.bottom) * nAAA;
        if (gEnableZBuffer) {
          fZZ2 -= (nY2 - m_rectScreen.bottom) * fAZZ;
        }
        nY2 = m_rectScreen.bottom;
      }
      nDX = nX2 - nX1;
      nDY = nY2 - nY1;
      nSignX = (nDX > 0) ? 1 : -1;
      nDX = abs(nDX);
      nDY = abs(nDY);
      if(nDY > 0) {
        nAX = nSignX * (nDX << 16) / nDY;
      }
    }

    // Line move
    if(nY1 == nY2) {
      if(nX1 < nX2) {
        if(nX1 < m_pTriSetupBuffer[nY1].nX1) {
          m_pTriSetupBuffer[nY1].nX1 = nX1;
          m_pTriSetupBuffer[nY1].nRR1 = nRR1;
          m_pTriSetupBuffer[nY1].nGG1 = nGG1;
          m_pTriSetupBuffer[nY1].nBB1 = nBB1;
          m_pTriSetupBuffer[nY1].nAA1 = nAA1;
          if (tex_cv) {
            m_pTriSetupBuffer[nY1].nUU1 = nUU0 + nAUUX * (nX1 - nX0) + nAUUY * (nY1 - nY0);
            m_pTriSetupBuffer[nY1].nVV1 = nVV0 + nAVVX * (nX1 - nX0) + nAVVY * (nY1 - nY0);
          }
          if (gEnableZBuffer) {
            m_pTriSetupBuffer[nY1].fZZ1 = fZZ1;
          }
        }
        if(nX2 > m_pTriSetupBuffer[nY1].nX2) {
          m_pTriSetupBuffer[nY1].nX2 = nX2;
          m_pTriSetupBuffer[nY1].nRR2 = nRR2;
          m_pTriSetupBuffer[nY1].nGG2 = nGG2;
          m_pTriSetupBuffer[nY1].nBB2 = nBB2;
          m_pTriSetupBuffer[nY1].nAA2 = nAA2;
          if (tex_cv) {
            m_pTriSetupBuffer[nY1].nUU2 = nUU0 + nAUUX * (nX2 - nX0) + nAUUY * (nY1 - nY0);
            m_pTriSetupBuffer[nY1].nVV2 = nVV0 + nAVVX * (nX2 - nX0) + nAVVY * (nY1 - nY0);
          }
          if (gEnableZBuffer) {
            m_pTriSetupBuffer[nY1].fZZ2 = fZZ2;
          }
        }
      }else{
        if(nX2 < m_pTriSetupBuffer[nY1].nX1) {
          m_pTriSetupBuffer[nY1].nX1 = nX2;
          m_pTriSetupBuffer[nY1].nRR1 = nRR2;
          m_pTriSetupBuffer[nY1].nGG1 = nGG2;
          m_pTriSetupBuffer[nY1].nBB1 = nBB2;
          m_pTriSetupBuffer[nY1].nAA1 = nAA2;
          if (tex_cv) {
            m_pTriSetupBuffer[nY1].nUU1 = nUU0 + nAUUX * (nX2 - nX0) + nAUUY * (nY1 - nY0);
            m_pTriSetupBuffer[nY1].nVV1 = nVV0 + nAVVX * (nX2 - nX0) + nAVVY * (nY1 - nY0);
          }
          if (gEnableZBuffer) {
            m_pTriSetupBuffer[nY1].fZZ1 = fZZ2;
          }
        }
        if(nX1 > m_pTriSetupBuffer[nY1].nX2) {
          m_pTriSetupBuffer[nY1].nX2 = nX1;
          m_pTriSetupBuffer[nY1].nRR2 = nRR1;
          m_pTriSetupBuffer[nY1].nGG2 = nGG1;
          m_pTriSetupBuffer[nY1].nBB2 = nBB1;
          m_pTriSetupBuffer[nY1].nAA2 = nAA1;
          if (tex_cv) {
            m_pTriSetupBuffer[nY1].nUU2 = nUU0 + nAUUX * (nX1 - nX0) + nAUUY * (nY1 - nY0);
            m_pTriSetupBuffer[nY1].nVV2 = nVV0 + nAVVX * (nX1 - nX0) + nAVVY * (nY1 - nY0);
          }
          if (gEnableZBuffer) {
            m_pTriSetupBuffer[nY1].fZZ2 = fZZ1;
          }
        }
      }
    }else{
      int nXX = (nX1 << 16) + (1 << 15);
      int nRR = nRR1;
      int nGG = nGG1;
      int nBB = nBB1;
      int nAA = nAA1;
      float fZZ = fZZ1;
      int nY = nY1;
      for(j = 0; j < nDY; j++) {
        int nX = nXX >> 16;
        if(nX < m_pTriSetupBuffer[nY].nX1) {
          m_pTriSetupBuffer[nY].nX1 = nX;
          m_pTriSetupBuffer[nY].nRR1 = nRR;
          m_pTriSetupBuffer[nY].nGG1 = nGG;
          m_pTriSetupBuffer[nY].nBB1 = nBB;
          m_pTriSetupBuffer[nY].nAA1 = nAA;
          if (tex_cv) {
            m_pTriSetupBuffer[nY].nUU1 = nUU0 + nAUUX * (nX - nX0) + nAUUY * (nY - nY0);
            m_pTriSetupBuffer[nY].nVV1 = nVV0 + nAVVX * (nX - nX0) + nAVVY * (nY - nY0);
          }
          if (gEnableZBuffer) {
            m_pTriSetupBuffer[nY].fZZ1 = fZZ;
          }
        }
        if(nX > m_pTriSetupBuffer[nY].nX2) {
          m_pTriSetupBuffer[nY].nX2 = nX;
          m_pTriSetupBuffer[nY].nRR2 = nRR;
          m_pTriSetupBuffer[nY].nGG2 = nGG;
          m_pTriSetupBuffer[nY].nBB2 = nBB;
          m_pTriSetupBuffer[nY].nAA2 = nAA;
          if (tex_cv) {
            m_pTriSetupBuffer[nY].nUU2 = nUU0 + nAUUX * (nX - nX0) + nAUUY * (nY - nY0);
            m_pTriSetupBuffer[nY].nVV2 = nVV0 + nAVVX * (nX - nX0) + nAVVY * (nY - nY0);
          }
          if (gEnableZBuffer) {
            m_pTriSetupBuffer[nY].fZZ2 = fZZ;
          }
        }
        nXX += nAX;
        nRR += nARR;
        nGG += nAGG;
        nBB += nABB;
        nAA += nAAA;
        fZZ += fAZZ;
        nY++;
      }
    }
  }

  // Clip at left and right edges of screen
  for(y = m_nTriSY; y < m_nTriEY; y++) {
    int nX1 = m_pTriSetupBuffer[y].nX1;
    int nX2 = m_pTriSetupBuffer[y].nX2;
    int nDX = nX2 - nX1;
    if(nX1 > m_rectScreen.right || nX2 < 0 || nDX == 0) {
      continue;
    }

    if(nX1 < 0 || nX2 > m_rectScreen.right) {
      int nUU1, nUU2, nVV1, nVV2, nAUU, nAVV;
      float fZZ1, fZZ2, fAZZ;
      int nRR1 = m_pTriSetupBuffer[y].nRR1;
      int nGG1 = m_pTriSetupBuffer[y].nGG1;
      int nBB1 = m_pTriSetupBuffer[y].nBB1;
      int nAA1 = m_pTriSetupBuffer[y].nAA1;
      int nRR2 = m_pTriSetupBuffer[y].nRR2;
      int nGG2 = m_pTriSetupBuffer[y].nGG2;
      int nBB2 = m_pTriSetupBuffer[y].nBB2;
      int nAA2 = m_pTriSetupBuffer[y].nAA2;
      int nARR = (nRR2 - nRR1) / nDX;
      int nAGG = (nGG2 - nGG1) / nDX;
      int nABB = (nBB2 - nBB1) / nDX;
      int nAAA = (nAA2 - nAA1) / nDX;
      if (tex_cv) {
        nUU1 = m_pTriSetupBuffer[y].nUU1;
        nUU2 = m_pTriSetupBuffer[y].nUU2;
        nVV1 = m_pTriSetupBuffer[y].nVV1;
        nVV2 = m_pTriSetupBuffer[y].nVV2;
        nAUU = (nUU2 - nUU1) / nDX;
        nAVV = (nVV2 - nVV1) / nDX;
      }
      if (gEnableZBuffer) {
        fZZ1 = m_pTriSetupBuffer[y].fZZ1;
        fZZ2 = m_pTriSetupBuffer[y].fZZ2;
        fAZZ = (fZZ2 - fZZ1) / nDX;
      }
      if(nX1 < 0) {
        nRR1 -= nX1 * nARR;
        nGG1 -= nX1 * nAGG;
        nBB1 -= nX1 * nABB;
        nAA1 -= nX1 * nAAA;
        if (tex_cv) {
          nUU1 -= nX1 * nAUU;
          nVV1 -= nX1 * nAVV;
        }
        if (gEnableZBuffer) {
          fZZ1 -= nX1 * fAZZ;
        }
        nX1 = 0;
      }

      if(nX2 > m_rectScreen.right) {
        nRR2 -= (nX2 - m_rectScreen.right) * nARR;
        nGG2 -= (nX2 - m_rectScreen.right) * nAGG;
        nBB2 -= (nX2 - m_rectScreen.right) * nABB;
        nAA2 -= (nX2 - m_rectScreen.right) * nAAA;
        if (tex_cv) {
          nUU2 -= (nX2 - m_rectScreen.right) * nAUU;
          nVV2 -= (nX2 - m_rectScreen.right) * nAVV;
        }
        if (gEnableZBuffer) {
          fZZ2 -= (nX2 - m_rectScreen.right) * fAZZ;
        }
        nX2 = m_rectScreen.right;
      }

      m_pTriSetupBuffer[y].nX1 = nX1;
      m_pTriSetupBuffer[y].nX2 = nX2;
      m_pTriSetupBuffer[y].nRR1 = nRR1;
      m_pTriSetupBuffer[y].nGG1 = nGG1;
      m_pTriSetupBuffer[y].nBB1 = nBB1;
      m_pTriSetupBuffer[y].nAA1 = nAA1;
      m_pTriSetupBuffer[y].nRR2 = nRR2;
      m_pTriSetupBuffer[y].nGG2 = nGG2;
      m_pTriSetupBuffer[y].nBB2 = nBB2;
      m_pTriSetupBuffer[y].nAA2 = nAA2;
      if (tex_cv) {
        m_pTriSetupBuffer[y].nUU1 = nUU1;
        m_pTriSetupBuffer[y].nUU2 = nUU2;
        m_pTriSetupBuffer[y].nVV1 = nVV1;
        m_pTriSetupBuffer[y].nVV2 = nVV2;
      }
      if (gEnableZBuffer) {
        m_pTriSetupBuffer[y].fZZ1 = fZZ1;
        m_pTriSetupBuffer[y].fZZ2 = fZZ2;
      }
    }
  }
}


#if 0
//////////////////////////////////////////////////////////////////////////////
// TriangleSetup_Tex
//   Enable texture, enable color gouraud, disable alpha gouraud
static void TriangleSetup_Tex()
{
  double fX[3], fY[3];
  double fNX[3], fNY[3];
  int nNX[3], nNY[3];
  double fVX[2], fVY[2];
  double fD = fVX[0] * fVY[1] - fVY[0] * fVX[1];
  double fU_[3], fV_[3];
  double fVU[2], fVV[2];
  double fK[3][2];
  double fU0, fV0, fAUX, fAVX, fAUY, fAVY;
  int nUU0, nVV0, nAUUX, nAVVX, nAUUY, nAVVY;
  int nX0, nY0;
  int y;
  int nSignX, nAX;
  int i, j;

  fX[0] = m_pDrawTri->fX[0];
  fY[0] = m_pDrawTri->fY[0];
  fX[1] = m_pDrawTri->fX[1];
  fY[1] = m_pDrawTri->fY[1];
  fX[2] = m_pDrawTri->fX[2];
  fY[2] = m_pDrawTri->fY[2];

  fNX[0] = floor(fX[0]);
  fNY[0] = floor(fY[0]);
  fNX[1] = floor(fX[1]);
  fNY[1] = floor(fY[1]);
  fNX[2] = floor(fX[2]);
  fNY[2] = floor(fY[2]);

  nNX[0] = (int)fNX[0];
  nNY[0] = (int)fNY[0];
  nNX[1] = (int)fNX[1];
  nNY[1] = (int)fNY[1];
  nNX[2] = (int)fNX[2];
  nNY[2] = (int)fNY[2];

  fVX[0] = fX[1] - fX[0];
  fVY[0] = fY[1] - fY[0];
  fVX[1] = fX[2] - fX[0];
  fVY[1] = fY[2] - fY[0];

  if(fD == 0.0) {
    // Area is zero
    return;
  }

  //// Texture coordinate correction
  //// TODO: slightly faster when converting to Fxp?

  fU_[0] = (1.0 / (double)(1 << 16)) * (double)(m_pDrawTri->nUU[0]);
  fV_[0] = (1.0 / (double)(1 << 16)) * (double)(m_pDrawTri->nVV[0]);
  fU_[1] = (1.0 / (double)(1 << 16)) * (double)(m_pDrawTri->nUU[1]);
  fV_[1] = (1.0 / (double)(1 << 16)) * (double)(m_pDrawTri->nVV[1]);
  fU_[2] = (1.0 / (double)(1 << 16)) * (double)(m_pDrawTri->nUU[2]);
  fV_[2] = (1.0 / (double)(1 << 16)) * (double)(m_pDrawTri->nVV[2]);


  fVU[0] = fU_[1] - fU_[0];
  fVV[0] = fV_[1] - fV_[0];
  fVU[1] = fU_[2] - fU_[0];
  fVV[1] = fV_[2] - fV_[0];

  fK[0][0] = (  fVY[1] * (fNX[0] - fX[0]) - fVX[1] * (fNY[0] - fY[0])) / fD;
  fK[0][1] = (-fVY[0] * (fNX[0] - fX[0]) + fVX[0] * (fNY[0] - fY[0])) / fD;
  fK[1][0] = (  fVY[1] * (1.0)            - fVX[1] * (0.0)           ) / fD;
  fK[1][1] = (-fVY[0] * (1.0)            + fVX[0] * (0.0)           ) / fD;
  fK[2][0] = (  fVY[1] * (0.0)            - fVX[1] * (1.0)           ) / fD;
  fK[2][1] = (-fVY[0] * (0.0)            + fVX[0] * (1.0)           ) / fD;

  fU0  = fU_[0] + fK[0][0] * fVU[0] + fK[0][1] * fVU[1];
  fV0  = fV_[0] + fK[0][0] * fVV[0] + fK[0][1] * fVV[1];
  fAUX =          fK[1][0] * fVU[0] + fK[1][1] * fVU[1];
  fAVX =          fK[1][0] * fVV[0] + fK[1][1] * fVV[1];
  fAUY =          fK[2][0] * fVU[0] + fK[2][1] * fVU[1];
  fAVY =          fK[2][0] * fVV[0] + fK[2][1] * fVV[1];

  // Fine adjustment of texture position in multiple pixel unit drawing
  fU0 += ((double)(m_pDrawTri->nStep - 1) * 0.5) * fAUX + ((double)(m_pDrawTri->nStep - 1) * 0.5) * fAUY;
  fV0 += ((double)(m_pDrawTri->nStep - 1) * 0.5) * fAVX + ((double)(m_pDrawTri->nStep - 1) * 0.5) * fAVY;


  nUU0  = (int)((double)(1 << 16) * fU0);
  nVV0  = (int)((double)(1 << 16) * fV0);
  nAUUX = (int)((double)(1 << 16) * fAUX);
  nAVVX = (int)((double)(1 << 16) * fAVX);
  nAUUY = (int)((double)(1 << 16) * fAUY);
  nAVVY = (int)((double)(1 << 16) * fAVY);


  nX0 = nNX[0];
  nY0 = nNY[0];

  //// Line move at each line

  y = m_nTriSY;
  for(y = m_nTriSY; y < m_nTriEY; y++) {
    m_pTriSetupBuffer[y].nX1 = m_rectScreen.right;
    m_pTriSetupBuffer[y].nX2 = -1;
  }

  for(i = 0; i < 3; i++) {

    int nX1, nY1, nX2, nY2, nDX, nDY;

    int i1 = i;
    int i2 = (i+1 < 3) ? i+1 : 0;
    if(nNY[i1] > nNY[i2]) {
      int w = i1;
      i1 = i2;
      i2 = w;
    }

    nX1 = nNX[i1];
    nY1 = nNY[i1];
    nX2 = nNX[i2];
    nY2 = nNY[i2];
    nDX = nX2 - nX1;
    nDY = nY2 - nY1;

    if(nY1 >= m_rectScreen.bottom || nY2 <= 0) {
      // Line out of screen
      continue;
    }


    nSignX = (nDX > 0) ? 1 : -1;
    nDX = abs(nDX);
    nDY = abs(nDY);
    nAX = 0;
    if(nDY > 0) {
      nAX = nSignX * (nDX << 16) / nDY;
    }
    // Clip at the top and bottom of the screen
    if(nY1 < 0 || nY2 > m_rectScreen.bottom) {
      if(nY1 < 0) {
        nX1 -= (nY1 * nAX) >> 16;
        nY1 = 0;
      }
      if(nY2 > m_rectScreen.bottom) {
        nX2 -= ((nY2 - m_rectScreen.bottom) * nAX) >> 16;
        nY2 = m_rectScreen.bottom;
      }
      nDX = nX2 - nX1;
      nDY = nY2 - nY1;
      nSignX = (nDX > 0) ? 1 : -1;
      nDX = abs(nDX);
      nDY = abs(nDY);
      if(nDY > 0) {
        nAX = nSignX * (nDX << 16) / nDY;
      }
    }

    // Line move
    if(nY1 == nY2) {
      if(nX1 < nX2) {
        if(nX1 < m_pTriSetupBuffer[nY1].nX1) {
          m_pTriSetupBuffer[nY1].nX1 = nX1;
          m_pTriSetupBuffer[nY1].nUU1 = nUU0 + nAUUX * (nX1 - nX0) + nAUUY * (nY1 - nY0);
          m_pTriSetupBuffer[nY1].nVV1 = nVV0 + nAVVX * (nX1 - nX0) + nAVVY * (nY1 - nY0);
        }
        if(nX2 > m_pTriSetupBuffer[nY1].nX2) {
          m_pTriSetupBuffer[nY1].nX2 = nX2;
          m_pTriSetupBuffer[nY1].nUU2 = nUU0 + nAUUX * (nX2 - nX0) + nAUUY * (nY1 - nY0);
          m_pTriSetupBuffer[nY1].nVV2 = nVV0 + nAVVX * (nX2 - nX0) + nAVVY * (nY1 - nY0);
        }
      }else{
        if(nX2 < m_pTriSetupBuffer[nY1].nX1) {
          m_pTriSetupBuffer[nY1].nX1 = nX2;
          m_pTriSetupBuffer[nY1].nUU1 = nUU0 + nAUUX * (nX2 - nX0) + nAUUY * (nY1 - nY0);
          m_pTriSetupBuffer[nY1].nVV1 = nVV0 + nAVVX * (nX2 - nX0) + nAVVY * (nY1 - nY0);
        }
        if(nX1 > m_pTriSetupBuffer[nY1].nX2) {
          m_pTriSetupBuffer[nY1].nX2 = nX1;
          m_pTriSetupBuffer[nY1].nUU2 = nUU0 + nAUUX * (nX1 - nX0) + nAUUY * (nY1 - nY0);
          m_pTriSetupBuffer[nY1].nVV2 = nVV0 + nAVVX * (nX1 - nX0) + nAVVY * (nY1 - nY0);
        }
      }
    }else{
      int nXX = (nX1 << 16) + (1 << 15);
      int nY = nY1;
      for(j = 0; j < nDY; j++) {
        int nX = nXX >> 16;
        if(nX < m_pTriSetupBuffer[nY].nX1) {
          m_pTriSetupBuffer[nY].nX1 = nX;
          m_pTriSetupBuffer[nY].nUU1 = nUU0 + nAUUX * (nX - nX0) + nAUUY * (nY - nY0);
          m_pTriSetupBuffer[nY].nVV1 = nVV0 + nAVVX * (nX - nX0) + nAVVY * (nY - nY0);
        }
        if(nX > m_pTriSetupBuffer[nY].nX2) {
          m_pTriSetupBuffer[nY].nX2 = nX;
          m_pTriSetupBuffer[nY].nUU2 = nUU0 + nAUUX * (nX - nX0) + nAUUY * (nY - nY0);
          m_pTriSetupBuffer[nY].nVV2 = nVV0 + nAVVX * (nX - nX0) + nAVVY * (nY - nY0);
        }
        nXX += nAX;
        nY++;
      }
    }
  }

  // Clip at left and right edges of screen
  for(y = m_nTriSY; y < m_nTriEY; y++) {
    int nX1 = m_pTriSetupBuffer[y].nX1;
    int nX2 = m_pTriSetupBuffer[y].nX2;
    int nDX = nX2 - nX1;
    // 2006.04.05. Add
    if(nX1 > m_rectScreen.right || nX2 < 0 || nDX == 0) {
      continue;
    }
    if(nX1 < 0 || nX2 > m_rectScreen.right) {
      int nUU1 = m_pTriSetupBuffer[y].nUU1;
      int nUU2 = m_pTriSetupBuffer[y].nUU2;
      int nVV1 = m_pTriSetupBuffer[y].nVV1;
      int nVV2 = m_pTriSetupBuffer[y].nVV2;
      int nAUU = (nUU2 - nUU1) / nDX;
      int nAVV = (nVV2 - nVV1) / nDX;
      if(nX1 < 0) {
        nUU1 -= nX1 * nAUU;
        nVV1 -= nX1 * nAVV;
        nX1 = 0;
      }
      if(nX2 > m_rectScreen.right) {
        nUU2 -= (nX2 - m_rectScreen.right) * nAUU;
        nVV2 -= (nX2 - m_rectScreen.right) * nAVV;
        nX2 = m_rectScreen.right;
      }
      m_pTriSetupBuffer[y].nX1 = nX1;
      m_pTriSetupBuffer[y].nX2 = nX2;
      m_pTriSetupBuffer[y].nUU1 = nUU1;
      m_pTriSetupBuffer[y].nUU2 = nUU2;
      m_pTriSetupBuffer[y].nVV1 = nVV1;
      m_pTriSetupBuffer[y].nVV2 = nVV2;
    }
  }
}
#endif


//////////////////////////////////////////////////////////////////////
//	DrawTriangle function
//////////////////////////////////////////////////////////////////////
static void DrawTriangle(PDRAWTRI pDrawTri)
{
  double fVX[2], fVY[2];
  double fD;

  m_pDrawTri = pDrawTri;

  if(m_pDrawTri->bBiLinear) {
    m_pDrawTri->nStep = 1;
  }

  // Determine whether it is within screen
  RECT rc;
  rc.left = (int)m_pDrawTri->fX[0];
  rc.left = min(rc.left, (int)m_pDrawTri->fX[1]);
  rc.left = min(rc.left, (int)m_pDrawTri->fX[2]);

  rc.right = (int)m_pDrawTri->fX[0];
  rc.right = max(rc.right, (int)m_pDrawTri->fX[1]);
  rc.right = max(rc.right, (int)m_pDrawTri->fX[2]);

  rc.top = (int)m_pDrawTri->fY[0];
  rc.top = min(rc.top, (int)m_pDrawTri->fY[1]);
  rc.top = min(rc.top, (int)m_pDrawTri->fY[2]);

  rc.bottom = (int)m_pDrawTri->fY[0];
  rc.bottom = max(rc.bottom, (int)m_pDrawTri->fY[1]);
  rc.bottom = max(rc.bottom, (int)m_pDrawTri->fY[2]);

  if(rc.left >= m_rectScreen.right || rc.right <= 0 || rc.top >= m_rectScreen.bottom || rc.bottom <= 0) {
    // out of screen
    return;
  }

  // Drawing / non-drawing judgment by outer product (Do not draw if area is zero)
  fVX[0] = pDrawTri->fX[1] - pDrawTri->fX[0];
  fVY[0] = pDrawTri->fY[1] - pDrawTri->fY[0];
  fVX[1] = pDrawTri->fX[2] - pDrawTri->fX[0];
  fVY[1] = pDrawTri->fY[2] - pDrawTri->fY[0];

  fD = fVX[0] * fVY[1] - fVY[0] * fVX[1];
  if(isDoubleAlmostEquals(fD,0.0)) {
    // Area is zero
    return;
  }

  m_nTriSY = max(rc.top,    0);
  m_nTriEY = min(rc.bottom, m_rectScreen.bottom);


  TriangleSetup_Tex_ClrGouraud();
  DrawTriangle_ClrGouraud_Normal();
}


///////////////////////////////////////////////////////////////////////////////
//	StlDrawTriangle
///////////////////////////////////////////////////////////////////////////////
void StlDrawTriangle(PSTL_DRAWTRI pStlDrawTri)
{
  // as well as STL_DRAWTRI
  DRAWTRI DrawTri;

  DrawTri.nStep = 1;

  DrawTri.clrVtx[0] = pStlDrawTri->clrVtx[0];
  DrawTri.clrVtx[1] = pStlDrawTri->clrVtx[1];
  DrawTri.clrVtx[2] = pStlDrawTri->clrVtx[2];
  DrawTri.clrSky[0] = 0;
  DrawTri.clrSky[1] = 0;
  DrawTri.clrSky[2] = 0;
  DrawTri.nAlpha[0] = 255;
  DrawTri.nAlpha[1] = 255;
  DrawTri.nAlpha[2] = 255;


  DrawTri.fX[0] = pStlDrawTri->po[0].m_fX;
  DrawTri.fY[0] = pStlDrawTri->po[0].m_fY;
  DrawTri.fX[1] = pStlDrawTri->po[1].m_fX;
  DrawTri.fY[1] = pStlDrawTri->po[1].m_fY;
  DrawTri.fX[2] = pStlDrawTri->po[2].m_fX;
  DrawTri.fY[2] = pStlDrawTri->po[2].m_fY;

  // Z-Buffer
  if (gEnableZBuffer) {
    DrawTri.fZ[0] = pStlDrawTri->z[0];
    DrawTri.fZ[1] = pStlDrawTri->z[1];
    DrawTri.fZ[2] = pStlDrawTri->z[2];
  }

  // texture coordinate
  if (tex_cv) {
    int w = GetTexWidth()  - 1;
    int h = GetTexHeight() - 1;
    DrawTri.nUU[0] = (int)(w * pStlDrawTri->fUU[0]) << 16;
    DrawTri.nVV[0] = (int)(h * pStlDrawTri->fVV[0]) << 16;
    DrawTri.nUU[1] = (int)(w * pStlDrawTri->fUU[1]) << 16;
    DrawTri.nVV[1] = (int)(h * pStlDrawTri->fVV[1]) << 16;
    DrawTri.nUU[2] = (int)(w * pStlDrawTri->fUU[2]) << 16;
    DrawTri.nVV[2] = (int)(h * pStlDrawTri->fVV[2]) << 16;
  }

  DrawTriangle(&DrawTri);
}


/**
 * set texture
 * @param texture  texture image. NULL will release it.
 */
void tri_set_texture(canvas* texture)
{
  tex_cv = texture;
}

/**
 * set enable/disable of Z-Buffer
 */
void tri_set_depth_test(int flag)
{
  gEnableZBuffer = flag;
}

void tri_set_depth_mask(int flag)
{
  gDepthMask = flag;
}

/**
 * set enable/disable of color blend
 */
void tri_set_blend(int flag)
{
  gEnableBlend = flag;
}

/**
 * Set blend method
 */
void tri_set_blend_func(int sfactor, int dfactor)
{
  gBlendSrcFactor = sfactor;
  gBlendDstFactor = dfactor;
}
