/**
 * @file      wms.c
 * @brief     Web Mapping Service handling class
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2010-2011 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#ifdef ENABLE_WMS

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include "wms.h"
#include "wms_error.h"

typedef wmsBool (* xml_find_node_callback)(xmlNodePtr node, void* userdata);
static xmlNodePtr _xml_find_node(xmlNodePtr node, xml_find_node_callback func, void* userdata);

wms* wms_create(void)
{
  wms* self = malloc(sizeof(wms));
  if (!self) {
    return NULL;
  }
  memset(self, 0, sizeof(wms));
  self->error = wmsError_create();
  self->formats = strbary_create(10);
  return self;
}

void wms_free(wms* self)
{
  wms_clear(self);
  wmsError_clear(self->error);
  strbary_free(self->formats);
  free(self);
}

void wms_clear(wms* self)
{
  if (self->error) {
    wmsError_clear(self->error);
  }
  if (self->xml) {
    xmlFreeDoc(self->xml);
    self->xml = NULL;
  }
  if (self->query_url) {
    free(self->query_url);
    self->query_url = NULL;
  }
  if (self->layer) {
    wmsLayer_free(self->layer);
    self->layer = NULL;
  }
  if (strbary_size(self->formats)) {
    strbary_truncate(self->formats, 0);
  }
  self->version.major = 0;
  self->version.minor = 0;
  self->version.release = 0;
}

wmsVersion wms_version(wms* self)
{
  return self->version;
}

const char* wms_query_url(wms* self)
{
  return self->query_url;
}

wmsBool wms_has_format(wms* self, const char* format)
{
  strb* s;
  char* str;
  int max = strbary_size(self->formats);
  int i;

  for (i = 0; i < max; i++) {
    s = strbary_at(self->formats, i);
    str = (char *)strb_ptr(s);
    if (0 == strcmp(str, format)) {
      return WMS_TRUE;
    }
  }
  return WMS_FALSE;
}

wmsLayer* wms_root_layer(wms* self)
{
  return self->layer;
}

size_t wms_named_layer_count(wms* self)
{
  size_t n = 0;
  wmsLayer* layer = self->layer;
  while (layer) {
    n++;
    layer = wmsLayer_next(layer);
  }
  return n;
}

wmsLayer* wms_named_layer_at(wms* self, size_t index)
{
  size_t n = 0;
  wmsLayer* layer = self->layer;
  for (n = 0; n < index; n++) {
    layer = wmsLayer_next(layer);
    if (!layer) {
      return NULL;
    }
  }
  return layer;
}

wmsLayer* wms_find_layer(wms* self, enum WMSFindType type, const char* key)
{
  wmsLayer* layer = self->layer;
  const char* s;

  while (layer) {
    if (type == WMSFindTypeTitle) {
      s = wmsLayer_title(layer);
    }
    else if (type == WMSFindTypeName) {
      s = wmsLayer_name(layer);
    }
    if (0 == strcmp(s, key)) {
      return layer;
    }
    layer = wmsLayer_next(layer);
  }
  return NULL;
}

wmsBool wmsRegion_include(wmsRegion self, wmsRegion other)
{
  return (self.min.lng <= other.min.lng &&
          self.min.lat <= other.min.lat &&
          self.max.lng >= other.max.lng &&
          self.max.lat >= other.max.lat) ? WMS_TRUE : WMS_FALSE;
}

static xmlNodePtr _xml_find_node(xmlNodePtr node, xml_find_node_callback func, void* userdata)
{
  xmlNodePtr c_node;
  xmlNodePtr m_node;

  while (node) {
    if (func(node, userdata)) {
      return node;
    }
    c_node = node->children;
    if (c_node && (m_node = _xml_find_node(c_node, func, userdata)) != NULL) {
      return m_node;
    }
    node = node->next;
  }
  return NULL;
}

#endif
