/**
 * @file      wms.h
 * @brief     Web Mapping Service handling class
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2010-2011 Japan Aerospace Exploration Agency
 *
 */

#ifndef _WMS_H
#define _WMS_H

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <stdint.h>
#include <pthread.h>
#include "strb.h"
#include "memb.h"
#include "wms_layer.h"

#define WMS_USER_AGENT "wms-client/1.0"

struct _wmsReq;
struct _wmsRes;
typedef void (* wmsReqCallback)(struct _wmsReq *,struct _wmsRes *,void *);

/* **********************************************************************
* Enums
********************************************************************** */

/**
 * Error codes of wms functions
 */
enum WMSErrorCode {
  WMS_NO_ERROR = 0,
  WMS_IO_ERROR,
  WMS_MEMORY_ERROR,
  WMS_HOST_NOT_FOUND_ERROR,
  WMS_FILE_NOT_FOUND_ERROR,
  WMS_CONNECTION_REFUSED_ERROR,
  WMS_FORMAT_ERROR,
  WMS_RANGE_ERROR,
  WMS_LAYER_NOT_FOUND_ERROR,
  WMS_VERSION_NOT_COMPATIBLE_ERROR,
  WMS_CREATE_XML_CONTEXT_ERROR,
  WMS_XPATH_EVAL_EXPRESSION_ERROR,
  WMS_XPATH_RESULT_EMPTY_ERROR,
  WMS_XML_ATTRIBUTE_NOT_FOUND_ERROR,
  WMS_PARSE_ERROR,
  WMS_CURL_INIT_ERROR,
  WMS_HTTP_REQUEST_ERROR,
  WMS_UNKNOWN_CRS_ERROR,
  WMS_XPATH_REGISTER_NS_ERROR
};

/**
 * Find target of wms_find_layer
 */
enum WMSFindType {
  WMSFindTypeTitle = 1,         /**< Title search */
  WMSFindTypeName               /**< Name search */
};

/**
 * Coordinate Reference Systems (CRS) in WMS
 */
enum WMSCrs {
  WMSCrsUnsupportedLayer = -1,  /**< Unsupported Layer */
  WMSCrsCRS83    = 83,          /**< EPSG:4269 lon/lat */
  WMSCrsEPSG4326 = 4326         /**< WGS84 lat/lon */
};


/* **********************************************************************
* Structs
********************************************************************** */

/**
 * WMS Error structure
 */
typedef struct _wmsError {
  enum WMSErrorCode code;
  strb* message;
  strb* function;
  strb* file;
  int32_t line;
} wmsError;

/**
 * WMS Version structure
 */
typedef struct _wmsVersion {
  short major;
  short minor;
  short release;
} wmsVersion;

/**
 * WMS main object
 */
typedef struct _wms {
  wmsError* error;
  wmsVersion version;
  xmlDocPtr xml;
  char* query_url;
  strbary* titles;
  strbary* names;
  strbary* formats;
  char* online_resource;
  char* exception_format;
  wmsLayer* layer;
} wms;


/**
 * WMS map request structure
 */
typedef struct _wmsReq {
  wms* w;
  wmsError* error;
  strb* title;
  strb* name;
  strb* format;
  enum WMSCrs crs;
  wmsRegion region;
  size_t width;
  size_t height;
  wmsReqCallback error_callback;
  void* error_data;
  wmsReqCallback success_callback;
  void* success_data;
  strb* url;
  pthread_t pth;
  pthread_attr_t pth_attr[1];
  pthread_mutex_t mutex[1];
  pthread_mutexattr_t mutex_attr[1];
} wmsReq;

/**
 * WMS resource structure
 */
typedef struct _wmsRes {
  int status;
  wmsError* error;
  strb* content_type;
  memb* data;
} wmsRes;

__BEGIN_DECLS
/**
 * @brief Constructor of a wms instance
 */
wms* wms_create(void);

/**
 * @brief Destructor of a wms instance
 *
 * @param[in]  self  wms instance
 */
void wms_free(wms* self);

/**
 * @brief Free contents of a wms instance.
 * The instance is reusable.
 *
 * @param[in]  self  wms instance
 */
void wms_clear(wms* self);

/**
 * @brief Get WMS version
 *
 * @param[in] self  wms instance.
 * @return wms_version() returns a wms version structure.
 * @see wmsVersion
 */
wmsVersion wms_version(wms* self);

/**
 * @brief Get WMS root layer
 *
 * @param[in]  self  wms instance
 * @return wms_root_layer() returns a wms layer instance of the root layer.
 */
wmsLayer* wms_root_layer(wms* self);

/**
 * @brief Get the number of names in named layers
 *
 * @param self  wms instance.
 * @return wms_named_layer_count() returns the number of names in named layers.
 */
size_t wms_named_layer_count(wms* self);

/**
 * @brief Get n-th layer's name
 *
 * @param[in]  self  wms instance
 * @param[in]  index index of n-th layer's name
 * @return wms_named_layer_at() returns a wms layer instance of n-th layer's
 * name.
 */
wmsLayer* wms_named_layer_at(wms* self, size_t index);

/**
 * Get the query url in a wms instance
 *
 * @param[in]  self  wms instance
 * @return wms_query_url() return the string of the query url in a wms instance.
 */
const char* wms_query_url(wms* self);

/**
 * Check the specified format is available
 *
 * @param[in]  self  wms instance
 * @param[in]  format image format
 * wms_has_format() returns WMS_TRUE if the specified format is available.
 * Otherwise, WMS_FALSE is returned.
 */
wmsBool wms_has_format(wms* self, const char* format);

/**
 * @brief Find a wms layer satisfying a requirement
 *
 * @param[in] self  wms instance
 * @param[in] type  search type
 * @param[in] key   search key
 * @return wms_find_layer() returns a wms layer instance if a layer is found
 * satisfying a requirement. Otherwise, NULL is returned.
 */
wmsLayer* wms_find_layer(wms* self, enum WMSFindType type, const char* key);

/**
 * @brief Check to see if a region is included in another region.
 *
 * @param[in] self  wms region
 * @param[in] other the other wms region
 * @return wmsRegion_include() returns WMS_TRUE if the other wms region is
 * included in the self wms region. Otherwise, WMS_FALSE is returned.
 */
wmsBool wmsRegion_include(wmsRegion self, wmsRegion other);

__END_DECLS

#endif /* _WMS_H */
