/**
 * @file      wms_error.c
 * @brief     WMS error object
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2010-2011 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdint.h>
#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include "wms_error.h"

wmsError* wmsError_create()
{
  wmsError* self;
  self = malloc(sizeof(*self));
  if (!self) {
    return NULL;
  }
  memset(self, 0, sizeof(*self));
  self->file     = strb_create(64);
  self->message  = strb_create(128);
  self->function = strb_create(64);
  wmsError_clear(self);
  return self;
}

void wmsError_free(wmsError* self)
{
  if (self) {
    if (self->message) strb_free(self->message);
    if (self->function) strb_free(self->function);
    if (self->file) strb_free(self->file);
    free(self);
  }
}

void wmsError_set_ex(wmsError* self, enum WMSErrorCode code,
                     const char* function, const char* file, int32_t line,
                     const char* message)
{
  self->code = code;
  strb_set_cstring(self->function, function);
  strb_set_cstring(self->file, file);
  self->line = line;
  strb_set_cstring(self->message, message);
}

void wmsError_format(wmsError* self, enum WMSErrorCode code,
                     const char* function, const char* file, int32_t line,
                     const char* format, ...)
{
  va_list ap;
  va_start(ap, format);
  wmsError_vformat(self, code, function, file, line, format, ap);
}
void wmsError_vformat(wmsError* self, enum WMSErrorCode code,
                      const char* function, const char* file, int32_t line,
                      const char* format, va_list ap)
{
  self->code = code;
  strb_set_cstring(self->function, function);
  strb_set_cstring(self->file, file);
  self->line = line;
  strb_vsprintf(self->message, format, ap);
}

enum WMSErrorCode wmsError_code(wmsError* self)
{
  return self->code;
}

const char* wmsError_message(wmsError* self)
{
  return strb_ptr(self->message);
}

const char* wmsError_function(wmsError* self)
{
  return strb_ptr(self->function);
}

const char* wmsError_file(wmsError* self)
{
  return strb_ptr(self->file);
}

int32_t wmsError_line(wmsError* self)
{
  return self->line;
}

void wmsError_print(wmsError* self, FILE* io)
{
  if (1 < strb_size(self->message)) {
    fprintf(io, "%s\nat file %s:%d (function %s)\n",
            (char *)strb_ptr(self->message),
            (char *)strb_ptr(self->file),
            self->line,
            (char *)strb_ptr(self->function));
  }
}

void wmsError_clear(wmsError* self)
{
  wmsError_set(self, WMS_NO_ERROR, "no error");
}
