/**
 * @file      wms_layer.c
 * @brief     WMS Layer object
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2010-2011 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include "wms_layer.h"

wmsLayer* wmsLayer_create()
{
  wmsLayer* self = malloc(sizeof(wmsLayer));
  if (!self) {
    return NULL;
  }
  memset(self, 0, sizeof(wmsLayer));
  self->srs = ptrb_create(5);
  self->crs = ptrb_create(5);
  self->child = ptrb_create(10);
  self->title = strb_create(128);
  self->name = strb_create(32);
  self->abstract = strb_create(512);
  return self;
}

void wmsLayer_free(wmsLayer* self)
{
  ptrb_each(self->child, (ptrb_each_callback)wmsLayer_free);
  ptrb_free(self->child);
  strb_free(self->title);
  strb_free(self->name);
  strb_free(self->abstract);
  ptrb_each(self->srs, (ptrb_each_callback)strb_free);
  ptrb_free(self->srs);
  ptrb_each(self->crs, (ptrb_each_callback)strb_free);
  ptrb_free(self->crs);
  free(self);
}

const char* wmsLayer_name(wmsLayer* self)
{
  return strb_ptr(self->name);
}

const char* wmsLayer_title(wmsLayer* self)
{
  return strb_ptr(self->title);
}

wmsRegion wmsLayer_region(wmsLayer* self)
{
  wmsRegion zero = {{0,0},{0,0}};
  if (self->nobbox) {
    if (self->parent) {
      return wmsLayer_region(self->parent);
    }
    else {
      return zero;
    }
  }
  return self->bbox;
}

wmsBool wmsLayer_is_fixed(wmsLayer* self)
{
  // TODO: implement
  return WMS_FALSE;
}

size_t wmsLayer_crs_length(wmsLayer* self)
{
  if (self->nocrs) {
    if (self->nosrs) {
      if (self->parent) {
        return wmsLayer_crs_length(self->parent);
      }
      else {
        return 0;
      }
    }
    else {
      return ptrb_size(self->srs);
    }
  }
  return ptrb_size(self->crs);
}

const char* wmsLayer_crs_at(wmsLayer* self, size_t index)
{
  assert(index < wmsLayer_crs_length(self));
  if (self->nocrs) {
    if (self->nosrs) {
      if (self->parent) {
        return wmsLayer_crs_at(self->parent, index);
      }
      else {
        return NULL;
      }
    }
    else {
      return strb_ptr((strb *)ptrb_at(self->srs, index));
    }
  }
  return strb_ptr((strb *)ptrb_at(self->crs, index));
}

size_t wmsLayer_child_length(wmsLayer* self)
{
  return ptrb_size(self->child);
}

wmsLayer* wmsLayer_child_at(wmsLayer* self, size_t index)
{
  assert(index < wmsLayer_child_length(self));
  return (wmsLayer *)ptrb_at(self->child, index);
}

wmsBool wmsLayer_has_crs(wmsLayer* self, const char* crs)
{
  ptrb* p;
  strb* str;
  size_t i;
  if (0 == ptrb_size(self->crs)) {
    if (0 == ptrb_size(self->srs)) {
      if (self->parent) {
        return wmsLayer_has_crs(self->parent, crs);
      }
      else {
      }
    }
    else {
      p = self->srs;
    }
  }
  else {
    p = self->crs;
  }
  for (i = 0; i < ptrb_size(p); i++) {
    str = (strb *)ptrb_at(p, i);
    if (0 == strcmp(strb_ptr(str), crs)) {
      return WMS_TRUE;
    }
  }
  return WMS_FALSE;
}

static void _wmsLayer_find_named_layer_filter(voidPtr p, void* data)
{
  struct {
    ptrb* ary;
    const char* name;
  }* udata = data;
  wmsLayer* layer = (wmsLayer *)p;
  voidPtr ptr;

  if (0 == strcmp(wmsLayer_name(layer), udata->name)) {
    ptr = layer;
    ptrb_append(udata->ary, &ptr, 1);
  }
  ptrb_each_with_data(layer->child, _wmsLayer_find_named_layer_filter, data);
}
wmsLayer* wmsLayer_find_named_layer(wmsLayer* self, const char* name)
{
  struct {
    ptrb* ary;
    const char* name;
  } data;
  wmsLayer* layer = NULL;

  data.ary = ptrb_create(5);
  data.name = name;
  ptrb_each_with_data(self->child, _wmsLayer_find_named_layer_filter, &data);
  if (ptrb_size(data.ary)) {
    layer = ptrb_at(data.ary, 0);
  }
  ptrb_free(data.ary);
  return layer;
}

static void _wmsLayer_find_titled_layer_filter(voidPtr p, void* data)
{
  struct {
    ptrb* ary;
    const char* title;
  }* udata = data;
  wmsLayer* layer = (wmsLayer *)p;
  voidPtr ptr;

  if (0 == strcmp(wmsLayer_title(layer), udata->title)) {
    ptr = layer;
    ptrb_append(udata->ary, &ptr, 1);
  }
  ptrb_each_with_data(layer->child, _wmsLayer_find_titled_layer_filter, data);
}
wmsLayer* wmsLayer_find_titled_layer(wmsLayer* self, const char* title)
{
  struct {
    ptrb* ary;
    const char* title;
  } data;
  wmsLayer* layer = NULL;

  data.ary = ptrb_create(5);
  data.title = title;
  ptrb_each_with_data(self->child, _wmsLayer_find_titled_layer_filter, &data);
  if (ptrb_size(data.ary)) {
    layer = ptrb_at(data.ary, 0);
  }
  ptrb_free(data.ary);
  return layer;
}

wmsLayer* wmsLayer_parent(wmsLayer* self)
{
  return self->parent;
}

wmsLayer* wmsLayer_next(wmsLayer* self)
{
  wmsLayer* layer;
  if (ptrb_size(self->child)) {
    return (wmsLayer *)ptrb_at(self->child, 0);
  }
  if (self->next) {
    return self->next;
  }
  layer = self;
  while (layer->parent) {
    layer = layer->parent;
    if (layer->next) {
      return layer->next;
    }
  }
  return NULL;
}
