/**
 * @file      wms_layer.h
 * @brief     WMS Layer object
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2010-2011 Japan Aerospace Exploration Agency
 *
 */

#ifndef _WMSLAYER_H
#define _WMSLAYER_H

#include <stdio.h>
#include "strb.h"
#include "ptrb.h"

#define WMS_TRUE   1
#define WMS_FALSE  0

typedef int wmsBool;

/**
 * WMS point structure
 */
typedef struct _wmsPoint {
  double lng;
  double lat;
} wmsPoint;

/**
 * WMS region structure
 */
typedef struct _wmsRegion {
  wmsPoint min;
  wmsPoint max;
} wmsRegion;

/**
 * @brief WMS layer structure
 *
 * WMS layer structure is designed by Tree structure.
 */
typedef struct _wmsLayer {
  strb* title;
  strb* name;
  strb* abstract;
  ptrb* srs;       /* <strb> */
  ptrb* crs;       /* <strb> */
  struct _wmsLayer* parent;
  struct _wmsLayer* next;
  struct _wmsLayer* previous;
  ptrb* child;       /* <wmsLayer> */
  wmsBool noscale;
  wmsBool noclip;
  wmsBool nosrs;
  wmsBool nocrs;
  wmsBool nobbox;
  wmsRegion bbox;
} wmsLayer;

__BEGIN_DECLS

/**
 * @brief Create WMS layer instance
 *
 * @return wmsLayer_create() returns a wms layer instance if success,
 * NULL otherwise.
 */
wmsLayer* wmsLayer_create();

/**
 * @brief Free a wms layer instance
 *
 * @param[in]  self  wms layer instance
 */
void wmsLayer_free(wmsLayer* self);

/**
 * @brief Getter of the layer name in the specified wms layer instance
 *
 * @param[in]  self  wms layer instance
 * @return wmsLayer_name() returns the string of the layer name in the specified
 * wms instance.
 */
const char* wmsLayer_name(wmsLayer* self);

/**
 * @brief Getter of the layer title in the specified wms layer instance
 *
 * @param[in]  self  wms layer instance
 * @return wmsLayer_title() returns the string of the layer title in the
 * specified wms instance.
 */
const char* wmsLayer_title(wmsLayer* self);

/**
 * @brief Getter of the layer region in the specified wms layer instance
 *
 * @return wmsLayer_region() returns the instance of the layer region in the
 * specified wms instance.
 */
wmsRegion wmsLayer_region(wmsLayer* self);

/**
 * @brief Check if the specified wms layer is fixed
 *
 * @param[in]  self  wms layer instance
 * @return wmsLayer_is_fixed() always returns WMS_FALSE because this function
 * is under construction.
 */
wmsBool wmsLayer_is_fixed(wmsLayer* self);

/**
 * @brief Get the crs length of a wms layer instance
 *
 * @param[in]  self  wms layer instance
 * @return wmsLayer_crs_length() returns the crs length of a wms layer instance.
 * If there is no crs or srs and there is a parent, the crs length of the
 * parent is returned.
 */
size_t wmsLayer_crs_length(wmsLayer* self);

/**
 * @brief Get the CRS string of n-th layer
 *
 * @param[in]  self  wms layer instance
 * @param[in]  index index of the layer in the wms layer instance
 * @return wmsLayer_crs_at() returns the CRS string of n-th layer in the
 * specified wms layer instance.
 */
const char* wmsLayer_crs_at(wmsLayer* self, size_t index);

/**
 * @brief Get the parent layer
 *
 * @param[in]  self  wms layer instance
 * @return wmsLayer_parent() returns the parent layer instance.
 */
wmsLayer* wmsLayer_parent(wmsLayer* self);

/**
 * @brief Get the length of children in the specified layer instance
 *
 * @param[in]  self  wms layer instance
 * @return wmsLayer_child_length() returns the length of children in the
 * specified layer instance.
 */
size_t wmsLayer_child_length(wmsLayer* self);

/**
 * @brief Get the n-th children layer
 *
 * @param[in]  self  wms layer instance
 * @param[in]  index index of children
 * @return wmsLayer_child_at() returns the wms layer instance of the children
 * specified by index.
 */
wmsLayer* wmsLayer_child_at(wmsLayer* self, size_t index);

/**
 * @brief Check if a wms layer has CRS.
 *
 * @param[in]  self  wms layer instance
 * @param[in]  crs   CRS
 * @return wmsLayer_has_crs() returns WMS_TRUE if the specified CRS is found
 * in the wms layer instance, WMS_FALSE otherwise.
 */
wmsBool wmsLayer_has_crs(wmsLayer* self, const char* crs);

/**
 * @brief Get the next sibling layer
 *
 * @param[in]  self  wms layer instance
 * @return wmsLayer_next() returns the instance of the next sibling layer.
 */
wmsLayer* wmsLayer_next(wmsLayer* self);

__END_DECLS

#endif /* _WMSLAYER_H */
