/**
 * @file      wms_parser.c
 * @brief     Load and parse WMS data
 * @date      2017-05-22
 *
 * @copyright
 * Copyright 2010-2011 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#ifdef ENABLE_WMS

#ifdef ASPRINTF_NEED_GNU_SOURCE
#  define _GNU_SOURCE
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <assert.h>
#include <curl/curl.h>
#include <libxml/xpath.h>
#include <libxml/parser.h>
#include <libxml/xpathInternals.h>
#include <libxml/tree.h>
#include <regex.h>
#include "memb.h"
#include "strb.h"
#include "wms_parser.h"
#include "wms_error.h"
#include "xml_util.h"

static wmsBool _wms_parse_capability(wms* self);
static wmsBool _wms_parse_version(wms* self);
static wmsBool _wms_parse_query_url(wms* self);
static wmsBool _wms_parse_layer(wms* self);
static wmsBool _wms_parse_format(wms* self);
static wmsLayer* _wms_parse_layer_with_layer_node(wms* self, xmlNodePtr node);
static wmsBool _wms_parser_parse_layer_bbox(wms* self, wmsLayer* layer, xmlNodePtr node);

wmsBool wms_read_from_file(wms* self, const char* path)
{
  wms_clear(self);
  self->xml = xmlReadFile(path, NULL, 0);
  if (!self->xml) {
    return WMS_FALSE;
  }
  if (!_wms_parse_version(self)) {
    return WMS_FALSE;
  }
  if (!_wms_parse_query_url(self)) {
    return WMS_FALSE;
  }
  xmlCleanupParser();
  return WMS_TRUE;
}

static size_t _wms_read_from_uri_write_callback(void* ptr, size_t size, size_t nmemb, void* data)
{
  size_t realsize = size*nmemb;
  memb* buffer = (memb *)data;
  if (!memb_append(buffer, ptr, realsize)) {
    return 0;
  }
  return realsize;
}

wmsBool wms_read_from_uri(wms* self, const char* url)
{
  CURLcode res;
  CURL* curl = NULL;
  memb* buffer = NULL;
  char* query_url = NULL;
  wmsBool success;
  const char* query = "SERVICE=WMS&VERSION=1.3.0&REQUEST=GetCapabilities";

  query_url = malloc(strlen(url) + strlen(query) + 2);
  strcpy(query_url, url);
  strcat(query_url, "?");
  strcat(query_url, query);

  wms_clear(self);
  buffer = memb_create(16384);
  if (!buffer) {
    wmsError_set(self->error, WMS_MEMORY_ERROR, "buffer cannot reserve.");
    goto wms_read_from_uri_error;
  }
  /* GET from REMOTE */
  curl = curl_easy_init();
  if (!curl) {
    wmsError_set(self->error, WMS_CURL_INIT_ERROR, "curl_easy_init failed.");
    goto wms_read_from_uri_error;
  }
  curl_easy_setopt(curl, CURLOPT_URL, query_url);
  curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, _wms_read_from_uri_write_callback);
  curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)buffer);
  curl_easy_setopt(curl, CURLOPT_USERAGENT, WMS_USER_AGENT);
  res = curl_easy_perform(curl);
  curl_easy_cleanup(curl);
  curl = NULL;

  success = wms_read_from_buffer(self, query_url, memb_ptr(buffer), memb_size(buffer));
  if (!success) goto wms_read_from_uri_error;
  memb_free(buffer);
  free(query_url);
  return WMS_TRUE;

wms_read_from_uri_error:
  if (curl) curl_easy_cleanup(curl);
  if (buffer) memb_free(buffer);
  if (query_url) free(query_url);
  return WMS_FALSE;
}

wmsBool wms_read_from_std(wms* self, FILE* io)
{
  wms_clear(self);
  self->xml = xmlReadFd(fileno(io), "http://localhost/", NULL, 0);
  if (!self->xml) {
    wmsError_set(self->error, WMS_PARSE_ERROR, "cannot create XMLDoc instance");
    return WMS_FALSE;
  }
  if (!_wms_parse_capability(self)) {
    xmlCleanupParser();
    return WMS_FALSE;
  }
  xmlCleanupParser();
  return WMS_TRUE;
}

wmsBool wms_read_from_buffer(wms* self, const char* url, const char* buffer, size_t nbuf)
{
  if (!url) url = "http://localhost/";
  wms_clear(self);
  self->xml = xmlReadMemory(buffer, nbuf, url, NULL, 0);
  if (!self->xml) {
    char* ptr;
    xmlErrorPtr xerr = xmlGetLastError();
    asprintf(&ptr, "cannot create XMLDoc instance.\n%s", xerr->message);
    wmsError_set(self->error, WMS_PARSE_ERROR, ptr);
    free(ptr);
    return WMS_FALSE;
  }
  if (!_wms_parse_capability(self)) {
    xmlCleanupParser();
    return WMS_FALSE;
  }
  xmlCleanupParser();
  return WMS_TRUE;
}

/* **********************************************************************
* internal WMS parse routines
********************************************************************** */

static wmsBool _wms_parse_capability(wms* self)
{
  if (!_wms_parse_version(self)) return WMS_FALSE;
  if (!_wms_parse_query_url(self)) return WMS_FALSE;
  if (!_wms_parse_layer(self)) return WMS_FALSE;
  if (!_wms_parse_format(self)) return WMS_FALSE;
  return WMS_TRUE;
}

static wmsBool _wms_parse_version(wms* self)
{
  xmlNode* node = xmlDocGetRootElement(self->xml);
  xmlChar* xs_version = xmlGetProp(node, (xmlChar *)"version");
  char* version;
  char* p;
  regex_t reg;
  int status;
  regmatch_t m[4];
  char* msg;

  if (!xs_version) {
    wmsError_set(self->error, WMS_XML_ATTRIBUTE_NOT_FOUND_ERROR,
                 "version attribute not found");
    return WMS_FALSE;
  }
  status = regcomp(&reg, "^([0-9]+)\\.([0-9]+)\\.([0-9]+)$", REG_EXTENDED);
  assert(0 == status);

  version = strdup((char *)xs_version);
  xmlFree(xs_version);
  if (0 != (status = regexec(&reg, version, 4, m, 0))) {
    asprintf(&msg, "unknown version %s", version);
    wmsError_set(self->error, WMS_FORMAT_ERROR, msg);
    free(msg);
    goto wms_parse_version_error;
  }
  p = version + m[1].rm_so;
  self->version.major = strtol(p, NULL, 10);
  p = version + m[2].rm_so;
  self->version.minor = strtol(p, NULL, 10);
  p = version + m[3].rm_so;
  self->version.release = strtol(p, NULL, 10);
  free(version);
  regfree(&reg);

  return WMS_TRUE;

wms_parse_version_error:
  free(version);
  regfree(&reg);
  return WMS_FALSE;
}

static wmsBool _wms_parse_query_url(wms* self)
{
  xmlNodePtr node;
  xmlXPathContextPtr xcontext;
  xmlXPathObjectPtr res;
  xmlChar* childPath = (xmlChar *)"/Capability/Request/GetMap/DCPType/HTTP/Get/OnlineResource";
  xmlChar* url = NULL;
  strb* str;

  str = strb_create(256);
  node = xmlDocGetRootElement(self->xml);

  if (!self->query_url) {
    xcontext = xmlXPathNewContext(self->xml);
    if (!xcontext) {
      wmsError_set(self->error, WMS_CREATE_XML_CONTEXT_ERROR,
                   "XPathContext can't create.");
      goto wms_parse_query_url_error;
    }
    if (node->ns == NULL) {
      xmlChar* childPath = (xmlChar *)"/Capability/Request/GetMap/DCPType/HTTP/Get/OnlineResource";
      strb_set_cstring(str, "/");
      strb_append_cstring(str, (char *)node->name);
      strb_append_cstring(str, (char *)childPath);
    } else {
      xmlChar* childPath = (xmlChar *)"/w:Capability/w:Request/w:GetMap/w:DCPType/w:HTTP/w:Get/w:OnlineResource";
      if (xmlXPathRegisterNs(xcontext, BAD_CAST"w", BAD_CAST(node->ns->href)) != 0) {
        wmsError_set(self->error, WMS_XPATH_REGISTER_NS_ERROR,
                     "xmlXPathRegisterNs can't register namespace.");
        goto wms_parse_query_url_error;
      }

      strb_set_cstring(str, "/w:");
      strb_append_cstring(str, (char *)node->name);
      strb_append_cstring(str, (char *)childPath);
    }
    printf("path: %s\n", (char*)strb_ptr(str));
        
    res = xmlXPathEvalExpression((xmlChar *)strb_ptr(str), xcontext);
    if (!res) {
      wmsError_set(self->error, WMS_XPATH_EVAL_EXPRESSION_ERROR,
                   "xmlXPathEvalExpression result NULL");
      goto wms_parse_query_url_error;
    }
    if (xmlXPathNodeSetIsEmpty(res->nodesetval)) {
      wmsError_set(self->error, WMS_XPATH_RESULT_EMPTY_ERROR,
                   "OnlineResource tag not found");
      goto wms_parse_query_url_error;
    }
    url = xmlGetProp(res->nodesetval->nodeTab[0], (xmlChar *)"href");
    if (!url) {
      wmsError_set(self->error, WMS_XPATH_RESULT_EMPTY_ERROR,
                   "href attribute not found");
      goto wms_parse_query_url_error;
    }
    self->query_url = strdup((char *)url);

    xmlXPathFreeContext(xcontext);
    xcontext = NULL;
    xmlXPathFreeObject(res);
    res = NULL;
    xmlFree(url);
    url = NULL;
  }
  strb_free(str);
  str = NULL;

  return WMS_TRUE;

wms_parse_query_url_error:
  if (xcontext) xmlXPathFreeContext(xcontext);
  if (str) strb_free(str);
  if (url) xmlFree(url);
  if (res) xmlXPathFreeObject(res);
  return WMS_FALSE;
}

static wmsBool _wms_parse_layer(wms* self)
{
  xmlNodePtr root = xmlDocGetRootElement(self->xml);
  size_t len = 0;
  xmlNodePtr* cap_nodes;
  xmlNodePtr* layer_nodes;
  wmsLayer* layer;

  cap_nodes = xml_util_getElementsByTagName(root, (xmlChar *)"Capability", &len);
  if (len == 0) {
    wmsError_set(self->error, WMS_PARSE_ERROR, "Capability entity does not found.");
    goto wms_parse_layer_error;
  }
  layer_nodes = xml_util_getElementsByTagName(cap_nodes[0], (xmlChar *)"Layer", &len);
  if (len == 0) {
    wmsError_set(self->error, WMS_PARSE_ERROR, "Layer entity does not found.");
    goto wms_parse_layer_error;
  }
  layer = _wms_parse_layer_with_layer_node(self, layer_nodes[0]);
  self->layer = layer;
  free(cap_nodes);
  free(layer_nodes);
  return WMS_TRUE;

wms_parse_layer_error:
  if (cap_nodes) free(cap_nodes);
  if (layer_nodes) free(layer_nodes);
  return WMS_FALSE;
}

static wmsLayer* _wms_parse_layer_with_layer_node(wms* self, xmlNodePtr node)
{
  wmsLayer* layer = wmsLayer_create();
  xmlNodePtr cur;
  xmlChar* s;
  wmsLayer* prev_layer = NULL;
  wmsLayer* next_layer = NULL;
  voidPtr ptr;

  /* parse child layers */
  cur = xmlFirstElementChild(node);
  while (cur) {
    if (0 == strcmp((char *)cur->name, "Layer")) {
      next_layer = _wms_parse_layer_with_layer_node(self, cur);
      next_layer->parent = layer;
      ptr = next_layer;
      ptrb_append(layer->child, &ptr, 1);
      if (prev_layer) {
        prev_layer->next = next_layer;
        next_layer->previous = prev_layer;
      }
      prev_layer = next_layer;
    }
    cur = xmlNextElementSibling(cur);
  }

  /* parse name */
  cur = xmlFirstElementChild(node);
  while (cur) {
    if (0 == strcmp((char *)cur->name, "Name")) {
      s = xmlNodeGetContent(cur);
      strb_set_cstring(layer->name, (char *)s);
      xmlFree(s);
      break;
    }
    cur = xmlNextElementSibling(cur);
  }

  /* parse title */
  cur = xmlFirstElementChild(node);
  while (cur) {
    if (0 == strcmp((char *)cur->name, "Title")) {
      s = xmlNodeGetContent(cur);
      strb_set_cstring(layer->title, (char *)s);
      xmlFree(s);
      break;
    }
    cur = xmlNextElementSibling(cur);
  }

  /* parse abstract */
  cur = xmlFirstElementChild(node);
  while (cur) {
    if (0 == strcmp((char *)cur->name, "Abstract")) {
      s = xmlNodeGetContent(cur);
      strb_set_cstring(layer->abstract, (char *)s);
      xmlFree(s);
      break;
    }
    cur = xmlNextElementSibling(cur);
  }

  /* parse srs */
  cur = xmlFirstElementChild(node);
  while (cur) {
    if (0 == strcmp((char *)cur->name, "CRS")) {
      s = xmlNodeGetContent(cur);
      ptr = strb_create_with_cstring((char *)s);
      ptrb_append(layer->srs, &ptr, 1);
      xmlFree(s);
    }
    cur = xmlNextElementSibling(cur);
  }

  /* parse crs */
  cur = xmlFirstElementChild(node);
  while (cur) {
    if (0 == strcmp((char *)cur->name, "SRS")) {
      s = xmlNodeGetContent(cur);
      ptr = strb_create_with_cstring((char *)s);
      ptrb_append(layer->crs, &ptr, 1);
      xmlFree(s);
    }
    cur = xmlNextElementSibling(cur);
  }

  /* bounding box */
  _wms_parser_parse_layer_bbox(self, layer, node);

  return layer;
}

static wmsBool _wms_parser_parse_layer_bbox(wms* self, wmsLayer* layer, xmlNodePtr node)
{
  xmlChar* s;
  char* e;
  double num;
  xmlNodePtr cur;

  cur = xmlFirstElementChild(node);
  while (cur) {
    if (strcmp((char *)cur->name, "LatLongBoundingBox")) {
      s = xmlGetProp(cur, (xmlChar *)"minx");
      if (!s) {
        wmsError_set(self->error, WMS_PARSE_ERROR, "minx attribute does not found");
        goto wms_parser_parse_layer_bbox_error;
      }
      num = strtod((char *)s, &e);
      if ((char *)s == e) {
        wmsError_set(self->error, WMS_PARSE_ERROR, "minx attribute format error");
        goto wms_parser_parse_layer_bbox_error;
      }
      layer->bbox.min.lng = num;
      xmlFree(s);

      s = xmlGetProp(cur, (xmlChar *)"miny");
      if (!s) {
        wmsError_set(self->error, WMS_PARSE_ERROR, "miny attribute does not found");
        goto wms_parser_parse_layer_bbox_error;
      }
      num = strtod((char *)s, &e);
      if ((char *)s == e) {
        wmsError_set(self->error, WMS_PARSE_ERROR, "miny attribute format error");
        goto wms_parser_parse_layer_bbox_error;
      }
      layer->bbox.min.lat = num;
      xmlFree(s);

      s = xmlGetProp(cur, (xmlChar *)"maxx");
      if (!s) {
        wmsError_set(self->error, WMS_PARSE_ERROR, "maxx attribute does not found");
        goto wms_parser_parse_layer_bbox_error;
      }
      num = strtod((char *)s, &e);
      if ((char *)s == e) {
        wmsError_set(self->error, WMS_PARSE_ERROR, "maxx attribute format error");
        goto wms_parser_parse_layer_bbox_error;
      }
      layer->bbox.max.lng = num;
      xmlFree(s);

      s = xmlGetProp(cur, (xmlChar *)"maxy");
      if (!s) {
        wmsError_set(self->error, WMS_PARSE_ERROR, "maxy attribute does not found");
        goto wms_parser_parse_layer_bbox_error;
      }
      num = strtod((char *)s, &e);
      if ((char *)s == e) {
        wmsError_set(self->error, WMS_PARSE_ERROR, "maxy attribute format error");
        goto wms_parser_parse_layer_bbox_error;
      }
      layer->bbox.max.lat = num;
      xmlFree(s);
      break;
    }
    cur = xmlNextElementSibling(cur);
  }
  return WMS_TRUE;

wms_parser_parse_layer_bbox_error:
  if (s) xmlFree(s);
  return WMS_FALSE;
}

// Capability > Request > GetMap > Format
static wmsBool _wms_parse_format(wms* self)
{
  xmlNodePtr root = xmlDocGetRootElement(self->xml);
  xmlNodePtr* cap_nodes;
  xmlNodePtr* req_nodes;
  xmlNodePtr* map_nodes;
  xmlNodePtr* format_nodes;
  xmlChar* s;
  strb* str;
  int i;
  size_t len;

  cap_nodes = xml_util_getElementsByTagName(root, (xmlChar *)"Capability", &len);
  if (len == 0) {
    wmsError_set(self->error, WMS_PARSE_ERROR, "Capability entity does not found.");
    goto wms_parse_format_error;
  }
  req_nodes = xml_util_getElementsByTagName(cap_nodes[0], (xmlChar *)"Request", &len);
  if (len == 0) {
    wmsError_set(self->error, WMS_PARSE_ERROR, "Request entity does not found.");
    goto wms_parse_format_error;
  }
  map_nodes = xml_util_getElementsByTagName(req_nodes[0], (xmlChar *)"GetMap", &len);
  if (len == 0) {
    wmsError_set(self->error, WMS_PARSE_ERROR, "GetMap entity does not found.");
    goto wms_parse_format_error;
  }
  format_nodes = xml_util_getElementsByTagName(map_nodes[0], (xmlChar *)"Format", &len);
  if (len == 0) {
    wmsError_set(self->error, WMS_PARSE_ERROR, "Format entity does not found.");
    goto wms_parse_format_error;
  }
  for (i = 0; i < len; i++) {
    s = xmlNodeGetContent(format_nodes[i]);
    str = strb_create_with_cstring((char *)s);
    strbary_append(self->formats, &str, 1);
    xmlFree(s);
  }
  if (cap_nodes) free(cap_nodes);
  if (req_nodes) free(req_nodes);
  if (map_nodes) free(map_nodes);
  if (format_nodes) free(format_nodes);
  return WMS_TRUE;

wms_parse_format_error:
  if (cap_nodes) free(cap_nodes);
  if (req_nodes) free(req_nodes);
  if (map_nodes) free(map_nodes);
  if (format_nodes) free(format_nodes);
  return WMS_FALSE;
}

#endif
