/**
 * @file      wms_req.c
 * @brief     WMS request interface
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2010-2011 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#ifdef ENABLE_WMS

#ifdef ASPRINTF_NEED_GNU_SOURCE
#  define _GNU_SOURCE
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <curl/curl.h>
#include "regscan.h"
#include "wms.h"
#include "wms_req.h"
#include "wms_res.h"
#include "wms_layer.h"
#include "wms_error.h"

int _wmsReq_lock(wmsReq* self);
int _wmsReq_unlock(wmsReq* self);
static size_t _wmsReq_write_body_callback(void* ptr, size_t size, size_t nmemb, void* data);
static size_t _wmsReq_write_header_callback(void* ptr, size_t size, size_t nmemb, void* data);
static void* _wmsReq_fetch_pth(void* data);

wmsReq* wmsReq_create(wms* w)
{
  wmsReq* self;
  self = malloc(sizeof(*self));
  if (!self) {
    return NULL;
  }
  memset(self, 0, sizeof(*self));
  self->w = w;
  pthread_mutexattr_init(self->mutex_attr);
  pthread_mutex_init(self->mutex, self->mutex_attr);
  self->url = strb_create(256);
  self->error = wmsError_create();
  return self;
}

void wmsReq_free(wmsReq* self)
{
  strb_free(self->url);
  wmsError_free(self->error);
  pthread_mutex_destroy(self->mutex);
  pthread_mutexattr_destroy(self->mutex_attr);
  free(self);
}

const char* wmsReq_url(wmsReq* self)
{
  return (char *)strb_ptr(self->url);
}

wmsError* wmsReq_error(wmsReq* self)
{
  return self->error;
}

void wmsReq_set_url(wmsReq* self, const char* url)
{
  strb_set_cstring(self->url, url);
}

void wmsReq_set_capability_url(wmsReq* self, const char* url)
{
  size_t len = strlen(url);

  strb_set_cstring(self->url, url);
  if (url[len-1] != '?' && url[len-1] != '&') {
    strb_append_cstring(self->url, "?");
  }
  strb_append_cstring(self->url, "SERVICE=WMS&VERSION=1.3.0&REQUEST=GetCapabilities");
}

wmsBool wmsReq_set_map_url(wmsReq* self, const char* name, const char* format, enum WMSCrs crs,
                           wmsRegion region, size_t width, size_t height)
{
  wms* w = self->w;
  wmsLayer* layer;
  strb* url = strb_create_with_cstring(wms_query_url(w));
  strb* query;
  strb* str = NULL;
  strb* enc_str = NULL;
  wmsVersion ver = wms_version(w);
  char version[10];
  const char* scrs;
  char* errstr;

  query = strb_create(512);
  str = strb_create(256);
  if (!query || !str) {
    goto wmsReq_fetch_main_error;
  }

  // CRS
  switch (crs) {
  case WMSCrsEPSG4326:
    scrs = "EPSG:4326";
    break;
  case WMSCrsCRS83:
    scrs = "CRS:83";
    break;
  default:
    asprintf(&errstr, "Unknown CRS code #%d.", crs);
    wmsError_set(self->error, WMS_UNKNOWN_CRS_ERROR, errstr);
    free(errstr);
    goto wmsReq_fetch_main_error;
  }

  // check layer
  layer = wms_find_layer(w, WMSFindTypeName, name);
  if (!layer) {
    asprintf(&errstr, "layer '%s' does not found.", name);
    wmsError_set(self->error, WMS_LAYER_NOT_FOUND_ERROR, errstr);
    free(errstr);
    goto wmsReq_fetch_main_error;
  }
  if (!wmsLayer_has_crs(layer, scrs)) {
    asprintf(&errstr, "CRS '%s' can't use in layer '%s'.", scrs, name);
    wmsError_set(self->error, WMS_LAYER_NOT_FOUND_ERROR, errstr);
    free(errstr);
    goto wmsReq_fetch_main_error;
  }

  // check version
  sprintf(version, "%d.%d", ver.major, ver.minor);
  if (0 != strcmp(version, "1.1") && 0 != strcmp(version, "1.3")) {
    fprintf(stderr, "unknown WMS version %s", version);
    abort();
  }

  /* Set query URL */
  strb_append_sprintf(query,
                      "SERVICE=WMS&REQUEST=GetMap&"
                      "VERSION=%s&FORMAT=%s&WIDTH=%d&HEIGHT=%d&"
                      "BBOX=%.4f,%.4f,%.4f,%.4f",
                      version,
                      format,
                      width, height,
                      region.min.lng, region.min.lat,
                      region.max.lng, region.max.lat);
  strb_set_cstring(str, name);
  enc_str = strb_convert_urlencode(str);
  strb_append_sprintf(query, "&LAYERS=%s", (char *)strb_ptr(enc_str));
  strb_free(enc_str);
  if (0 == strcmp(version, "1.1")) {
    strb_append_sprintf(query, "&STYLES=&SRS=%s", scrs);
  }
  else if (0 == strcmp(version, "1.3")) {
    strb_append_sprintf(query, "&CRS=%s", scrs);
  }
  strb_append_cstring(url, (char *)strb_ptr(query));
  //fprintf(stderr, "%s\n", (char*)strb_ptr(url));

  strb_set_cstring(self->url, (char *)strb_ptr(url));
  if (query) strb_free(query);
  if (url) strb_free(url);
  if (str) strb_free(str);
  return WMS_TRUE;

wmsReq_fetch_main_error:
  if (query) strb_free(query);
  if (url) strb_free(url);
  if (str) strb_free(str);
  return WMS_FALSE;
}

void wmsReq_set_error_handler(wmsReq* self, wmsReqCallback func, void* userdata)
{
  self->error_callback = func;
  self->error_data = userdata;
}

void wmsReq_set_success_handler(wmsReq* self, wmsReqCallback func, void* userdata)
{
  self->success_callback = func;
  self->success_data = userdata;
}

int wmsReq_wait(wmsReq* self)
{
  void* exitValue;
  pthread_join(self->pth, &exitValue);
  return 1;
}

static void* _wmsReq_fetch_pth(void* data)
{
  wmsRes* res = wmsReq_fetch((wmsReq *)data);
  wmsRes_free(res);
  return data;
}
void wmsReq_fetch_async(wmsReq* self)
{
  pthread_attr_init(self->pth_attr);
  pthread_create(&self->pth, self->pth_attr, _wmsReq_fetch_pth, self);
  pthread_detach(self->pth);
}

wmsRes* wmsReq_fetch(wmsReq* self)
{
  CURLcode curl_err;
  CURL* curl = NULL;
  char curl_errstr[CURL_ERROR_SIZE];
  wmsRes* res;
  const char* url = (char *)strb_ptr(self->url);

  res = wmsRes_create(65536);
  if (!res) {
    wmsError_set(self->error, WMS_MEMORY_ERROR, "wmsRes_create failed.");
    goto wmsReq_fetch_error;
  }
  memset(curl_errstr, 0, CURL_ERROR_SIZE);
  curl = curl_easy_init();
  if (!curl) {
    wmsError_set(self->error, WMS_CURL_INIT_ERROR, "curl_easy_init failed.");
    goto wmsReq_fetch_error;
  }
  curl_easy_setopt(curl, CURLOPT_URL, url);
  curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, _wmsReq_write_header_callback);
  curl_easy_setopt(curl, CURLOPT_WRITEHEADER, (void *)res);
  curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, _wmsReq_write_body_callback);
  curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)res);
  curl_easy_setopt(curl, CURLOPT_USERAGENT, WMS_USER_AGENT);
  curl_easy_setopt(curl, CURLOPT_TIMEOUT, 60);
  curl_easy_setopt(curl, CURLOPT_FAILONERROR, 1);
  curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, curl_errstr);

  curl_err = curl_easy_perform(curl);
  if (0 != curl_err) {
    wmsError_set(res->error, WMS_HTTP_REQUEST_ERROR, curl_errstr);
    if (self->error_callback) {
      self->error_callback(self, res, self->error_data);
    }
    goto wmsReq_fetch_error;
  }
  curl_easy_cleanup(curl);
  curl = NULL;

  if (self->success_callback) {
    self->success_callback(self, res, self->success_data);
  }
  return res;

wmsReq_fetch_error:
  if (curl) curl_easy_cleanup(curl);
  if (res) wmsRes_free(res);
  return NULL;
}

int _wmsReq_lock(wmsReq* self)
{
  return pthread_mutex_lock(self->mutex);
}

int _wmsReq_unlock(wmsReq* self)
{
  return pthread_mutex_unlock(self->mutex);
}

static size_t _wmsReq_write_header_callback(void* ptr, size_t size, size_t nmemb, void* data)
{
  wmsRes* res = (wmsRes *)data;
  size_t realsize = size*nmemb;
  char* str = (char *)ptr;
  char term = '\0';
  regscan* re;
  int len;
  const char* match;

  re = regscan_create();
  if (REGSCAN_STATUS_NO_ERROR == regscan_compile(re, "^HTTP/1\\.[01] +([0-9]+)", REG_ICASE|REG_EXTENDED)&&
      REGSCAN_STATUS_MATCH == regscan_exec(re, str)) {
    match = regscan_match(re, 1, &len);
    res->status = strtol(match, NULL, 10);
  }
  else if (REGSCAN_STATUS_NO_ERROR == regscan_compile(re, "^Content-Type: +([^ ;\r\n]+)", REG_ICASE|REG_EXTENDED)&&
           REGSCAN_STATUS_MATCH == regscan_exec(re, str)) {
    match = regscan_match(re, 1, &len);
    strb_set(res->content_type, match, len);
    strb_append(res->content_type, &term, 1);
  }
  regscan_free(re);
  return realsize;
}

static size_t _wmsReq_write_body_callback(void* ptr, size_t size, size_t nmemb, void* data)
{
  size_t realsize = size*nmemb;
  wmsRes* res = (wmsRes *)data;
  if (!memb_append(res->data, ptr, realsize)) {
    return 0;
  }
  return realsize;
}

#endif
