/**
 * @file      wms_req.h
 * @brief     WMS request interface
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2010-2011 Japan Aerospace Exploration Agency
 *
 */

#ifndef _WMS_REQ_H
#define _WMS_REQ_H

#include "wms.h"

/**
 * @brief Create a wms request instance
 *
 * wmsReq_create() creates a wms request instance
 * which includes a URL, thread attributes, and
 * an error instance.
 *
 * @param[in]   w   wms instance
 */
wmsReq* wmsReq_create(wms* w);

/**
 * @brief Free a wms request instance
 *
 * wmsReq_free() deallocates the memory previously allocated by
 * wmsReq_create().
 *
 * @param[in]   self  wms request instance
 */
void wmsReq_free(wmsReq* self);

/**
 * @brief Getter of the URL in a wms request instance
 *
 * @param[in]  self  wms request instance
 * @return wmsReq_url() returns a URL string in the specified wms request
 * instance.
 */
const char* wmsReq_url(wmsReq* self);

/**
 * @brief Getter of the error instance in a wms request instance
 *
 * @param[in]  self  wms request instance
 * @return wmsReq_error() returns the wms error instance in the specified
 * wms request instance.
 */
wmsError* wmsReq_error(wmsReq* self);

/**
 * @brief Setter of a URL in the specified wms request instance
 *
 * @param[in]  self  wms request instance
 * @param[in]  url   URL string
 */
void wmsReq_set_url(wmsReq* self, const char* url);

/**
 * @brief Setter of a URL in the specified wms request instance
 * with REQUEST=GetCapabilities
 *
 * @param[in]  self  wms request instance
 * @param[in]  url   URL string
 */
void wmsReq_set_capability_url(wmsReq* self, const char* url);

/**
 * @brief Setter of a URL in the specified wms request instance
 * with REQUEST=GetMap
 *
 * @param[in]  self   wms request instance
 * @param[in]  name   layer name
 * @param[in]  format image format
 * @param[in]  crs    Spatial Reference System for map output
 * @param[in]  region region of interest
 * @param[in]  width  image width
 * @param[in]  height image height
 * @return wmsReq_set_map_url() returns WMS_TRUE if WMS map is successfully
 * obtained. Otherwise, WMS_FALSE is returned.
 */
wmsBool wmsReq_set_map_url(wmsReq* self, const char* name, const char* format, enum WMSCrs crs,
                           wmsRegion region, size_t width, size_t height);

/**
 * @brief Setter of an error callback function
 *
 * @param[in]  self     wms request instance
 * @param[in]  func     error callback function pointer
 * @param[in]  userdata error data
 */
void wmsReq_set_error_handler(wmsReq* self, wmsReqCallback func, void* userdata);

/**
 * @brief Setter of an success callback function
 *
 * @param[in]  self     wms request instance
 * @param[in]  func     success callback function pointer
 * @param[in]  userdata success data
 */
void wmsReq_set_success_handler(wmsReq* self, wmsReqCallback func, void* userdata);

/**
 * @brief Issue HTTP GET method
 *
 * @param[in] self  wmsReq instance.
 * @result wmsRes object, or NULL at error.
 *         object must call wmsRes_free() when no longer used.
 */
wmsRes* wmsReq_fetch(wmsReq* self);

/**
 * @brief Issue HTTP GET method in a new thread
 *
 * @param self  a wmsReq instance.
 */
void wmsReq_fetch_async(wmsReq* self);

/**
 * @brief Wait for the thread created by wmsReq_fetch_async
 *
 * @param self  a wmsReq instance.
 */
int wmsReq_wait(wmsReq* self);

#endif /* _WMS_REQ_H */
